<?php
/**
 * File: templates/speaker-edit.php
 * Purpose: Front-end "My Profile" editor with access control (owner/email) and secure image upload.
 * Notes:
 *  - Shows ALL enabled fields from Form Builder (skips Section 8 / Private).
 *  - Access: admin OR post author (owner) OR logged-in user whose email matches the profile's email field.
 *  - If no ?speaker_id= is provided:
 *      * Admins see a picker (search + list of speakers) to choose a profile.
 *      * Logged-in speakers auto-resolve to their own profile if exactly one; otherwise they see a list or a link to register.
 *  - Image fields: Upload/Replace (local file) + Remove. No numeric/URL box. Replacing deletes old attachment ID.
 *  - Additive only; no removal of existing functionality.
 */

if (!defined('ABSPATH')) exit;

// Load bootstrap (role + admin restrictions + login shortcode)
$__sb_auth = dirname(__DIR__).'/includes/sb-auth-bootstrap.php';
if (file_exists($__sb_auth)) require_once $__sb_auth;

$is_logged_in = is_user_logged_in();

/** Identify which speaker profile to edit (accept ?speaker_id=) */
$requested_id = isset($_GET['speaker_id']) ? intval($_GET['speaker_id']) : 0;
$edit_post_id = ($requested_id > 0 && get_post_type($requested_id) === 'speaker') ? $requested_id : 0;

/** If no explicit ID, try to auto-resolve */
if (!$edit_post_id) {
    if (current_user_can('manage_options')) {
        // ADMIN PICKER: optional search + latest speakers list
        $search = isset($_GET['s']) ? sanitize_text_field(wp_unslash($_GET['s'])) : '';
        $args   = [
            'post_type'      => 'speaker',
            'posts_per_page' => 25,
            's'              => $search,
            'orderby'        => 'date',
            'order'          => 'DESC',
            'post_status'    => ['publish', 'draft', 'pending', 'future', 'private'],
        ];
        $admin_posts = get_posts($args);

        echo '<div class="sb-wrap sb-edit">';
        echo '<h2 class="sb-title">'.esc_html__('Select a Speaker to Edit', 'speakers-bureau').'</h2>';
        echo '<form method="get" action="" style="margin:0 0 12px 0; display:flex; gap:8px;">';
        echo '<input type="text" name="s" value="'.esc_attr($search).'" placeholder="'.esc_attr__('Search speakers…','speakers-bureau').'" />';
        echo '<button class="button">'.esc_html__('Search','speakers-bureau').'</button>';
        echo '</form>';

        if ($admin_posts) {
            echo '<ul class="sb-admin-speaker-list">';
            foreach ($admin_posts as $p) {
                $url = add_query_arg('speaker_id', $p->ID, get_permalink());
                echo '<li><a href="'.esc_url($url).'">'.esc_html(get_the_title($p)).'</a> <span style="opacity:.7;">(#'.intval($p->ID).')</span></li>';
            }
            echo '</ul>';
        } else {
            echo '<p>'.esc_html__('No speakers found. Try another search, or register a new speaker.', 'speakers-bureau').'</p>';
        }
        echo '</div>';
        return;
    }

    // SPEAKER USER: try to find their own profile(s)
    if ($is_logged_in) {
        $mine = get_posts([
            'post_type'      => 'speaker',
            'posts_per_page' => 50,
            'author'         => get_current_user_id(),
            'post_status'    => ['publish', 'draft', 'pending', 'future', 'private'],
            'orderby'        => 'modified',
            'order'          => 'DESC',
        ]);
        if (count($mine) === 1) {
            $edit_post_id = $mine[0]->ID; // auto-resolve to the single profile
        } elseif (count($mine) > 1) {
            echo '<div class="sb-wrap sb-edit">';
            echo '<h2 class="sb-title">'.esc_html__('Choose Your Profile', 'speakers-bureau').'</h2>';
            echo '<ul class="sb-my-speaker-list">';
            foreach ($mine as $p) {
                $url = add_query_arg('speaker_id', $p->ID, get_permalink());
                echo '<li><a href="'.esc_url($url).'">'.esc_html(get_the_title($p)).'</a> <span style="opacity:.7;">(#'.intval($p->ID).')</span></li>';
            }
            echo '</ul>';
            echo '</div>';
            return;
        } else {
            $reg_page = get_page_by_path('register-speaker');
            $reg_url  = $reg_page ? get_permalink($reg_page) : home_url('/register-speaker/');
            echo '<div class="sb-wrap sb-edit"><p>'.esc_html__('You don’t have a speaker profile yet.', 'speakers-bureau').' ';
            echo '<a class="button" href="'.esc_url($reg_url).'">'.esc_html__('Register a Speaker','speakers-bureau').'</a></p></div>';
            return;
        }
    }
}

/** Still nothing valid? Bail with a clear message (shouldn’t hit for admins now) */
if (!$edit_post_id) {
    echo '<div class="sb-wrap sb-edit"><p>'.esc_html__('No speaker profile found to edit.', 'speakers-bureau').'</p></div>';
    return;
}

/** Load layout & fields (non-destructive defaults) */
$defaults_layout = [
    1 => ['title'=>'Profile','heading'=>'h4','display'=>1,'hidden'=>0,'class'=>'sb-section-1'],
    2 => ['title'=>'Overview','heading'=>'h2','display'=>1,'hidden'=>0,'class'=>'sb-section-2'],
    3 => ['title'=>'Topics','heading'=>'h3','display'=>1,'hidden'=>0,'class'=>'sb-section-3'],
    4 => ['title'=>'Background','heading'=>'h3','display'=>1,'hidden'=>0,'class'=>'sb-section-4'],
    5 => ['title'=>'','heading'=>'h3','display'=>0,'hidden'=>1,'class'=>'sb-section-5'],
    6 => ['title'=>'','heading'=>'h3','display'=>0,'hidden'=>1,'class'=>'sb-section-6'],
    7 => ['title'=>'More Information','heading'=>'h2','display'=>1,'hidden'=>0,'class'=>'sb-section-7'],
    8 => ['title'=>'Private Fields','heading'=>'h3','display'=>0,'hidden'=>1,'class'=>'sb-section-8'],
];
$layout = get_option('sb_profile_layout', $defaults_layout);
foreach (range(1,8) as $i) {
    $layout[$i] = wp_parse_args(isset($layout[$i]) ? (array)$layout[$i] : [], $defaults_layout[$i]);
}

$fields = function_exists('sb_normalize_form_fields') ? sb_normalize_form_fields() : [];
if (!is_array($fields)) $fields = [];

/** Determine primary email meta key from builder (fallback to 'email') */
$profile_email_key = 'email';
foreach ($fields as $f) {
    if (!empty($f['enabled']) && ($f['type'] ?? '') === 'email' && !empty($f['key'])) {
        $profile_email_key = $f['key'];
        break;
    }
}

/** Access control:
 *  - Admin: allowed.
 *  - Owner: allowed (current user ID equals post_author).
 *  - Email match: allowed (user_email === profile email meta, case-insensitive).
 */
$can_access = current_user_can('manage_options');
if (!$can_access) {
    if (!$is_logged_in) {
        echo '<div class="sb-wrap sb-edit"><p>'.esc_html__('You must be logged in to edit this profile.', 'speakers-bureau').'</p></div>';
        return;
    }
    $owner_id  = (int) get_post_field('post_author', $edit_post_id);
    $curr_id   = get_current_user_id();
    if ($owner_id && $curr_id && $owner_id === $curr_id) {
        $can_access = true;
    } else {
        $user      = wp_get_current_user();
        $user_mail = strtolower(trim((string)$user->user_email));
        $prof_mail = strtolower(trim((string)get_post_meta($edit_post_id, $profile_email_key, true)));
        if ($user_mail && $prof_mail && $user_mail === $prof_mail) {
            $can_access = true;
        }
    }
}
if (!$can_access) {
    echo '<div class="sb-wrap sb-edit"><p>'.esc_html__('You do not have permission to edit this profile.', 'speakers-bureau').'</p></div>';
    return;
}

/** Group enabled fields by section (skip Section 8 and edit_link fields on public edit form) */
$fields_by_section = [];
foreach ($fields as $f) {
    if (empty($f['enabled'])) continue;
    $sec = isset($f['section']) ? (int) $f['section'] : 2;
    if ($sec < 1 || $sec > 8) $sec = 2;
    if ($sec === 8) continue;
    $type = $f['type'] ?? 'text';
    $key  = $f['key']  ?? '';
    if ($key === 'placeholder' || $type === 'edit_link') continue;
    $fields_by_section[$sec][] = $f;
}

/** Upload helper (local file only; returns attachment ID) */
if (!function_exists('sb_handle_frontend_upload')) {
function sb_handle_frontend_upload($field_name, $attach_to_post) {
    if (!isset($_FILES[$field_name]) || empty($_FILES[$field_name]['name'])) {
        return new WP_Error('no_file', __('No file uploaded', 'speakers-bureau'));
    }
    require_once ABSPATH.'wp-admin/includes/file.php';
    require_once ABSPATH.'wp-admin/includes/image.php';
    require_once ABSPATH.'wp-admin/includes/media.php';

    $overrides = array('test_form' => false);
    $file = wp_handle_upload($_FILES[$field_name], $overrides);
    if (isset($file['error'])) return new WP_Error('upload_error', $file['error']);

    $filetype = wp_check_filetype(basename($file['file']), null);
    $attachment = [
        'guid'           => $file['url'],
        'post_mime_type' => $filetype['type'],
        'post_title'     => sanitize_file_name(basename($file['file'])),
        'post_content'   => '',
        'post_status'    => 'inherit',
    ];
    $attach_id = wp_insert_attachment($attachment, $file['file'], $attach_to_post);
    if (is_wp_error($attach_id)) return $attach_id;

    $attach_data = wp_generate_attachment_metadata($attach_id, $file['file']);
    wp_update_attachment_metadata($attach_id, $attach_data);
    return $attach_id;
}}
/** END helper */

/** ===== Save handler ===== */
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['sb_edit_nonce']) && wp_verify_nonce($_POST['sb_edit_nonce'], 'sb_edit_'.$edit_post_id)) {

    // Update title
    if (isset($_POST['sb_post_title'])) {
        $new_title = sanitize_text_field(wp_unslash($_POST['sb_post_title']));
        if ($new_title !== '') {
            wp_update_post(['ID' => $edit_post_id, 'post_title' => $new_title]);
        }
    }

    // Save meta
    foreach ($fields as $f) {
        if (empty($f['enabled'])) continue;
        $sec = isset($f['section']) ? (int)$f['section'] : 2;
        if ($sec === 8) continue;
        $type = $f['type'] ?? 'text';
        $key  = $f['key']  ?? '';
        if ($key === '' || $key === 'placeholder' || $type === 'edit_link') continue;

        if ($type === 'image') {
            $file_field   = 'sb_'.$key.'_file';
            $remove_field = 'sb_'.$key.'_remove';
            $old          = get_post_meta($edit_post_id, $key, true);

            $wants_remove = !empty($_POST[$remove_field]);
            $has_upload   = isset($_FILES[$file_field]) && !empty($_FILES[$file_field]['name']);


            if ($wants_remove) {
                if (ctype_digit((string)$old)) wp_delete_attachment((int)$old, true);
                delete_post_meta($edit_post_id, $key);

                // If this is the profile image, also remove featured image
                if ($key === 'profile_image') {
                    delete_post_thumbnail($edit_post_id);
                }
                continue;
            }
            if ($has_upload) {
                $res = sb_handle_frontend_upload($file_field, $edit_post_id);
                if (!is_wp_error($res)) {
                    if (ctype_digit((string)$old)) wp_delete_attachment((int)$old, true);
                    update_post_meta($edit_post_id, $key, (int)$res);

                    // If this is the profile image, also set as featured image for SEO
                    if ($key === 'profile_image') {
                        set_post_thumbnail($edit_post_id, (int)$res);
                    }
                }
                continue;
            }
            continue;
        }

        $name = 'sb_'.$key;
        if ($type === 'checkbox' || (!empty($f['multiple']))) {
            $val = isset($_POST[$name]) ? (array) $_POST[$name] : [];
            $val = array_values(array_filter(array_map('sanitize_text_field', wp_unslash($val))));
            update_post_meta($edit_post_id, $key, $val);
        } else {
            $raw = isset($_POST[$name]) ? wp_unslash($_POST[$name]) : '';
            if (is_array($raw)) $raw = implode(', ', $raw);
            $raw = trim((string)$raw);

            switch ($type) {
                case 'email':
                    $val = sanitize_email($raw);
                    break;
                case 'url':
                    $val = esc_url_raw($raw);
                    break;
                case 'textarea':
                    // Preserve line breaks for textareas
                    $val = sanitize_textarea_field($raw);
                    break;
                default:
                    $val = sanitize_text_field($raw);
                    break;
            }
            if ($val === '') delete_post_meta($edit_post_id, $key);
            else update_post_meta($edit_post_id, $key, $val);
        }

        // Save privacy preferences for email and phone fields
        if (in_array($type, ['email', 'phone', 'tel'])) {
            $privacy_name = 'sb_'.$key.'_private';
            if (!empty($_POST[$privacy_name])) {
                update_post_meta($edit_post_id, $key.'_private', '1');
            } else {
                delete_post_meta($edit_post_id, $key.'_private');
            }
        }
    }

    // Save admin's suppress notification preference (admins only)
    if (current_user_can('manage_options')) {
        if (isset($_POST['sb_suppress_update_notification'])) {
            update_post_meta($edit_post_id, '_sb_suppress_update_notification', '1');
        } else {
            update_post_meta($edit_post_id, '_sb_suppress_update_notification', '0');
        }
    }

    // Send profile update email if checkbox was checked AND admin has not suppressed it
    $send_update_email = !empty($_POST['sb_send_update_email']);
    $suppress_notification = get_post_meta($edit_post_id, '_sb_suppress_update_notification', true);

    // Only send email if user requested it AND admin hasn't suppressed it
    if ($send_update_email && $suppress_notification !== '1') {
        $user_email = get_post_meta($edit_post_id, $profile_email_key, true);
        $speaker_name = get_the_title($edit_post_id);

        if ($user_email && function_exists('sb_send_profile_update_email')) {
            sb_send_profile_update_email($edit_post_id, $user_email, $speaker_name);
        }
    }

    // Send admin notification (always)
    // Check if auto-publish should be applied
    $current_status = get_post_status($edit_post_id);
    if ($current_status === 'draft') {
        $display_settings = get_option('sb_display_settings', []);
        $auto_publish = !empty($display_settings['auto_publish_profiles']);

        if ($auto_publish) {
            // Check if all required fields are now completed
            $required_fields_completed = true;
            foreach ($fields as $f) {
                if (!empty($f['required']) && !empty($f['enabled'])) {
                    $key = $f['key'] ?? '';
                    if ($key) {
                        $value = get_post_meta($edit_post_id, $key, true);
                        if (empty($value)) {
                            $required_fields_completed = false;
                            break;
                        }
                    }
                }
            }

            // Auto-publish if all required fields are complete
            if ($required_fields_completed) {
                wp_update_post([
                    'ID' => $edit_post_id,
                    'post_status' => 'publish'
                ]);
                $auto_published = true;
            }
        }
    }

    $admin_email = get_option('admin_email');
    if ($admin_email) {
        $admin_subject = __('A speaker profile was updated','speakers-bureau');
        $admin_body = "<p>A speaker profile was updated:</p><p><strong>Post ID:</strong> {$edit_post_id}<br><strong>Title:</strong> ".get_the_title($edit_post_id)."<br><strong>Profile:</strong> <a href=\"".get_permalink($edit_post_id)."\">View Profile</a></p>";
        $headers = ['Content-Type: text/html; charset=UTF-8'];
        wp_mail($admin_email, $admin_subject, $admin_body, $headers);
    }

    if (!empty($auto_published)) {
        echo '<div class="notice notice-success is-dismissible" style="margin:16px 0;"><p>'.esc_html__('Profile saved and published! Your profile is now live in the speakers directory.', 'speakers-bureau').'</p></div>';
    } else {
        echo '<div class="notice notice-success is-dismissible" style="margin:16px 0;"><p>'.esc_html__('Profile saved.', 'speakers-bureau').'</p></div>';
    }
}

/** Helpers to render the form UI */
if (!function_exists('sb_fb_parse_options')) {
function sb_fb_parse_options($raw) {
    $out = [];
    $raw = trim((string) $raw);
    if ($raw === '') return $out;

    // Support both comma-separated and line-separated options
    $items = [];
    if (strpos($raw, "\n") !== false) {
        // Line-separated format
        $items = preg_split('/\r\n|\r|\n/', $raw);
    } else {
        // Comma-separated format
        $items = explode(',', $raw);
    }

    foreach ($items as $item) {
        $item = trim($item);
        if ($item === '') continue;

        // Support "Label|Value" format
        if (strpos($item, '|') !== false) {
            list($l, $v) = array_map('trim', explode('|', $item, 2));
            if ($v === '') $v = $l;
        } else {
            $l = $item;
            $v = $item;
        }
        $out[] = ['label' => $l, 'value' => $v];
    }
    return $out;
}}

if (!function_exists('sb_render_edit_control')) {
function sb_render_edit_control($field, $post_id) {
    $type  = $field['type'] ?? 'text';
    $key   = $field['key']  ?? '';
    $label = $field['label'] ?? '';
    if ($key === '') return '';

    $name = 'sb_'.$key;
    $val  = get_post_meta($post_id, $key, true);

    ob_start();
    echo '<div class="sb-input sb-input-'.esc_attr($type).'">';
    if (empty($field['hide_label']) && $label !== '') {
        echo '<label for="sb_'.esc_attr($key).'" class="sb-label">'.esc_html($label).'</label>';
    }

    switch ($type) {
        case 'textarea':
            echo '<textarea id="sb_'.esc_attr($key).'" name="'.esc_attr($name).'" rows="5" style="width:100%;">'.esc_textarea(is_array($val) ? implode(", ", $val) : (string)$val).'</textarea>';
            break;
        case 'email':
            echo '<input type="email" id="sb_'.esc_attr($key).'" name="'.esc_attr($name).'" value="'.esc_attr(is_array($val) ? implode(", ", $val) : (string)$val).'" style="width:100%;">';
            // Add privacy checkbox for email fields
            $is_private = get_post_meta($post_id, $key.'_private', true);
            echo '<div class="sb-privacy-option" style="margin-top:8px;">';
            echo '<label for="sb_'.esc_attr($key).'_private" style="display:flex;align-items:center;font-size:13px;color:#666;">';
            echo '<input type="checkbox" id="sb_'.esc_attr($key).'_private" name="sb_'.esc_attr($key).'_private" value="1"'.($is_private ? ' checked' : '').' style="margin-right:6px;"> ';
            echo esc_html__("Keep this email address private (won't be shown on public profile)", 'speakers-bureau');
            echo '</label></div>';
            break;
        case 'url':
            echo '<input type="url" id="sb_'.esc_attr($key).'" name="'.esc_attr($name).'" value="'.esc_attr(is_array($val) ? implode(", ", $val) : (string)$val).'" style="width:100%;">';
            break;
        case 'tel':
        case 'phone':
            $input_type = ($type === 'tel') ? 'tel' : 'text';
            echo '<input type="'.$input_type.'" id="sb_'.esc_attr($key).'" name="'.esc_attr($name).'" value="'.esc_attr(is_array($val) ? implode(", ", $val) : (string)$val).'" style="width:100%;">';
            // Add privacy checkbox for phone/tel fields
            $is_private = get_post_meta($post_id, $key.'_private', true);
            echo '<div class="sb-privacy-option" style="margin-top:8px;">';
            echo '<label for="sb_'.esc_attr($key).'_private" style="display:flex;align-items:center;font-size:13px;color:#666;">';
            echo '<input type="checkbox" id="sb_'.esc_attr($key).'_private" name="sb_'.esc_attr($key).'_private" value="1"'.($is_private ? ' checked' : '').' style="margin-right:6px;"> ';
            echo esc_html__("Keep this phone number private (won't be shown on public profile)", 'speakers-bureau');
            echo '</label></div>';
            break;
        case 'select':
            $options = sb_fb_parse_options($field['placeholder'] ?? '');
            $cur = is_array($val) ? $val : [$val];
            echo '<select id="sb_'.esc_attr($key).'" name="'.esc_attr($name).'" style="width:100%;">';
            echo '<option value="">'.esc_html__('— Select —','speakers-bureau').'</option>';
            foreach ($options as $o) {
                $sel = in_array($o['value'], $cur, true) ? ' selected' : '';
                echo '<option value="'.esc_attr($o['value']).'"'.$sel.'>'.esc_html($o['label']).'</option>';
            }
            echo '</select>';
            break;
        case 'checkbox':
            $options = sb_fb_parse_options($field['placeholder'] ?? '');
            $cur = is_array($val) ? $val : (strlen((string)$val) ? array_map('trim', explode(',', (string)$val)) : []);
            foreach ($options as $idx => $o) {
                $id = 'sb_'.esc_attr($key).'_'.$idx;
                $checked = in_array($o['value'], $cur, true) ? ' checked' : '';
                echo '<label for="'.$id.'" style="display:block;margin:2px 0;">';
                echo '<input type="checkbox" id="'.$id.'" name="'.esc_attr($name).'[]" value="'.esc_attr($o['value']).'"'.$checked.'> ';
                echo esc_html($o['label']).'</label>';
            }
            break;
        case 'image':
        case 'Image':
        case 'Image (Attachment)':
        case 'image_upload':
        case 'attachment':
            $file_field   = 'sb_'.esc_attr($key).'_file';
            $remove_field = 'sb_'.esc_attr($key).'_remove';
            $src = '';
            $attachment_id = 0;

            // Check for existing image value
            if ($val) {
                if (ctype_digit((string)$val)) {
                    $attachment_id = (int)$val;
                    $s = wp_get_attachment_image_src($attachment_id, 'medium');
                    if ($s && !empty($s[0])) $src = $s[0];
                } else {
                    $maybe = esc_url($val);
                    if ($maybe) $src = $maybe;
                }
            } else {
                // Fallback: Only use featured image for profile_image field
                if ($key === 'profile_image') {
                    $featured_id = get_post_thumbnail_id($post_id);
                    if ($featured_id) {
                        $s = wp_get_attachment_image_src($featured_id, 'medium');
                        if ($s && !empty($s[0])) {
                            $src = $s[0];
                            $attachment_id = $featured_id;
                        }
                    }
                }

                // Additional fallback: Check for alternative field names (only for profile fields)
                if (!$src && ($key === 'profile_image' || $key === 'avatar' || $key === 'headshot')) {
                    $alternative_fields = ['image', 'photo', 'picture', 'headshot', 'avatar', 'profile_photo'];
                    foreach ($alternative_fields as $alt_field) {
                        if ($alt_field !== $key) {
                            $alt_val = get_post_meta($post_id, $alt_field, true);
                            if ($alt_val && ctype_digit((string)$alt_val)) {
                                $alt_attachment_id = (int)$alt_val;
                                $s = wp_get_attachment_image_src($alt_attachment_id, 'medium');
                                if ($s && !empty($s[0])) {
                                    $src = $s[0];
                                    $attachment_id = $alt_attachment_id;
                                    break;
                                }
                            }
                        }
                    }
                }
            }

            $unique_id = 'sb_'.esc_attr($key).'_'.uniqid();

            echo '<div class="sb-image-field" id="sb_'.esc_attr($key).'_wrap">';

            // Image preview area
            echo '<div class="sb-image-preview-container" style="border: 2px dashed #ccc; border-radius: 8px; padding: 20px; text-align: center; margin-bottom: 16px; background: #fafafa;">';

            if ($src) {
                echo '<div class="sb-image-preview">';
                echo '<img src="'.esc_url($src).'" alt="'.esc_attr($label).'" style="max-width: 300px; max-height: 300px; border-radius: 8px; box-shadow: 0 2px 8px rgba(0,0,0,0.1);">';
                echo '</div>';
            } else {
                echo '<div class="sb-no-image" style="color: #666; padding: 40px;">';
                echo '<div style="font-size: 48px; margin-bottom: 16px;">📷</div>';
                echo '<p>'.esc_html__('No image selected', 'speakers-bureau').'</p>';
                echo '</div>';
            }

            echo '</div>';

            // Button controls
            echo '<div class="sb-image-controls" style="display: flex; gap: 12px; align-items: center; flex-wrap: wrap;">';

            if ($src) {
                // Replace button
                echo '<label for="'.$unique_id.'" class="button button-primary" style="cursor: pointer;">';
                echo '<span class="dashicons dashicons-update" style="margin-right: 4px;"></span>';
                echo esc_html__('Replace Image', 'speakers-bureau');
                echo '</label>';

                // Remove button
                $remove_js = "
                    var field = this.closest('.sb-image-field');
                    var preview = field.querySelector('.sb-image-preview-container');
                    var removeInput = field.querySelector('input[name=\"".$remove_field."\"]');
                    var fileInput = field.querySelector('input[type=\"file\"]');

                    if (removeInput) removeInput.value = '1';
                    if (fileInput) fileInput.value = '';

                    if (preview) {
                        preview.innerHTML = '<div class=\"sb-no-image\" style=\"color: #666; padding: 40px;\"><div style=\"font-size: 48px; margin-bottom: 16px;\">📷</div><p>".esc_js(__('Image will be removed when you save', 'speakers-bureau'))."</p></div>';
                    }

                    // Update buttons
                    var controls = field.querySelector('.sb-image-controls');
                    if (controls) {
                        controls.innerHTML = '<label for=\"".$unique_id."\" class=\"button button-primary\" style=\"cursor: pointer;\"><span class=\"dashicons dashicons-upload\" style=\"margin-right: 4px;\"></span>".esc_js(__('Upload Image', 'speakers-bureau'))."</label><input type=\"file\" id=\"".$unique_id."\" accept=\"image/*\" name=\"".$file_field."\" style=\"display: none;\" onchange=\"sb_preview_image(this)\"><input type=\"hidden\" name=\"".$remove_field."\" value=\"1\">';
                    }
                ";
                echo '<button type="button" class="button button-secondary" onclick="'.esc_attr($remove_js).'">';
                echo '<span class="dashicons dashicons-trash" style="margin-right: 4px;"></span>';
                echo esc_html__('Remove Image', 'speakers-bureau');
                echo '</button>';
            } else {
                // Upload button
                echo '<label for="'.$unique_id.'" class="button button-primary" style="cursor: pointer;">';
                echo '<span class="dashicons dashicons-upload" style="margin-right: 4px;"></span>';
                echo esc_html__('Upload Image', 'speakers-bureau');
                echo '</label>';
            }

            echo '<input type="file" id="'.$unique_id.'" accept="image/*" name="'.$file_field.'" style="display: none;" onchange="sb_preview_image(this)">';
            echo '<input type="hidden" name="'.$remove_field.'" value="">';


            echo '</div>';

            // Help text
            if ($key === 'profile_image') {
                echo '<p class="description" style="margin-top: 12px; color: #666; font-style: italic;">';
                echo esc_html__('This will be used as your profile photo and set as the featured image for SEO.', 'speakers-bureau');
                echo '</p>';
            }

            echo '</div>';
            break;
        default:
            echo '<input type="text" id="sb_'.esc_attr($key).'" name="'.esc_attr($name).'" value="'.esc_attr(is_array($val) ? implode(", ", $val) : (string)$val).'" style="width:100%;">';
            break;
    }

    echo '</div>';
    return ob_get_clean();
}}
/** END helpers */

/** ===== Markup (front-end) ===== */
?>
<div class="sb-wrap sb-edit">
    <?php
    // Show welcome message for new users who just completed verification
    if (isset($_GET['welcome']) && $_GET['welcome'] === '1') {
        echo '<div class="sb-welcome-message" style="margin-bottom: 30px; padding: 20px; background: #d4edda; border: 1px solid #c3e6cb; border-radius: 8px; border-left: 4px solid #28a745;">';
        echo '<h3 style="color: #155724; margin-top: 0; display: flex; align-items: center; gap: 10px;">';
        echo '<span style="font-size: 24px;">🎉</span>';
        echo esc_html__('Welcome! Your email has been verified', 'speakers-bureau');
        echo '</h3>';
        echo '<p style="color: #155724; margin-bottom: 12px;">';
        echo esc_html__('Your account has been created and you\'re now logged in. Please complete your speaker profile below.', 'speakers-bureau');
        echo '</p>';
        echo '<div style="font-size: 14px; color: #0d5d1a;">';
        echo '<p style="margin-bottom: 8px;"><strong>' . esc_html__('Next steps:', 'speakers-bureau') . '</strong></p>';
        echo '<ul style="margin: 0; padding-left: 20px;">';
        echo '<li>' . esc_html__('Fill in your biography, speaking topics, and contact information', 'speakers-bureau') . '</li>';
        echo '<li>' . esc_html__('Upload a professional profile photo', 'speakers-bureau') . '</li>';
        echo '<li>' . esc_html__('Review your privacy settings for email and phone', 'speakers-bureau') . '</li>';
        echo '<li>' . esc_html__('Save your profile - it will be reviewed before being published', 'speakers-bureau') . '</li>';
        echo '</ul>';
        echo '</div>';
        echo '</div>';
    }

    // Get and display profile status
    $post_status = get_post_status($edit_post_id);
    $status_text = '';
    $status_class = '';

    switch ($post_status) {
        case 'publish':
            $status_text = __('Published (Live)', 'speakers-bureau');
            $status_class = 'status-published';
            break;
        case 'draft':
            $status_text = __('Draft (Waiting for Review)', 'speakers-bureau');
            $status_class = 'status-draft';
            break;
        case 'pending':
            $status_text = __('Pending Review', 'speakers-bureau');
            $status_class = 'status-pending';
            break;
        case 'private':
            $status_text = __('Private', 'speakers-bureau');
            $status_class = 'status-private';
            break;
        default:
            $status_text = ucfirst($post_status);
            $status_class = 'status-other';
            break;
    }
    ?>

    <div style="text-align: right; margin-bottom: 10px;">
        <span class="sb-profile-status <?php echo esc_attr($status_class); ?>" style="
            display: inline-block;
            padding: 6px 12px;
            border-radius: 4px;
            font-size: 14px;
            font-weight: 600;
            <?php if ($post_status === 'publish'): ?>
                background: #d4edda;
                color: #155724;
                border: 1px solid #c3e6cb;
            <?php elseif ($post_status === 'draft'): ?>
                background: #fff3cd;
                color: #856404;
                border: 1px solid #ffeaa7;
            <?php else: ?>
                background: #f8f9fa;
                color: #6c757d;
                border: 1px solid #dee2e6;
            <?php endif; ?>
        ">
            <?php esc_html_e('Profile Status:', 'speakers-bureau'); ?> <?php echo esc_html($status_text); ?>
        </span>
        <?php if ($post_status === 'publish'): ?>
            <br>
            <a href="<?php echo esc_url(get_permalink($edit_post_id)); ?>" target="_blank" style="
                display: inline-block;
                margin-top: 8px;
                padding: 4px 8px;
                background: #2271b1;
                color: #fff;
                text-decoration: none;
                border-radius: 3px;
                font-size: 13px;
            ">
                <?php esc_html_e('View Public Profile', 'speakers-bureau'); ?>
                <span style="font-size: 12px;">↗</span>
            </a>
        <?php endif; ?>
    </div>

    <div style="display: flex; justify-content: space-between; align-items: center; margin-bottom: 10px;">
        <h2 class="sb-title" style="margin: 0; color: #0073aa;"><?php echo esc_html(get_the_title($edit_post_id)); ?></h2>
        <a href="<?php echo esc_url(wp_logout_url(get_permalink())); ?>" style="
            display: inline-block;
            padding: 6px 12px;
            background: #f6f7f7;
            color: #646970;
            text-decoration: none;
            border-radius: 3px;
            border: 1px solid #c3c4c7;
            font-size: 13px;
        ">
            <?php esc_html_e('Logout', 'speakers-bureau'); ?>
        </a>
    </div>

    <form method="post" action="" enctype="multipart/form-data">
        <?php wp_nonce_field('sb_edit_'.$edit_post_id, 'sb_edit_nonce'); ?>

        <div class="sb-input sb-input-text">
            <label for="sb_post_title" class="sb-label"><?php esc_html_e('Speaker Name','speakers-bureau'); ?></label>
            <input type="text" id="sb_post_title" name="sb_post_title" value="<?php echo esc_attr(get_the_title($edit_post_id)); ?>" style="width:100%;">
        </div>

        <div class="sb-edit-grid">
            <?php foreach ([1,2,3,4,5,6,7] as $sec):
                $class   = !empty($layout[$sec]['class']) ? $layout[$sec]['class'] : 'sb-section-'.$sec;
                $title   = isset($layout[$sec]['title']) ? trim($layout[$sec]['title']) : '';
                $heading = isset($layout[$sec]['heading']) ? strtolower($layout[$sec]['heading']) : 'h3';
                if (!in_array($heading, ['h1','h2','h3','h4','h5','h6'], true)) $heading = 'h3';
                $sec_fields = $fields_by_section[$sec] ?? [];
                if (!$sec_fields) continue; ?>
            <section class="<?php echo esc_attr($class); ?>">
                <?php if ($title !== ''): ?>
                    <<?php echo esc_attr($heading); ?> class="sb-section-title"><?php echo wp_kses_post($title); ?></<?php echo esc_attr($heading); ?>>
                <?php endif; ?>

                <?php foreach ($sec_fields as $f) echo sb_render_edit_control($f, $edit_post_id); ?>
            </section>
            <?php endforeach; ?>
        </div>

        <div style="background: #f9f9f9; border: 1px solid #ddd; padding: 15px; border-radius: 6px; margin: 20px 0;">
            <h4 style="margin-top: 0; color: #0073aa;"><?php esc_html_e('Email Notifications', 'speakers-bureau'); ?></h4>
            <?php
            $is_admin = current_user_can('manage_options');
            $send_email_default = $is_admin ? false : true; // Unchecked for admins, checked for regular users
            ?>
            <label style="display: flex; align-items: center; font-size: 14px;">
                <input type="checkbox" name="sb_send_update_email" value="1" <?php echo $send_email_default ? 'checked' : ''; ?> style="margin-right: 8px;">
                <?php esc_html_e('Send me an email notification when my profile is updated', 'speakers-bureau'); ?>
            </label>
            <p class="description" style="margin: 8px 0 0 24px; color: #666; font-size: 13px;">
                <?php esc_html_e('When checked, you will receive an email with your complete profile information whenever changes are saved.', 'speakers-bureau'); ?>
            </p>

            <?php if ($is_admin) : ?>
                <?php
                // Get current suppress notification setting (checked by default for new records)
                $suppress_notification = get_post_meta($edit_post_id, '_sb_suppress_update_notification', true);
                if ($suppress_notification === '') {
                    $suppress_notification = '1'; // Default to checked (suppress)
                }
                ?>
                <div style="margin-top: 15px; padding-top: 15px; border-top: 1px solid #ddd;">
                    <label style="display: flex; align-items: flex-start; font-size: 14px;">
                        <input type="checkbox" name="sb_suppress_update_notification" value="1" <?php checked($suppress_notification, '1'); ?> style="margin-right: 8px; margin-top: 2px; flex-shrink: 0;">
                        <span>
                            <strong><?php esc_html_e('Do NOT send notification when updating this record', 'speakers-bureau'); ?></strong>
                            <p class="description" style="margin: 4px 0 0 0; color: #666; font-size: 13px;">
                                <?php esc_html_e('When checked, email notifications to the speaker will be suppressed even if they have requested updates.', 'speakers-bureau'); ?>
                            </p>
                        </span>
                    </label>
                </div>
            <?php endif; ?>
        </div>

        <p><button type="submit" class="button button-primary"><?php esc_html_e('Save Profile','speakers-bureau'); ?></button></p>
    </form>

    <?php
    // Profile Statistics Section
    $views = get_post_meta($edit_post_id, '_speaker_views', true);
    $views = $views ? intval($views) : 0;
    $last_viewed = get_post_meta($edit_post_id, '_speaker_last_viewed', true);
    $date_added = get_the_date('Y-m-d H:i:s', $edit_post_id);
    $last_updated = get_the_modified_date('Y-m-d H:i:s', $edit_post_id);
    ?>

    <div style="margin-top: 30px; padding: 20px; background: #f9f9f9; border: 1px solid #e0e0e0; border-radius: 6px;">
        <h3 style="margin-top: 0; color: #1d2327; font-size: 16px; margin-bottom: 15px;">
            <?php esc_html_e('Profile Statistics', 'speakers-bureau'); ?>
        </h3>

        <div style="display: grid; grid-template-columns: repeat(auto-fit, minmax(200px, 1fr)); gap: 15px; font-size: 14px;">
            <div>
                <strong style="color: #2271b1;"><?php echo number_format($views); ?></strong>
                <br>
                <span style="color: #646970;"><?php esc_html_e('Profile Views', 'speakers-bureau'); ?></span>
            </div>

            <div>
                <strong style="color: #2271b1;"><?php echo esc_html(date('M j, Y', strtotime($date_added))); ?></strong>
                <br>
                <span style="color: #646970;"><?php esc_html_e('Date Added', 'speakers-bureau'); ?></span>
            </div>

            <div>
                <strong style="color: #2271b1;"><?php echo esc_html(date('M j, Y', strtotime($last_updated))); ?></strong>
                <br>
                <span style="color: #646970;"><?php esc_html_e('Last Updated', 'speakers-bureau'); ?></span>
            </div>

            <?php if ($last_viewed): ?>
            <div>
                <?php
                $last_viewed_date = new DateTime($last_viewed);
                $now = new DateTime();
                $diff = $now->diff($last_viewed_date);

                if ($diff->days == 0) {
                    $last_viewed_text = __('Today', 'speakers-bureau');
                    $color = '#00a32a';
                } elseif ($diff->days == 1) {
                    $last_viewed_text = __('Yesterday', 'speakers-bureau');
                    $color = '#dba617';
                } elseif ($diff->days <= 7) {
                    $last_viewed_text = sprintf(__('%d days ago', 'speakers-bureau'), $diff->days);
                    $color = '#dba617';
                } else {
                    $last_viewed_text = $last_viewed_date->format('M j, Y');
                    $color = '#2271b1';
                }
                ?>
                <strong style="color: <?php echo esc_attr($color); ?>;"><?php echo esc_html($last_viewed_text); ?></strong>
                <br>
                <span style="color: #646970;"><?php esc_html_e('Last Viewed', 'speakers-bureau'); ?></span>
            </div>
            <?php else: ?>
            <div>
                <strong style="color: #646970;"><?php esc_html_e('Never', 'speakers-bureau'); ?></strong>
                <br>
                <span style="color: #646970;"><?php esc_html_e('Last Viewed', 'speakers-bureau'); ?></span>
            </div>
            <?php endif; ?>
        </div>

        <?php if ($views > 0): ?>
        <div style="margin-top: 15px; padding-top: 15px; border-top: 1px solid #e0e0e0; font-size: 13px; color: #646970;">
            <p style="margin: 0;">
                <?php if ($views == 1): ?>
                    <?php esc_html_e('Your profile has been viewed once.', 'speakers-bureau'); ?>
                <?php else: ?>
                    <?php printf(esc_html__('Your profile has been viewed %s times.', 'speakers-bureau'), '<strong>' . number_format($views) . '</strong>'); ?>
                <?php endif; ?>

                <?php if ($last_viewed): ?>
                    <?php printf(esc_html__('Most recent view was %s.', 'speakers-bureau'), '<strong>' . esc_html($last_viewed_text) . '</strong>'); ?>
                <?php endif; ?>
            </p>
        </div>
        <?php else: ?>
        <div style="margin-top: 15px; padding-top: 15px; border-top: 1px solid #e0e0e0; font-size: 13px; color: #646970;">
            <p style="margin: 0;">
                <?php esc_html_e('Your profile hasn\'t been viewed yet. Make sure it\'s published and share your profile link to increase visibility.', 'speakers-bureau'); ?>
            </p>
        </div>
        <?php endif; ?>
    </div>
</div>

<script>
function sb_preview_image(input) {
    if (input.files && input.files[0]) {
        var reader = new FileReader();
        var field = input.closest('.sb-image-field');
        var previewContainer = field ? field.querySelector('.sb-image-preview-container') : null;
        var controls = field ? field.querySelector('.sb-image-controls') : null;
        var removeInput = field ? field.querySelector('input[type="hidden"]') : null;

        reader.onload = function(e) {
            if (previewContainer) {
                previewContainer.innerHTML = '<div class="sb-image-preview"><img src="' + e.target.result + '" alt="Preview" style="max-width: 300px; max-height: 300px; border-radius: 8px; box-shadow: 0 2px 8px rgba(0,0,0,0.1);"></div>';
            }

            // Update controls to show replace/remove buttons WITHOUT recreating the file input
            if (controls) {
                var fieldName = input.name;
                var removeFieldName = fieldName.replace('_file', '_remove');
                var uniqueId = input.id;

                // Keep the existing file input and hidden input
                var existingFileInput = controls.querySelector('input[type="file"]');
                var existingHiddenInput = controls.querySelector('input[type="hidden"]');

                // Update only the buttons
                controls.innerHTML =
                    '<label for="' + uniqueId + '" class="button button-primary" style="cursor: pointer;">' +
                    '<span class="dashicons dashicons-update" style="margin-right: 4px;"></span>' +
                    '<?php echo esc_js(__('Replace Image', 'speakers-bureau')); ?>' +
                    '</label>' +
                    '<button type="button" class="button button-secondary" onclick="sb_remove_image_preview(this, \'' + removeFieldName + '\', \'' + uniqueId + '\')">' +
                    '<span class="dashicons dashicons-trash" style="margin-right: 4px;"></span>' +
                    '<?php echo esc_js(__('Remove Image', 'speakers-bureau')); ?>' +
                    '</button>';

                // Re-append the existing inputs to preserve file selection
                if (existingFileInput) {
                    controls.appendChild(existingFileInput);
                }
                if (existingHiddenInput) {
                    existingHiddenInput.value = ''; // Reset remove flag
                    controls.appendChild(existingHiddenInput);
                } else {
                    // Create hidden input if it doesn't exist
                    var hiddenInput = document.createElement('input');
                    hiddenInput.type = 'hidden';
                    hiddenInput.name = removeFieldName;
                    hiddenInput.value = '';
                    controls.appendChild(hiddenInput);
                }
            }

            // Reset remove flag when new image is selected
            if (removeInput) {
                removeInput.value = '';
            }
        };

        reader.readAsDataURL(input.files[0]);
    }
}

function sb_remove_image_preview(button, removeFieldName, uniqueId) {
    var field = button.closest('.sb-image-field');
    var preview = field ? field.querySelector('.sb-image-preview-container') : null;
    var controls = field ? field.querySelector('.sb-image-controls') : null;
    var removeInput = field ? field.querySelector('input[name="' + removeFieldName + '"]') : null;

    if (removeInput) removeInput.value = '1';

    if (preview) {
        preview.innerHTML = '<div class="sb-no-image" style="color: #666; padding: 40px;"><div style="font-size: 48px; margin-bottom: 16px;">📷</div><p><?php echo esc_js(__('Image will be removed when you save', 'speakers-bureau')); ?></p></div>';
    }

    if (controls) {
        var fieldName = removeFieldName.replace('_remove', '_file');
        controls.innerHTML =
            '<label for="' + uniqueId + '" class="button button-primary" style="cursor: pointer;">' +
            '<span class="dashicons dashicons-upload" style="margin-right: 4px;"></span>' +
            '<?php echo esc_js(__('Upload Image', 'speakers-bureau')); ?>' +
            '</label>' +
            '<input type="file" id="' + uniqueId + '" accept="image/*" name="' + fieldName + '" style="display: none;" onchange="sb_preview_image(this)">' +
            '<input type="hidden" name="' + removeFieldName + '" value="1">';
    }
}
</script>

<?php
/** END OF FILE: templates/speaker-edit.php */
