<?php
/**
 * File: templates/single-speaker.php
 * Purpose: Single Speaker profile rendering with robust admin Edit-Link placement
 * Notes:
 *   - Preserves support for Form Builder field type "edit_link".
 *   - Button is ALWAYS shown at the bottom of Section 1 (per request). If an "edit_link"
 *     field exists, its text is used for the button label; location is standardized.
 *   - Link goes to /my-profile/?speaker_id=### (not the block editor).
 */

if (!defined('ABSPATH')) exit;

get_header();

/** Current post id */
$post_id = get_the_ID();

/** Track profile view (increment view count) */
if (!function_exists('sb_track_profile_view')) {
function sb_track_profile_view($speaker_id) {
    // Don't track views for admins or the speaker themselves
    if (current_user_can('manage_options')) return;

    // Don't track if this is the speaker viewing their own profile
    if (is_user_logged_in()) {
        $current_user = wp_get_current_user();
        $speaker_email = get_post_meta($speaker_id, 'email', true);
        if ($current_user->user_email === $speaker_email) return;

        $speaker_user_id = get_post_meta($speaker_id, '_speaker_user_id', true);
        if ($speaker_user_id && $current_user->ID == $speaker_user_id) return;
    }

    // Get current view count
    $views = get_post_meta($speaker_id, '_speaker_views', true);
    $views = $views ? intval($views) : 0;

    // Increment and save
    update_post_meta($speaker_id, '_speaker_views', $views + 1);

    // Also track last viewed date
    update_post_meta($speaker_id, '_speaker_last_viewed', current_time('mysql'));
}}

// Track the view for this profile
sb_track_profile_view($post_id);

/** Helper: resolve /my-profile/ URL and attach speaker_id */
if (!function_exists('sb_my_profile_url')) {
function sb_my_profile_url($speaker_post_id) {
    $page = get_page_by_path('my-profile');
    $base = $page ? get_permalink($page) : home_url('/my-profile/');
    return add_query_arg('speaker_id', intval($speaker_post_id), $base);
}}
/** END helper: sb_my_profile_url */

/** Defaults for layout (fallback if option is missing/partial) */
$defaults_layout = [
    1 => ['title'=>'Profile','heading'=>'h4','display'=>1,'hidden'=>0,'class'=>'sb-section-1'],
    2 => ['title'=>'Overview','heading'=>'h2','display'=>1,'hidden'=>0,'class'=>'sb-section-2'],
    3 => ['title'=>'Topics','heading'=>'h3','display'=>1,'hidden'=>0,'class'=>'sb-section-3'],
    4 => ['title'=>'Background','heading'=>'h3','display'=>1,'hidden'=>0,'class'=>'sb-section-4'],
    5 => ['title'=>'','heading'=>'h3','display'=>0,'hidden'=>1,'class'=>'sb-section-5'],
    6 => ['title'=>'','heading'=>'h3','display'=>0,'hidden'=>1,'class'=>'sb-section-6'],
    7 => ['title'=>'More Information','heading'=>'h2','display'=>1,'hidden'=>0,'class'=>'sb-section-7'],
    8 => ['title'=>'Private Fields','heading'=>'h3','display'=>0,'hidden'=>1,'class'=>'sb-section-8'],
];

/** Get layout and normalize with defaults */
$layout = get_option('sb_profile_layout', $defaults_layout);
foreach (range(1,8) as $i) {
    $layout[$i] = wp_parse_args(isset($layout[$i]) ? (array)$layout[$i] : [], $defaults_layout[$i]);
}

/** Fields (normalized from builder) */
$fields = function_exists('sb_normalize_form_fields') ? sb_normalize_form_fields() : [];
if (!is_array($fields)) $fields = [];

/** Track edit-link intent (text comes from builder if provided) */
$edit_link_text      = null;  // custom text from edit_link field
$had_edit_link_field = false; // whether any edit_link exists in builder (enabled)

/** Group fields by section for display (skip section 8 + hidden sections) */
$fields_by_section = [];
foreach ($fields as $f) {
    if (empty($f['enabled'])) continue;

    $sec = isset($f['section']) ? intval($f['section']) : 2;
    if ($sec < 1 || $sec > 8) $sec = 2;
    if ($sec === 8) continue; // never show private fields on public profile
    if (!empty($layout[$sec]['hidden'])) continue;

    // Capture edit_link presence + text, but do not render it inline;
    // we standardize placement at the bottom of Section 1.
    if (($f['type'] ?? '') === 'edit_link') {
        $had_edit_link_field = true;
        $txt = isset($f['placeholder']) ? trim($f['placeholder']) : '';
        if ($txt !== '') $edit_link_text = $txt;
        continue;
    }

    $fields_by_section[$sec][] = $f;
}

/** Featured image URL (avoid duplicating via an image field) */
$featured_url = '';
if (has_post_thumbnail($post_id)) {
    $img = wp_get_attachment_image_src(get_post_thumbnail_id($post_id), 'large');
    if (!empty($img[0])) $featured_url = $img[0];
}

/** Render a single field value (excluding edit_link which is handled globally) */
if (!function_exists('sb_render_field_value_single')) {
function sb_render_field_value_single($post_id, $field, $featured_url) {
    $type = isset($field['type']) ? $field['type'] : 'text';
    $key  = isset($field['key'])  ? $field['key']  : '';

    if ($type === 'edit_link') return ''; // handled globally at bottom of Section 1
    if ($key === '') return '';

    // Check if this field is marked as private
    if (in_array($type, ['email', 'phone', 'tel'])) {
        $is_private = get_post_meta($post_id, $key.'_private', true);
        if ($is_private) {
            return ''; // Don't display private fields on public profile
        }
    }

    $raw = get_post_meta($post_id, $key, true);

    // sensible fallbacks
    if ($raw === '' || $raw === null) {
        if ($key === 'headline' && has_excerpt($post_id)) $raw = get_the_excerpt($post_id);
        if (in_array($key, ['overview','about','bio','description'], true)) {
            $content = get_post_field('post_content', $post_id);
            if (!empty($content)) $raw = $content;
        }
    }
    if ($raw === '' || $raw === null) return '';

    switch ($type) {
        case 'textarea':
            // Ensure proper paragraph formatting for textareas
            $content = trim($raw);
            if (empty($content)) return '';

            // Apply WordPress auto-paragraph formatting
            return wpautop(wp_kses_post($content));
        case 'url':
            $u = esc_url($raw);
            if ($u) {
                // Get Font Awesome icon from field configuration
                $fa_icon = isset($field['fa_icon']) && !empty($field['fa_icon']) ? $field['fa_icon'] : 'fa-globe';
                $icon_html = sprintf('<i class="fa %s" style="font-size: 20px; margin-right: 8px; color: #333;"></i>', esc_attr($fa_icon));
                return sprintf('%s<a href="%s" target="_blank" rel="noopener">%s</a>', $icon_html, $u, $u);
            }
            return '';
        case 'email':
            $em = sanitize_email($raw);
            if ($em) {
                // Get Font Awesome icon from field configuration, default to envelope
                $fa_icon = isset($field['fa_icon']) && !empty($field['fa_icon']) ? $field['fa_icon'] : 'fa-envelope';
                $icon_html = sprintf('<i class="fa %s" style="font-size: 20px; margin-right: 8px; color: #333;"></i>', esc_attr($fa_icon));
                return sprintf('%s<a href="mailto:%s">%s</a>', $icon_html, esc_attr($em), esc_html($em));
            }
            return '';
        case 'image':
            // avoid duplicate featured image
            if (is_numeric($raw)) {
                $src = wp_get_attachment_image_src((int)$raw, 'medium_large');
                $large_src = wp_get_attachment_image_src((int)$raw, 'large');
                if ($src && !empty($src[0])) {
                    // Check if this is the same attachment ID as the featured image
                    if (has_post_thumbnail($post_id) && (int)$raw === get_post_thumbnail_id($post_id)) {
                        return ''; // Skip if this is the featured image
                    }
                    // Also check URL match as fallback
                    if ($featured_url && $src[0] === $featured_url) return '';
                    $large_url = ($large_src && !empty($large_src[0])) ? $large_src[0] : $src[0];
                    return sprintf('<img class="sb-topic-image sb-lightbox-trigger" src="%s" data-large="%s" alt="" style="cursor: pointer;" />', esc_url($src[0]), esc_url($large_url));
                }
            } else {
                $u = esc_url($raw);
                if ($u) {
                    if ($featured_url && $u === $featured_url) return '';
                    return sprintf('<img class="sb-topic-image sb-lightbox-trigger" src="%s" data-large="%s" alt="" style="cursor: pointer;" />', $u, $u);
                }
            }
            return '';
        case 'phone':
            $digits = preg_replace('/\D+/', '', $raw);
            $disp   = esc_html($raw);
            return $digits ? sprintf('<a href="tel:%s">%s</a>', esc_attr($digits), $disp) : $disp;
        case 'select':
        case 'checkbox':
            if (is_array($raw)) {
                $safe = array_map('esc_html', array_filter($raw));
                return implode(', ', $safe);
            }
            return esc_html($raw);
        default: // text
            if (is_array($raw)) {
                $safe = array_map('esc_html', array_filter($raw));
                return implode(', ', $safe);
            }
            return esc_html($raw);
    }
}}
/** END helper: sb_render_field_value_single */

/** Render a section */
if (!function_exists('sb_render_section_block_single')) {
function sb_render_section_block_single($section_index, $post_id, $layout, $fields_by_section, $featured_url) {

    if (!empty($layout[$section_index]['hidden'])) return;

    $class      = !empty($layout[$section_index]['class']) ? $layout[$section_index]['class'] : 'sb-section-'.$section_index;
    $title      = isset($layout[$section_index]['title']) ? trim((string)$layout[$section_index]['title']) : '';
    $heading    = isset($layout[$section_index]['heading']) ? strtolower((string)$layout[$section_index]['heading']) : 'h3';
    $show_title = !empty($layout[$section_index]['display']);
    if (!in_array($heading, ['h1','h2','h3','h4','h5','h6'], true)) $heading = 'h3';

    $fields = isset($fields_by_section[$section_index]) ? $fields_by_section[$section_index] : [];

    // First, check if there are any fields with actual data
    $has_content = false;
    $field_outputs = [];

    foreach ($fields as $f) {
        $val_html = sb_render_field_value_single($post_id, $f, $featured_url);
        if ($val_html !== '' && $val_html !== null) {
            $has_content = true;
            $field_outputs[] = ['field' => $f, 'html' => $val_html];
        }
    }

    // Only render section if it has content
    if ($has_content) {
        echo '<section class="'.esc_attr($class).'">';

        // Show title only if section has content
        if ($show_title && $title !== '') {
            printf('<%1$s class="sb-section-title">%2$s</%1$s>', esc_attr($heading), wp_kses_post($title));
        }

        // Render the fields that have content
        foreach ($field_outputs as $output) {
            $f = $output['field'];
            $val_html = $output['html'];

            $is_img = (isset($f['type']) && $f['type'] === 'image');
            $is_textarea = (isset($f['type']) && $f['type'] === 'textarea');

            if (empty($f['hide_label']) && !empty($f['label']) && !$is_img) {
                echo '<div class="sb-field"><strong class="sb-label">'.esc_html($f['label']).'</strong> ';
                if ($is_textarea) {
                    echo '<div class="sb-field-content">'.$val_html.'</div></div>';
                } else {
                    echo '<span class="sb-value">'.$val_html.'</span></div>';
                }
            } else {
                if ($is_textarea) {
                    echo '<div class="sb-field"><div class="sb-field-content">'.$val_html.'</div></div>';
                } else {
                    echo '<div class="sb-field"><span class="sb-value">'.$val_html.'</span></div>';
                }
            }
        }

        echo '</section>';
    }
}}
/** END helper: sb_render_section_block_single */

?>
<div class="sb-wrap sb-single">
    <?php while (have_posts()) : the_post(); ?>
        <header class="sb-header">
            <h1 class="sb-title"><?php the_title(); ?></h1>
            <?php
            $headline = get_post_meta($post_id, 'headline', true);
            if (!$headline && has_excerpt($post_id)) $headline = get_the_excerpt($post_id);
            if ($headline) echo '<div class="sb-headline">'.esc_html($headline).'</div>';
            ?>
        </header>

        <div class="sb-single-grid">
            <!-- Left column: featured image + Section 1 -->
            <aside class="sb-single-left">
                <?php if (has_post_thumbnail($post_id)):
                    $featured_id = get_post_thumbnail_id($post_id);
                    $featured_src = wp_get_attachment_image_src($featured_id, 'large');
                    $featured_large_src = wp_get_attachment_image_src($featured_id, 'full');
                    if ($featured_src && !empty($featured_src[0])) {
                        $large_url = ($featured_large_src && !empty($featured_large_src[0])) ? $featured_large_src[0] : $featured_src[0];
                        echo sprintf('<img class="sb-topic-image sb-lightbox-trigger" src="%s" data-large="%s" alt="" style="cursor: pointer;" />', esc_url($featured_src[0]), esc_url($large_url));
                    }
                endif; ?>

                <?php
                // Render Section 1 fields
                sb_render_section_block_single(1, $post_id, $layout, $fields_by_section, $featured_url);

                // Show "Send Message" button if contact info is private or missing
                if (function_exists('sb_should_show_contact_button') && sb_should_show_contact_button($post_id)) {
                    $speaker_name = get_the_title();
                    echo '<div class="sb-field">';
                    echo '<span class="sb-value">';
                    echo '<button class="sb-edit-link button button-primary sb-contact-speaker-btn" data-speaker-id="' . esc_attr($post_id) . '">';
                    echo '<span class="dashicons dashicons-email sb-contact-icon"></span>';
                    echo '<span class="sb-contact-btn-text">';
                    echo '<span class="sb-contact-btn-line1">' . esc_html__('Send a message to', 'speakers-bureau') . '</span>';
                    echo '<span class="sb-contact-btn-line2">' . esc_html($speaker_name) . '</span>';
                    echo '</span>';
                    echo '</button>';
                    echo '</span>';
                    echo '</div>';
                }

                // Admin-only "Edit this profile" at the VERY BOTTOM of Section 1
                if ( current_user_can('manage_options') ) {
                    $text = $edit_link_text ?: __('Edit My Profile', 'speakers-bureau');
                    $url  = sb_my_profile_url($post_id);
                    echo '<div class="sb-field"><span class="sb-value"><a class="sb-edit-link button button-primary" href="'.esc_url($url).'"><span class="dashicons dashicons-edit" style="margin-right: 6px; font-size: 16px; vertical-align: middle;"></span>'.esc_html($text).'</a></span></div>';
                }
                ?>
            </aside>

            <!-- Right column: Sections 2–6 -->
            <main class="sb-single-right">
                <?php foreach ([2,3,4,5,6] as $sec) {
                    sb_render_section_block_single($sec, $post_id, $layout, $fields_by_section, $featured_url);
                } ?>
            </main>

            <!-- Bottom full width: Section 7 -->
            <div class="sb-single-bottom">
                <?php sb_render_section_block_single(7, $post_id, $layout, $fields_by_section, $featured_url); ?>
            </div>
        </div>
    <?php endwhile; ?>

    <!-- Image Lightbox Modal -->
    <div id="sb-lightbox" class="sb-lightbox" style="display: none;">
        <div class="sb-lightbox-overlay"></div>
        <div class="sb-lightbox-container">
            <button class="sb-lightbox-close">&times;</button>
            <img class="sb-lightbox-image" src="" alt="" />
        </div>
    </div>

    <!-- Contact Form Modal -->
    <div id="sb-contact-modal" class="sb-modal" style="display: none;">
        <div class="sb-modal-overlay"></div>
        <div class="sb-modal-container">
            <button class="sb-modal-close">&times;</button>
            <div class="sb-modal-content">
                <h3><?php echo esc_html(sprintf(__('Send a Message to %s', 'speakers-bureau'), get_the_title())); ?></h3>
                <div id="sb-contact-message"></div>
                <form id="sb-contact-form">
                    <input type="hidden" name="speaker_id" value="<?php echo esc_attr($post_id); ?>">
                    <input type="hidden" name="captcha_hash" id="sb-captcha-hash">

                    <div class="sb-form-field">
                        <label><?php esc_html_e('Your Name', 'speakers-bureau'); ?> <span class="required">*</span></label>
                        <input type="text" name="sender_name" required>
                    </div>

                    <div class="sb-form-field">
                        <label><?php esc_html_e('Your Email', 'speakers-bureau'); ?> <span class="required">*</span></label>
                        <input type="email" name="sender_email" required>
                    </div>

                    <div class="sb-form-field">
                        <label><?php esc_html_e('Your Phone', 'speakers-bureau'); ?></label>
                        <input type="tel" name="sender_phone">
                    </div>

                    <div class="sb-form-field">
                        <label><?php esc_html_e('Message', 'speakers-bureau'); ?> <span class="required">*</span></label>
                        <textarea name="message" rows="5" required></textarea>
                    </div>

                    <div class="sb-form-field">
                        <label><?php esc_html_e('Security Question', 'speakers-bureau'); ?> <span class="required">*</span></label>
                        <div class="sb-captcha-question" id="sb-captcha-question"></div>
                        <input type="number" name="captcha_answer" required>
                    </div>

                    <button type="submit" class="button button-primary">
                        <?php esc_html_e('Send Message', 'speakers-bureau'); ?>
                    </button>
                </form>
            </div>
        </div>
    </div>
</div>

<style>
/* Lightbox Styles */
.sb-lightbox {
    position: fixed;
    top: 0;
    left: 0;
    width: 100%;
    height: 100%;
    z-index: 9999;
    display: flex;
    align-items: center;
    justify-content: center;
}

.sb-lightbox-overlay {
    position: absolute;
    top: 0;
    left: 0;
    width: 100%;
    height: 100%;
    background: rgba(0, 0, 0, 0.8);
    cursor: pointer;
}

.sb-lightbox-container {
    position: relative;
    max-width: 90vw;
    max-height: 90vh;
    z-index: 10000;
}

.sb-lightbox-image {
    max-width: 100%;
    max-height: 90vh;
    width: auto;
    height: auto;
    border-radius: 8px;
    box-shadow: 0 4px 20px rgba(0, 0, 0, 0.5);
}

.sb-lightbox-close {
    position: absolute;
    top: -40px;
    right: 0;
    background: rgba(255, 255, 255, 0.9);
    border: none;
    width: 40px;
    height: 40px;
    border-radius: 50%;
    font-size: 24px;
    font-weight: bold;
    cursor: pointer;
    display: flex;
    align-items: center;
    justify-content: center;
    color: #333;
    transition: background-color 0.2s;
}

.sb-lightbox-close:hover {
    background: rgba(255, 255, 255, 1);
}

/* Loading state */
.sb-lightbox-image.loading {
    opacity: 0.5;
}

/* Contact Form Modal Styles */
.sb-modal {
    position: fixed;
    top: 0;
    left: 0;
    width: 100%;
    height: 100%;
    z-index: 9999;
    display: flex;
    align-items: center;
    justify-content: center;
}

.sb-modal-overlay {
    position: absolute;
    top: 0;
    left: 0;
    width: 100%;
    height: 100%;
    background: rgba(0, 0, 0, 0.7);
    cursor: pointer;
}

.sb-modal-container {
    position: relative;
    max-width: 600px;
    width: 90%;
    max-height: 90vh;
    overflow-y: auto;
    background: white;
    border-radius: 8px;
    box-shadow: 0 4px 20px rgba(0, 0, 0, 0.5);
    z-index: 10000;
}

.sb-modal-content {
    padding: 30px;
}

.sb-modal-close {
    position: absolute;
    top: 10px;
    right: 10px;
    background: transparent;
    border: none;
    font-size: 28px;
    font-weight: bold;
    cursor: pointer;
    color: #666;
    width: 40px;
    height: 40px;
    display: flex;
    align-items: center;
    justify-content: center;
    transition: color 0.2s;
}

.sb-modal-close:hover {
    color: #000;
}

.sb-form-field {
    margin-bottom: 20px;
}

.sb-form-field label {
    display: block;
    font-weight: bold;
    margin-bottom: 5px;
}

.sb-form-field input[type="text"],
.sb-form-field input[type="email"],
.sb-form-field input[type="tel"],
.sb-form-field input[type="number"],
.sb-form-field textarea {
    width: 100%;
    padding: 10px;
    border: 1px solid #ddd;
    border-radius: 4px;
    font-size: 14px;
    box-sizing: border-box;
}

.sb-form-field textarea {
    resize: vertical;
}

.sb-form-field .required {
    color: #d63638;
}

.sb-captcha-question {
    font-size: 18px;
    font-weight: bold;
    padding: 10px;
    background: #f0f0f1;
    border-radius: 4px;
    margin-bottom: 10px;
}

#sb-contact-message {
    margin-bottom: 20px;
    padding: 10px;
    border-radius: 4px;
    display: none;
}

#sb-contact-message.success {
    display: block;
    background: #d1f0d8;
    border: 1px solid #4caf50;
    color: #155724;
}

#sb-contact-message.error {
    display: block;
    background: #f8d7da;
    border: 1px solid #d63638;
    color: #721c24;
}

.sb-contact-speaker-btn {
    display: inline-flex;
    align-items: center;
}
</style>

<script>
document.addEventListener('DOMContentLoaded', function() {
    const lightbox = document.getElementById('sb-lightbox');
    const lightboxImage = lightbox.querySelector('.sb-lightbox-image');
    const closeButton = lightbox.querySelector('.sb-lightbox-close');
    const overlay = lightbox.querySelector('.sb-lightbox-overlay');

    // Function to open lightbox
    function openLightbox(imageSrc) {
        lightboxImage.classList.add('loading');
        lightboxImage.src = imageSrc;
        lightbox.style.display = 'flex';
        document.body.style.overflow = 'hidden'; // Prevent background scroll

        // Remove loading state when image loads
        lightboxImage.onload = function() {
            lightboxImage.classList.remove('loading');
        };
    }

    // Function to close lightbox
    function closeLightbox() {
        lightbox.style.display = 'none';
        document.body.style.overflow = ''; // Restore scroll
        lightboxImage.src = '';
    }

    // Add click listeners to all lightbox trigger images
    document.querySelectorAll('.sb-lightbox-trigger').forEach(function(img) {
        img.addEventListener('click', function() {
            const largeImageSrc = this.getAttribute('data-large') || this.src;
            openLightbox(largeImageSrc);
        });
    });

    // Close lightbox when clicking close button
    closeButton.addEventListener('click', closeLightbox);

    // Close lightbox when clicking overlay
    overlay.addEventListener('click', closeLightbox);

    // Close lightbox when pressing Escape key
    document.addEventListener('keydown', function(e) {
        if (e.key === 'Escape' && lightbox.style.display === 'flex') {
            closeLightbox();
        }
    });
});
</script>

<?php
get_footer();

/** END OF FILE: templates/single-speaker.php */
