<?php
/**
 * File: includes/sb-shortcodes.php
 * Purpose: Part of the Speakers Bureau plugin.
 */

if (!defined('ABSPATH')) exit;

function sb_render_speaker_list($atts) {
    $atts = shortcode_atts(['per_page'=>24,'columns'=>3], $atts, 'speaker_list');

    $paged = max(1, (int) (get_query_var('paged') ?: (isset($_GET['paged']) ? (int)$_GET['paged'] : 1)));
    $q     = isset($_GET['q']) ? sanitize_text_field(wp_unslash($_GET['q'])) : '';

    $args = [
        'post_type'      => 'speaker',
        'posts_per_page' => (int) $atts['per_page'],
        'paged'          => $paged,
    ];

    if ($q !== '') {
        $form_fields = sb_normalize_form_fields(get_option('sb_form_fields'));
        $keys = [];
        foreach ($form_fields as $f) {
            if (!empty($f['enabled']) && !empty($f['searchable']) && !in_array($f['type'], ['featured_image','image','section_header'], true)) {
                $keys[] = $f['key'];
            }
        }
        $meta_or = ['relation' => 'OR'];
        foreach ($keys as $k) {
            $meta_or[] = ['key'=>$k,'value'=>$q,'compare'=>'LIKE'];
        }

        // DON'T use 's' parameter as it's being corrupted by security filters
        // Instead we'll add a custom WHERE clause for post_title
        if (count($meta_or) > 1) {
            $args['meta_query'] = [$meta_or];
        }

        // Add custom WHERE clause for post_title search
        add_filter('posts_where', function($where) use ($q) {
            global $wpdb;
            $search_term = '%' . $wpdb->esc_like($q) . '%';
            $where .= $wpdb->prepare(" OR {$wpdb->posts}.post_title LIKE %s", $search_term);
            return $where;
        }, 10, 1);
    }

    // Allow radius search module to alter query
    $args = apply_filters('sb_speaker_list_query_args', $args);

    $query = new WP_Query($args);

    $form_fields = sb_normalize_form_fields(get_option('sb_form_fields'));

    ob_start();
    $template = plugin_dir_path(dirname(__FILE__)) . 'templates/speaker-list.php';
    if (file_exists($template)) {
        include $template;
    } else {
        // Minimal fallback
        echo '<form method="get" class="sb-search-bar" style="margin-bottom:1rem;">';
        echo '<input type="text" name="q" placeholder="'.esc_attr__('Search speakers...','sb').'" value="'.esc_attr($q).'" /> ';
        echo '<input type="text" name="zip" placeholder="'.esc_attr__('Zip','sb').'" value="'.esc_attr($_GET['zip'] ?? '').'" size="6" /> ';
        echo '<select name="radius">
                <option value="">' . esc_html__('Radius','sb') . '</option>
                <option value="10">10</option>
                <option value="25">25</option>
                <option value="50">50</option>
                <option value="100">100</option>
              </select> ';
        echo '<button type="submit">'.esc_html__('Search','sb').'</button>';
        echo '</form>';

        echo '<div class="sb-grid sb-cols-'.intval($atts['columns']).'">';
        if ($query->have_posts()) {
            while ($query->have_posts()) { $query->the_post();
                $img = get_the_post_thumbnail_url(get_the_ID(),'medium') ?: get_option('sb_default_image');
                echo '<article class="sb-card">';
                if ($img) echo '<a href="'.esc_url(get_permalink()).'"><img class="sb-card-img" src="'.esc_url($img).'" alt="'.esc_attr(get_the_title()).'"></a>';
                echo '<h3 class="sb-card-title"><a href="'.esc_url(get_permalink()).'">'.esc_html(get_the_title()).'</a></h3>';
                // show selected "card" fields
                foreach ($form_fields as $f) {
                    if (!empty($f['card']) && !empty($f['enabled'])) {
                        $val = get_post_meta(get_the_ID(), $f['key'], true);
                        if ($val) echo '<div class="sb-card-field sb-'.esc_attr($f['key']).'">'.esc_html($val).'</div>';
                    }
                }
                echo '</article>';
            }
            wp_reset_postdata();
        } else {
            echo '<p>'.esc_html__('No speakers found.','sb').'</p>';
        }
        echo '</div>';

        // Pagination
        $big = 999999999;
        $pagination = paginate_links([
            'base'      => str_replace($big, '%#%', esc_url(get_pagenum_link($big))),
            'format'    => '?paged=%#%',
            'current'   => max(1, $paged),
            'total'     => max(1, $query->max_num_pages),
            'type'      => 'list'
        ]);
        if ($pagination) echo $pagination;
    }
    return ob_get_clean();
}
add_shortcode('speaker_list', 'sb_render_speaker_list');

// speaker_register shortcode moved to sb-auth-bootstrap.php for consistency

// speaker_edit shortcode moved to sb-auth-bootstrap.php with authentication handling

/**
 * Shortcode: [speaker_topics_index]
 * Display index of all speaker topics with search links
 */
function sb_speaker_topics_index_shortcode($atts) {
    $atts = shortcode_atts([
        'columns' => 4,
        'show_counts' => 'yes',
        'sort' => 'alphabetical',
        'layout' => 'grid',
        'button_size' => 'medium',
        'button_spacing' => 8,
        'button_color' => '#0073aa',
        'button_hover_color' => '#005177',
        'button_text_color' => '#ffffff',
        'button_text_hover_color' => '#ffffff',
        'topic_limit' => 0,
        'grid_column_spacing' => 20,
        'grid_row_spacing' => 10,
        'grid_font_color' => '#0073aa',
        'grid_font_size' => 16,
        'topic_max_length' => 15
    ], $atts, 'speaker_topics_index');

    // Convert to block attributes format
    $block_attributes = [
        'columns' => max(1, min(6, (int)$atts['columns'])),
        'showCounts' => ($atts['show_counts'] === 'yes' || $atts['show_counts'] === 'true' || $atts['show_counts'] === '1'),
        'sortOrder' => in_array($atts['sort'], ['alphabetical', 'popular']) ? $atts['sort'] : 'alphabetical',
        'layout' => in_array($atts['layout'], ['grid', 'buttons']) ? $atts['layout'] : 'grid',
        'buttonSize' => in_array($atts['button_size'], ['small', 'medium', 'large']) ? $atts['button_size'] : 'medium',
        'buttonSpacing' => max(0, (int)$atts['button_spacing']),
        'buttonColor' => sanitize_hex_color($atts['button_color']) ?: '#0073aa',
        'buttonHoverColor' => sanitize_hex_color($atts['button_hover_color']) ?: '#005177',
        'buttonTextColor' => sanitize_hex_color($atts['button_text_color']) ?: '#ffffff',
        'buttonTextHoverColor' => sanitize_hex_color($atts['button_text_hover_color']) ?: '#ffffff',
        'topicLimit' => max(0, (int)$atts['topic_limit']),
        'gridColumnSpacing' => max(0, (int)$atts['grid_column_spacing']),
        'gridRowSpacing' => max(0, (int)$atts['grid_row_spacing']),
        'gridFontColor' => sanitize_hex_color($atts['grid_font_color']) ?: '#0073aa',
        'gridFontSize' => max(8, min(32, (int)$atts['grid_font_size'])),
        'topicMaxLength' => max(5, min(100, (int)$atts['topic_max_length']))
    ];

    // Use the same render function as the block
    return sb_render_topics_index_block($block_attributes);
}
add_shortcode('speaker_topics_index', 'sb_speaker_topics_index_shortcode');

/**
 * Shortcode: [logged_in_user]
 * Displays "Logged in as {username}" with logout link
 * Only displays if user is logged in
 */
function sb_logged_in_user_shortcode($atts) {
    // Return empty if user is not logged in
    if (!is_user_logged_in()) {
        return '';
    }

    // Get current user
    $current_user = wp_get_current_user();
    $logout_url = wp_logout_url(home_url());

    // Build output with transparent background and white text/icons
    $output = '<div class="sb-logged-in-user" style="display: inline-flex; align-items: center; gap: 8px; padding: 8px 12px; background: transparent; border-radius: 4px; font-size: 14px;">';

    // User icon (white)
    $output .= '<svg width="18" height="18" viewBox="0 0 24 24" fill="none" stroke="#ffffff" stroke-width="2" style="flex-shrink: 0;">';
    $output .= '<path d="M20 21v-2a4 4 0 0 0-4-4H8a4 4 0 0 0-4 4v2"></path>';
    $output .= '<circle cx="12" cy="7" r="4"></circle>';
    $output .= '</svg>';

    // Username (white, bold)
    $output .= '<span style="color: #ffffff; font-weight: 600;">' . esc_html($current_user->display_name) . '</span>';

    // Logout icon (white)
    $output .= '<a href="' . esc_url($logout_url) . '" style="display: inline-flex; align-items: center; text-decoration: none; color: #ffffff; font-weight: 500; margin-left: 4px;" title="' . esc_attr__('Log Out', 'speakers-bureau') . '">';
    $output .= '<svg width="18" height="18" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">';
    $output .= '<path d="M9 21H5a2 2 0 0 1-2-2V5a2 2 0 0 1 2-2h4"></path>';
    $output .= '<polyline points="16 17 21 12 16 7"></polyline>';
    $output .= '<line x1="21" y1="12" x2="9" y2="12"></line>';
    $output .= '</svg>';
    $output .= '</a>';
    $output .= '</div>';

    return $output;
}
add_shortcode('logged_in_user', 'sb_logged_in_user_shortcode');

/** END OF FILE: includes/sb-shortcodes.php */
