<?php
/**
 * File: includes/sb-registration-verification.php
 * Purpose: Two-step registration verification system for non-logged-in users
 * Features:
 * - Email verification with UUID tokens
 * - Temporary data storage during verification
 * - Password setup after email verification
 */

if (!defined('ABSPATH')) exit;

class SB_Registration_Verification {

    private $option_prefix = 'sb_pending_registration_';
    private $verification_page_slug = 'speaker-verify';

    public function __construct() {
        add_action('init', [$this, 'add_rewrite_rules']);
        add_action('wp', [$this, 'create_virtual_verification_page']);
        add_action('wp_enqueue_scripts', [$this, 'enqueue_verification_styles']);
        add_filter('query_vars', [$this, 'add_query_vars']);
    }

    /**
     * Create a pending registration entry with UUID
     */
    public function create_pending_registration($email, $name, $form_data = []) {
        $uuid = wp_generate_uuid4();
        $expiry = time() + (24 * 60 * 60); // 24 hours

        $pending_data = [
            'email' => sanitize_email($email),
            'name' => sanitize_text_field($name),
            'form_data' => $form_data,
            'uuid' => $uuid,
            'created' => time(),
            'expiry' => $expiry,
            'status' => 'pending_verification'
        ];

        // Store with UUID as key for easy lookup
        update_option($this->option_prefix . $uuid, $pending_data);

        // Also store a mapping by email for duplicate detection
        $email_key = 'sb_pending_email_' . md5($email);
        update_option($email_key, $uuid);

        return $uuid;
    }

    /**
     * Send verification email
     */
    public function send_verification_email($uuid, $email, $name) {
        $verification_url = home_url('/speaker-verify/?token=' . $uuid);

        $subject = sprintf(__('Verify your email for %s Speaker Registration', 'speakers-bureau'), get_bloginfo('name'));

        $body = sprintf(__("Hi %s,\n\nThank you for registering to be a speaker with %s.\n\nPlease click the link below to verify your email address and complete your registration:\n\n%s\n\nThis link will expire in 24 hours.\n\nIf you did not request this registration, you can safely ignore this email.\n\nBest regards,\n%s Team", 'speakers-bureau'),
            $name,
            get_bloginfo('name'),
            $verification_url,
            get_bloginfo('name')
        );

        return wp_mail($email, $subject, $body);
    }

    /**
     * Verify token and get pending registration data
     */
    public function get_pending_registration($uuid) {
        if (!$uuid) return false;

        $data = get_option($this->option_prefix . $uuid);
        if (!$data || !is_array($data)) return false;

        // Check if expired
        if (time() > $data['expiry']) {
            $this->cleanup_pending_registration($uuid);
            return false;
        }

        return $data;
    }

    /**
     * Complete registration after verification
     */
    public function complete_registration($uuid, $password) {
        $pending = $this->get_pending_registration($uuid);
        if (!$pending) {
            return new WP_Error('invalid_token', __('Invalid or expired verification token.', 'speakers-bureau'));
        }

        $email = $pending['email'];
        $name = $pending['name'];
        $form_data = $pending['form_data'];

        // Check if user already exists
        $existing_user = get_user_by('email', $email);
        if ($existing_user) {
            // User exists, just add speaker role if needed
            if (!in_array('speaker', (array)$existing_user->roles, true)) {
                $existing_user->add_role('speaker');
            }
            $user_id = $existing_user->ID;
        } else {
            // Create new user with provided password
            $username = sanitize_user(current(explode('@', $email)));
            if ($username === '') $username = 'speaker';

            $i = 1;
            $original_username = $username;
            while (username_exists($username)) {
                $username = $original_username . $i++;
            }

            $user_id = wp_create_user($username, $password, $email);
            if (is_wp_error($user_id)) {
                return $user_id;
            }

            $user = get_user_by('id', $user_id);
            $user->set_role('speaker');

            // Update display name
            wp_update_user([
                'ID' => $user_id,
                'display_name' => $name,
                'first_name' => $name
            ]);
        }

        // Check if auto-publish is enabled (but keep as draft for simplified registration)
        // Users will complete their profile after verification, then it may be auto-published
        $display_settings = get_option('sb_display_settings', []);
        $auto_publish = !empty($display_settings['auto_publish_profiles']);

        // For simplified registration, always start as draft since profile is incomplete
        // Auto-publish will be checked when they complete their full profile
        $status = 'draft';

        // Create speaker post with basic info only (name + email)
        $new_post_id = wp_insert_post([
            'post_type'   => 'speaker',
            'post_title'  => $name,
            'post_status' => $status,
            'post_author' => $user_id,
        ], true);

        if (is_wp_error($new_post_id)) {
            return $new_post_id;
        }

        // Save basic registration info as meta (name and email from simplified form)
        update_post_meta($new_post_id, 'speaker_name', $name);
        update_post_meta($new_post_id, 'email', $email);

        // Save any additional form data (usually empty for simplified registration)
        foreach ($form_data as $key => $value) {
            if ($key && $value !== '' && !in_array($key, ['speaker_name', 'email'])) {
                update_post_meta($new_post_id, $key, $value);
            }
        }

        // Automatically log in the user
        wp_set_current_user($user_id);
        wp_set_auth_cookie($user_id, true);
        do_action('wp_login', $username, get_user_by('id', $user_id));

        // Clean up pending registration
        $this->cleanup_pending_registration($uuid);

        return [
            'post_id' => $new_post_id,
            'user_id' => $user_id,
            'redirect_url' => home_url('/my-profile/')
        ];
    }

    /**
     * Check if email has pending registration
     */
    public function has_pending_registration($email) {
        $email_key = 'sb_pending_email_' . md5($email);
        $uuid = get_option($email_key);

        if (!$uuid) return false;

        $pending = $this->get_pending_registration($uuid);
        return $pending !== false;
    }

    /**
     * Clean up expired registrations
     */
    public function cleanup_pending_registration($uuid) {
        $data = get_option($this->option_prefix . $uuid);
        if ($data && isset($data['email'])) {
            $email_key = 'sb_pending_email_' . md5($data['email']);
            delete_option($email_key);
        }
        delete_option($this->option_prefix . $uuid);
    }

    /**
     * Add rewrite rules for verification page
     */
    public function add_rewrite_rules() {
        add_rewrite_rule('^speaker-verify/?$', 'index.php?speaker_verify=1', 'top');

        // Clean up expired registrations daily
        if (!wp_next_scheduled('sb_cleanup_expired_registrations')) {
            wp_schedule_event(time(), 'daily', 'sb_cleanup_expired_registrations');
        }
    }

    /**
     * Add custom query vars
     */
    public function add_query_vars($vars) {
        $vars[] = 'speaker_verify';
        return $vars;
    }

    /**
     * Create virtual verification page
     */
    public function create_virtual_verification_page() {
        global $wp_query;

        if (get_query_var('speaker_verify')) {
            $wp_query->is_404 = false;
            $wp_query->is_home = false;
            $wp_query->is_page = true;
            status_header(200);

            add_filter('template_include', [$this, 'verification_page_template']);
        }
    }

    /**
     * Verification page template
     */
    public function verification_page_template($template) {
        return plugin_dir_path(dirname(__FILE__)) . 'templates/speaker-verify.php';
    }

    /**
     * Enqueue styles for verification page
     */
    public function enqueue_verification_styles() {
        if (get_query_var('speaker_verify')) {
            wp_enqueue_style('sb-verification', plugin_dir_url(dirname(__FILE__)) . 'assets/sb-styles.css', [], SB_PLUGIN_VER);
        }
    }

    /**
     * Clean up expired registrations (scheduled task)
     */
    public static function cleanup_expired_registrations() {
        global $wpdb;

        $expired_time = time();
        $option_pattern = 'sb_pending_registration_%';

        $results = $wpdb->get_results($wpdb->prepare(
            "SELECT option_name, option_value FROM {$wpdb->options} WHERE option_name LIKE %s",
            $option_pattern
        ));

        foreach ($results as $row) {
            $data = maybe_unserialize($row->option_value);
            if (is_array($data) && isset($data['expiry']) && $data['expiry'] < $expired_time) {
                $uuid = str_replace('sb_pending_registration_', '', $row->option_name);
                $instance = new self();
                $instance->cleanup_pending_registration($uuid);
            }
        }
    }
}

// Initialize the verification system
new SB_Registration_Verification();

// Register cleanup hook
add_action('sb_cleanup_expired_registrations', ['SB_Registration_Verification', 'cleanup_expired_registrations']);

// Flush rewrite rules on plugin activation to register the new URL
register_activation_hook(SB_PLUGIN_FILE, function() {
    $verification = new SB_Registration_Verification();
    $verification->add_rewrite_rules();
    flush_rewrite_rules();
});

// Also flush on deactivation to clean up
register_deactivation_hook(SB_PLUGIN_FILE, function() {
    flush_rewrite_rules();
});

/** END OF FILE: includes/sb-registration-verification.php */