<?php
/**
 * File: includes/sb-meta-boxes.php
 * Function: Save logic for speaker meta (admin and front-end submit handlers)
 */
if (!defined('ABSPATH')) exit;

/**
 * Helper: get normalized field config
 */
function sb_get_fields_config() {
    $fields = get_option('sb_form_fields');
    if (is_string($fields)) {
        $dec = json_decode($fields, true);
        if (is_array($dec)) $fields = $dec;
    }
    if (!is_array($fields)) $fields = function_exists('sb_default_form_fields') ? sb_default_form_fields() : [];
    return $fields;
}

/**
 * Helper: sanitize a value by field type
 */
function sb_sanitize_field_value($type, $value, $field_config = []) {
    switch ($type) {
        case 'email':
            return sanitize_email($value);
        case 'url':
        case 'image':
            // Apply URL validation if configured for this field
            if ($type === 'url' && !empty($field_config['validate_url'])) {
                if (class_exists('SB_Form_Builder')) {
                    $validation = SB_Form_Builder::validate_url_security($value);
                    if (!$validation['valid']) {
                        // Store validation error for later display
                        add_filter('sb_form_errors', function($errors) use ($field_config, $validation) {
                            $field_name = isset($field_config['label']) ? $field_config['label'] : 'URL field';
                            $errors[] = sprintf(__('%s: %s', 'speakers-bureau'), $field_name, $validation['error']);
                            return $errors;
                        });
                        return ''; // Return empty on validation failure
                    }
                    return $validation['url']; // Return validated URL
                }
            }
            return esc_url_raw($value);
        case 'tel':
            return sanitize_text_field($value);
        case 'number':
            return is_numeric($value) ? $value : '';
        case 'checkbox':
            return !empty($value) ? '1' : '0';
        case 'textarea':
            // Preserve line breaks and allow basic HTML for paragraphs
            $value = sanitize_textarea_field($value);
            // Convert line breaks to proper paragraph breaks
            $value = str_replace("\r\n", "\n", $value); // Normalize line endings
            $value = str_replace("\r", "\n", $value);   // Normalize line endings
            return wp_kses_post($value);
        case 'select':
        case 'text':
        default:
            return sanitize_text_field($value);
    }
}

/**
 * ADMIN: Save on post save (classic/Block Editor)
 */
function sb_save_speaker_meta_admin($post_id, $post) {
    if ($post->post_type !== 'speaker') return;

    // Bail on autosave/revisions
    if (defined('DOING_AUTOSAVE') && DOING_AUTOSAVE) return;
    if (wp_is_post_revision($post_id)) return;

    if (!current_user_can('edit_post', $post_id)) return;

    $fields = sb_get_fields_config();
    if (!$fields) return;

    $featured_image_url = null;

    foreach ($fields as $field) {
        $key   = $field['key']  ?? '';
        $type  = $field['type'] ?? 'text';
        if (!$key || $type === 'section_header') continue;

        if (isset($_POST[$key])) {
            $raw  = $_POST[$key];
            $val  = sb_sanitize_field_value($type, $raw, $field);
            update_post_meta($post_id, $key, $val);

            // If this image field is flagged "featured", sync to Featured Image
            if ($type === 'image' && !empty($field['featured']) && !empty($val)) {
                $featured_image_url = $val;
            }
        } else {
            // For checkboxes, if not set, ensure we store 0
            if ($type === 'checkbox') {
                update_post_meta($post_id, $key, '0');
            }
        }
    }

    if ($featured_image_url) {
        $attachment_id = attachment_url_to_postid($featured_image_url);
        if ($attachment_id) {
            set_post_thumbnail($post_id, $attachment_id);
        }
    }

    // Ensure a linked user exists (compatible with older behavior)
    if (!get_post_meta($post_id, '_speaker_user_id', true)) {
        if ($post && $post->post_author) {
            update_post_meta($post_id, '_speaker_user_id', $post->post_author);
        } elseif (is_user_logged_in()) {
            update_post_meta($post_id, '_speaker_user_id', get_current_user_id());
        }
    }

    // Send profile update email to speaker if admin has not suppressed notifications
    // Check if this is an actual update to an existing published post
    $old_status = isset($_POST['original_post_status']) ? $_POST['original_post_status'] : '';
    $is_update = $old_status === 'publish' && $post->post_status === 'publish';

    if ($is_update && !wp_is_post_revision($post_id)) {
        $suppress_notification = get_post_meta($post_id, '_sb_suppress_update_notification', true);

        // Only send email if NOT suppressed (unchecked box means suppress_notification will be '0')
        if ($suppress_notification === '0') {
            $speaker_email = sb_get_speaker_email($post_id);
            $speaker_name = get_the_title($post_id);

            if ($speaker_email && function_exists('sb_send_profile_update_email')) {
                sb_send_profile_update_email($post_id, $speaker_email, $speaker_name);
            }
        }
    }
}
add_action('save_post', 'sb_save_speaker_meta_admin', 10, 2);

/**
 * FRONT-END: Register new speaker submission
 * - Expects nonce 'sb_register_nonce'
 */
function sb_handle_frontend_register_submit() {
    if (empty($_POST['sb_register_nonce']) || !wp_verify_nonce($_POST['sb_register_nonce'], 'sb_register_speaker')) {
        return;
    }
    if (!is_user_logged_in()) {
        wp_die(esc_html__('You must be logged in to register a speaker.', 'sb'));
    }

    $current_user = wp_get_current_user();

    // Create a new Speaker post (title falls back to user display name or "New Speaker")
    $post_title = sanitize_text_field($_POST['sb_company'] ?? $current_user->display_name ?? __('New Speaker','sb'));
    $post_id = wp_insert_post([
        'post_type'   => 'speaker',
        'post_status' => 'pending', // change to 'publish' if you want immediate publish
        'post_title'  => $post_title,
        'post_author' => get_current_user_id(),
    ]);

    if (is_wp_error($post_id) || !$post_id) {
        wp_die(esc_html__('Could not create speaker profile.', 'sb'));
    }

    $fields = sb_get_fields_config();
    $featured_image_url = null;

    foreach ($fields as $field) {
        $key  = $field['key'] ?? '';
        $type = $field['type'] ?? 'text';
        if (!$key || $type === 'section_header') continue;

        $raw = $_POST[$key] ?? null;
        if ($type === 'checkbox') {
            // checkbox not present means 0
            $raw = isset($_POST[$key]) ? $_POST[$key] : '';
        }

        if ($raw !== null) {
            $val = sb_sanitize_field_value($type, $raw, $field);
            update_post_meta($post_id, $key, $val);

            // Set featured image for profile_image or any image marked as featured
            if ($type === 'image' && !empty($val)) {
                if ($key === 'profile_image' || !empty($field['featured'])) {
                    $featured_image_url = $val;
                }
            }
        }
    }

    if ($featured_image_url) {
        $attachment_id = attachment_url_to_postid($featured_image_url);
        if ($attachment_id) {
            set_post_thumbnail($post_id, $attachment_id);
        }
    }

    update_post_meta($post_id, '_speaker_user_id', get_current_user_id());

    // Redirect to the new single speaker page
    wp_safe_redirect(get_permalink($post_id));
    exit;
}

/**
 * FRONT-END: Edit existing speaker submission
 * - Expects nonce 'sb_edit_nonce'
 * - Expects ?speaker_id={ID} in the URL
 */
function sb_handle_frontend_edit_submit() {
    if (empty($_POST['sb_edit_nonce']) || !wp_verify_nonce($_POST['sb_edit_nonce'], 'sb_edit_speaker')) {
        return;
    }

    $speaker_id = isset($_GET['speaker_id']) ? intval($_GET['speaker_id']) : 0;
    if (!$speaker_id || get_post_type($speaker_id) !== 'speaker') {
        wp_die(esc_html__('Invalid speaker profile.', 'sb'));
    }
    if (!current_user_can('edit_post', $speaker_id)) {
        wp_die(esc_html__('You do not have permission to edit this speaker.', 'sb'));
    }

    $fields = sb_get_fields_config();
    $featured_image_url = null;

    foreach ($fields as $field) {
        $key  = $field['key'] ?? '';
        $type = $field['type'] ?? 'text';
        if (!$key || $type === 'section_header') continue;

        $raw = $_POST[$key] ?? null;
        if ($type === 'checkbox') {
            $raw = isset($_POST[$key]) ? $_POST[$key] : '';
        }

        if ($raw !== null) {
            $val = sb_sanitize_field_value($type, $raw, $field);
            update_post_meta($speaker_id, $key, $val);

            // Set featured image for profile_image or any image marked as featured
            if ($type === 'image' && !empty($val)) {
                if ($key === 'profile_image' || !empty($field['featured'])) {
                    $featured_image_url = $val;
                }
            }
        }
    }

    if ($featured_image_url) {
        $attachment_id = attachment_url_to_postid($featured_image_url);
        if ($attachment_id) {
            set_post_thumbnail($speaker_id, $attachment_id);
        }
    }

    // Keep linked user in sync
    if (is_user_logged_in()) {
        update_post_meta($speaker_id, '_speaker_user_id', get_current_user_id());
    }

    // Redirect back to the single speaker page
    wp_safe_redirect(get_permalink($speaker_id));
    exit;
}

/**
 * Front-end submit router
 */
function sb_frontend_submit_router() {
    if (is_admin()) return;

    // Registration submit
    if (!empty($_POST['sb_register_nonce'])) {
        sb_handle_frontend_register_submit();
        return;
    }
    // Edit submit
    if (!empty($_POST['sb_edit_nonce'])) {
        sb_handle_frontend_edit_submit();
        return;
    }
}
add_action('template_redirect', 'sb_frontend_submit_router');

/**
 * Ensure WP Media Uploader is available on the front-end for image fields
 * (so the "Upload Image" buttons work on /register-speaker/ and /my-profile/)
 */
function sb_ensure_frontend_media_uploader() {
    if (!is_admin()) {
        wp_enqueue_media();
    }
}
add_action('wp_enqueue_scripts', 'sb_ensure_frontend_media_uploader');

/**
 * Add meta boxes for speaker user assignment in admin
 */
function sb_add_speaker_meta_boxes() {
    add_meta_box(
        'sb_speaker_user_assignment',
        __('Speaker User Assignment', 'speakers-bureau'),
        'sb_render_speaker_user_assignment_meta_box',
        'speaker',
        'side',
        'high'
    );

    add_meta_box(
        'sb_speaker_email_notifications',
        __('Email Notifications', 'speakers-bureau'),
        'sb_render_speaker_email_notifications_meta_box',
        'speaker',
        'side',
        'default'
    );
}
add_action('add_meta_boxes', 'sb_add_speaker_meta_boxes');

/**
 * Render the user assignment meta box
 */
function sb_render_speaker_user_assignment_meta_box($post) {
    wp_nonce_field('sb_speaker_user_assignment', 'sb_speaker_user_assignment_nonce');

    // Get current assigned user
    $assigned_user_id = get_post_meta($post->ID, '_sb_assigned_user_id', true);
    $assigned_user = $assigned_user_id ? get_user_by('id', $assigned_user_id) : null;

    // Get speaker email from form fields
    $speaker_email = '';
    $fields = sb_get_fields_config();
    foreach ($fields as $field) {
        if (($field['type'] ?? '') === 'email' && !empty($field['enabled'])) {
            $speaker_email = get_post_meta($post->ID, $field['key'] ?? 'email', true);
            break;
        }
    }

    echo '<div style="margin-bottom: 15px;">';
    echo '<p><strong>' . esc_html__('Current Assignment:', 'speakers-bureau') . '</strong></p>';

    if ($assigned_user) {
        echo '<p>👤 ' . esc_html($assigned_user->display_name) . ' (' . esc_html($assigned_user->user_email) . ')</p>';
        echo '<p><a href="' . esc_url(admin_url('user-edit.php?user_id=' . $assigned_user->ID)) . '" target="_blank">' . esc_html__('Edit User', 'speakers-bureau') . '</a></p>';
    } else {
        echo '<p><em>' . esc_html__('No user assigned', 'speakers-bureau') . '</em></p>';
    }
    echo '</div>';

    echo '<div style="margin-bottom: 15px;">';
    echo '<label for="sb_assign_existing_user"><strong>' . esc_html__('Assign to Existing User:', 'speakers-bureau') . '</strong></label><br>';

    // User dropdown
    $users = get_users(['orderby' => 'display_name', 'order' => 'ASC']);
    echo '<select id="sb_assign_existing_user" name="sb_assign_existing_user" style="width: 100%; margin-top: 5px;">';
    echo '<option value="">' . esc_html__('— Select User —', 'speakers-bureau') . '</option>';
    foreach ($users as $user) {
        $selected = selected($assigned_user_id, $user->ID, false);
        echo '<option value="' . esc_attr($user->ID) . '"' . $selected . '>';
        echo esc_html($user->display_name . ' (' . $user->user_email . ')');
        echo '</option>';
    }
    echo '</select>';
    echo '</div>';

    if ($speaker_email) {
        echo '<div style="margin-bottom: 15px; padding: 10px; background: #f0f8ff; border: 1px solid #0073aa; border-radius: 4px;">';
        echo '<p><strong>' . esc_html__('Auto-Create User Option:', 'speakers-bureau') . '</strong></p>';
        echo '<p>' . esc_html__('Speaker Email:', 'speakers-bureau') . ' <code>' . esc_html($speaker_email) . '</code></p>';

        // Check if user already exists with this email
        $existing_user = get_user_by('email', $speaker_email);
        if ($existing_user) {
            if ($existing_user->ID != $assigned_user_id) {
                echo '<p style="color: #d54e21;">⚠️ ' . sprintf(
                    esc_html__('User with email %s already exists: %s', 'speakers-bureau'),
                    '<code>' . esc_html($speaker_email) . '</code>',
                    '<strong>' . esc_html($existing_user->display_name) . '</strong>'
                ) . '</p>';
                echo '<label><input type="checkbox" name="sb_assign_to_existing_email_user" value="1"> ';
                echo esc_html__('Assign to this existing user', 'speakers-bureau') . '</label>';
            } else {
                echo '<p style="color: #46b450;">✅ ' . esc_html__('Already assigned to user with this email', 'speakers-bureau') . '</p>';
            }
        } else {
            echo '<label><input type="checkbox" name="sb_create_user_from_email" value="1"> ';
            echo esc_html__('Create new user account with this email', 'speakers-bureau') . '</label>';
        }
        echo '</div>';
    } else {
        echo '<div style="margin-bottom: 15px; padding: 10px; background: #fff3cd; border: 1px solid #ffc107; border-radius: 4px;">';
        echo '<p>📧 ' . esc_html__('No email address found in speaker profile. Add an email to enable auto-user creation.', 'speakers-bureau') . '</p>';
        echo '</div>';
    }

    echo '<div style="margin-top: 15px; font-size: 12px; color: #666;">';
    echo '<p><strong>' . esc_html__('How this works:', 'speakers-bureau') . '</strong></p>';
    echo '<ul style="margin-left: 15px;">';
    echo '<li>' . esc_html__('Assign to existing user: Speaker can log in with that user account', 'speakers-bureau') . '</li>';
    echo '<li>' . esc_html__('Create new user: Creates account with speaker role and sends password reset email', 'speakers-bureau') . '</li>';
    echo '<li>' . esc_html__('Assigned users can edit their speaker profile on the frontend', 'speakers-bureau') . '</li>';
    echo '</ul>';
    echo '</div>';
}

/**
 * Render the email notifications meta box
 */
function sb_render_speaker_email_notifications_meta_box($post) {
    wp_nonce_field('sb_speaker_email_notifications', 'sb_speaker_email_notifications_nonce');

    // Get current setting (checked by default means NO email will be sent)
    $suppress_notification = get_post_meta($post->ID, '_sb_suppress_update_notification', true);
    // Default to checked (suppress notification) if not set
    if ($suppress_notification === '') {
        $suppress_notification = '1';
    }

    echo '<div style="margin: 10px 0;">';
    echo '<label style="display: flex; align-items: flex-start; cursor: pointer;">';
    echo '<input type="checkbox" name="sb_suppress_update_notification" value="1" ' . checked($suppress_notification, '1', false) . ' style="margin-right: 8px; margin-top: 2px; flex-shrink: 0;">';
    echo '<span>';
    echo '<strong>' . esc_html__('Do NOT send notification when updating this record', 'speakers-bureau') . '</strong><br>';
    echo '<span style="font-size: 12px; color: #666;">' . esc_html__('When checked, email notifications to the speaker will be suppressed even if they have requested updates.', 'speakers-bureau') . '</span>';
    echo '</span>';
    echo '</label>';
    echo '</div>';

    // Show speaker's current email notification preference
    $speaker_email = sb_get_speaker_email($post->ID);
    if ($speaker_email) {
        echo '<div style="margin-top: 15px; padding: 10px; background: #f0f0f1; border-radius: 4px;">';
        echo '<p style="margin: 0; font-size: 13px; color: #666;">';
        echo '<strong>' . esc_html__('Speaker Email:', 'speakers-bureau') . '</strong><br>';
        echo esc_html($speaker_email);
        echo '</p>';
        echo '</div>';
    }
}

/**
 * Save user assignment when speaker post is saved
 */
function sb_save_speaker_user_assignment($post_id, $post) {
    // Only process speaker posts
    if ($post->post_type !== 'speaker') {
        return;
    }

    // Verify nonce
    if (empty($_POST['sb_speaker_user_assignment_nonce']) ||
        !wp_verify_nonce($_POST['sb_speaker_user_assignment_nonce'], 'sb_speaker_user_assignment')) {
        return;
    }

    // Check permissions
    if (!current_user_can('edit_post', $post_id)) {
        return;
    }

    $assigned_user_id = 0;
    $notifications = [];

    // Handle assign to existing user
    if (!empty($_POST['sb_assign_existing_user'])) {
        $user_id = (int) $_POST['sb_assign_existing_user'];
        $user = get_user_by('id', $user_id);

        if ($user) {
            $assigned_user_id = $user_id;

            // Add speaker role if not already present
            if (!in_array('speaker', (array)$user->roles, true)) {
                $user->add_role('speaker');
                $notifications[] = sprintf(__('Added speaker role to user %s', 'speakers-bureau'), $user->display_name);
            }

            $notifications[] = sprintf(__('Assigned speaker to user %s', 'speakers-bureau'), $user->display_name);
        }
    }
    // Handle create new user from email
    elseif (!empty($_POST['sb_create_user_from_email'])) {
        $speaker_email = sb_get_speaker_email($post_id);

        if ($speaker_email && is_email($speaker_email)) {
            $existing_user = get_user_by('email', $speaker_email);

            if (!$existing_user) {
                // Create new user
                $speaker_name = get_the_title($post_id);
                $username = sanitize_user($speaker_email);

                // Ensure unique username
                $base_username = $username;
                $counter = 1;
                while (username_exists($username)) {
                    $username = $base_username . $counter;
                    $counter++;
                }

                $user_data = [
                    'user_login'   => $username,
                    'user_email'   => $speaker_email,
                    'display_name' => $speaker_name,
                    'role'         => 'speaker',
                ];

                $user_id = wp_insert_user($user_data);

                if (!is_wp_error($user_id)) {
                    $assigned_user_id = $user_id;
                    $notifications[] = sprintf(__('Created new user account for %s', 'speakers-bureau'), $speaker_email);

                    // Send password reset email
                    $reset_key = get_password_reset_key(get_user_by('id', $user_id));
                    if (!is_wp_error($reset_key)) {
                        $reset_url = network_site_url("wp-login.php?action=rp&key=$reset_key&login=" . rawurlencode($username), 'login');

                        // Send welcome email with password reset link
                        $subject = sprintf(__('Your speaker account on %s', 'speakers-bureau'), get_bloginfo('name'));
                        $message = sprintf(
                            __("Hello %s,\n\nA speaker account has been created for you on %s.\n\nTo set your password and access your profile, please visit:\n%s\n\nOnce logged in, you can edit your speaker profile and manage your information.\n\nBest regards,\n%s", 'speakers-bureau'),
                            $speaker_name,
                            get_bloginfo('name'),
                            $reset_url,
                            get_bloginfo('name')
                        );

                        wp_mail($speaker_email, $subject, $message);
                        $notifications[] = __('Sent welcome email with password setup instructions', 'speakers-bureau');
                    }
                } else {
                    $notifications[] = sprintf(__('Error creating user: %s', 'speakers-bureau'), $user_id->get_error_message());
                }
            }
        }
    }
    // Handle assign to existing user with same email
    elseif (!empty($_POST['sb_assign_to_existing_email_user'])) {
        $speaker_email = sb_get_speaker_email($post_id);

        if ($speaker_email) {
            $existing_user = get_user_by('email', $speaker_email);

            if ($existing_user) {
                $assigned_user_id = $existing_user->ID;

                // Add speaker role if not already present
                if (!in_array('speaker', (array)$existing_user->roles, true)) {
                    $existing_user->add_role('speaker');
                    $notifications[] = sprintf(__('Added speaker role to existing user %s', 'speakers-bureau'), $existing_user->display_name);
                }

                $notifications[] = sprintf(__('Assigned speaker to existing user %s', 'speakers-bureau'), $existing_user->display_name);
            }
        }
    }

    // Save the assignment
    if ($assigned_user_id) {
        update_post_meta($post_id, '_sb_assigned_user_id', $assigned_user_id);

        // Also update the post author to match the assigned user for consistency
        wp_update_post([
            'ID' => $post_id,
            'post_author' => $assigned_user_id
        ]);
    } else {
        delete_post_meta($post_id, '_sb_assigned_user_id');
    }

    // Store notifications for display
    if (!empty($notifications)) {
        set_transient('sb_user_assignment_notices_' . get_current_user_id(), $notifications, 30);
    }
}
add_action('save_post', 'sb_save_speaker_user_assignment', 20, 2);

/**
 * Save email notification settings when speaker post is saved
 */
function sb_save_speaker_email_notifications($post_id, $post) {
    // Only process speaker posts
    if ($post->post_type !== 'speaker') {
        return;
    }

    // Verify nonce
    if (empty($_POST['sb_speaker_email_notifications_nonce']) ||
        !wp_verify_nonce($_POST['sb_speaker_email_notifications_nonce'], 'sb_speaker_email_notifications')) {
        return;
    }

    // Check permissions
    if (!current_user_can('edit_post', $post_id)) {
        return;
    }

    // Save the suppress notification setting
    if (isset($_POST['sb_suppress_update_notification'])) {
        update_post_meta($post_id, '_sb_suppress_update_notification', '1');
    } else {
        update_post_meta($post_id, '_sb_suppress_update_notification', '0');
    }
}
add_action('save_post', 'sb_save_speaker_email_notifications', 20, 2);

/**
 * Helper function to get speaker email from form fields
 */
function sb_get_speaker_email($post_id) {
    $fields = sb_get_fields_config();
    foreach ($fields as $field) {
        if (($field['type'] ?? '') === 'email' && !empty($field['enabled'])) {
            $email = get_post_meta($post_id, $field['key'] ?? 'email', true);
            if ($email && is_email($email)) {
                return $email;
            }
        }
    }
    return '';
}

/**
 * Display user assignment notices in admin
 */
function sb_display_user_assignment_notices() {
    $notices = get_transient('sb_user_assignment_notices_' . get_current_user_id());
    if ($notices) {
        delete_transient('sb_user_assignment_notices_' . get_current_user_id());

        echo '<div class="notice notice-success is-dismissible">';
        echo '<p><strong>' . esc_html__('Speaker User Assignment:', 'speakers-bureau') . '</strong></p>';
        echo '<ul>';
        foreach ($notices as $notice) {
            echo '<li>' . esc_html($notice) . '</li>';
        }
        echo '</ul>';
        echo '</div>';
    }
}
add_action('admin_notices', 'sb_display_user_assignment_notices');

/* End of file: includes/sb-meta-boxes.php */

/** END OF FILE: includes/sb-meta-boxes.php */
