<?php
/**
 * File: includes/sb-geocoding.php
 * Purpose: Part of the Speakers Bureau plugin.
 */

if (!defined('ABSPATH')) exit;

/**
 * Convert country name to ISO 2-letter code for Nominatim API
 */
function sb_get_country_code($country) {
    // Map common country names/codes to ISO 2-letter codes
    $country_map = [
        'USA' => 'US',
        'UNITED STATES' => 'US',
        'UNITED STATES OF AMERICA' => 'US',
        'CANADA' => 'CA',
        'MEXICO' => 'MX',
        'UK' => 'GB',
        'UNITED KINGDOM' => 'GB',
        'GREAT BRITAIN' => 'GB',
        'AUSTRALIA' => 'AU',
        'NEW ZEALAND' => 'NZ',
    ];

    $country_upper = strtoupper(trim($country));

    // If it's already a 2-letter code, return it
    if (strlen($country_upper) == 2) {
        return $country_upper;
    }

    // Check if we have a mapping
    if (isset($country_map[$country_upper])) {
        return $country_map[$country_upper];
    }

    // Default to US if we can't determine
    return 'US';
}

function sb_geocode_address($address_components, $country = '') {
    // Support both old format (string) and new format (array) for backward compatibility
    if (is_string($address_components)) {
        // Old format - convert to single query
        $query_params = [
            'q' => $address_components,
            'format' => 'json',
            'limit' => 1
        ];
        if (!empty($country)) {
            $query_params['countrycodes'] = sb_get_country_code($country);
        }
        $api_url = "https://nominatim.openstreetmap.org/search?" . http_build_query($query_params);

        $response = wp_remote_get($api_url, ['headers'=>['User-Agent'=>'Speakers Bureau Plugin']]);
        if (is_wp_error($response)) return false;
        $data = json_decode(wp_remote_retrieve_body($response), true);
        if (!empty($data[0])) {
            return ['lat'=>$data[0]['lat'], 'lng'=>$data[0]['lon']];
        }
        return false;
    }

    // New format - use fallback strategy for best precision
    // Strategy: Try zip-only first (best precision), then fall back to broader queries

    $has_city = !empty($address_components['city']);
    $has_zip = !empty($address_components['postalcode']);
    $has_state = !empty($address_components['state']);
    $has_street = !empty($address_components['street']);
    $country_value = !empty($address_components['country']) ? $address_components['country'] : $country;

    // Attempt 1: Zip code only (provides best precision for distinct zip codes)
    // Note: Nominatim works best with ONLY postalcode + countrycodes (no state/country)
    if ($has_zip) {
        $query_params = [
            'format' => 'json',
            'limit' => 1,
            'addressdetails' => 1,
            'postalcode' => $address_components['postalcode']
        ];

        // Only add country code (not state or country name) for zip-only query
        if (!empty($country_value)) {
            $query_params['countrycodes'] = sb_get_country_code($country_value);
        }

        $api_url = "https://nominatim.openstreetmap.org/search?" . http_build_query($query_params);
        $response = wp_remote_get($api_url, ['headers'=>['User-Agent'=>'Speakers Bureau Plugin']]);

        if (!is_wp_error($response)) {
            $data = json_decode(wp_remote_retrieve_body($response), true);
            if (!empty($data[0])) {
                return ['lat'=>$data[0]['lat'], 'lng'=>$data[0]['lon']];
            }
        }

        // Small delay before fallback attempt
        usleep(100000); // 0.1 second
    }

    // Attempt 2: Full structured query with all components
    if ($has_street || $has_city) {
        $query_params = [
            'format' => 'json',
            'limit' => 1,
            'addressdetails' => 1
        ];

        if ($has_street) {
            $query_params['street'] = $address_components['street'];
        }
        if ($has_city) {
            $query_params['city'] = $address_components['city'];
        }
        if ($has_state) {
            $query_params['state'] = $address_components['state'];
        }
        if ($has_zip) {
            $query_params['postalcode'] = $address_components['postalcode'];
        }
        if (!empty($country_value)) {
            $query_params['country'] = $country_value;
            $query_params['countrycodes'] = sb_get_country_code($country_value);
        }

        $api_url = "https://nominatim.openstreetmap.org/search?" . http_build_query($query_params);
        $response = wp_remote_get($api_url, ['headers'=>['User-Agent'=>'Speakers Bureau Plugin']]);

        if (!is_wp_error($response)) {
            $data = json_decode(wp_remote_retrieve_body($response), true);
            if (!empty($data[0])) {
                return ['lat'=>$data[0]['lat'], 'lng'=>$data[0]['lon']];
            }
        }

        usleep(100000); // 0.1 second
    }

    // Attempt 3: Free-form query as last resort
    $parts = array_filter([
        $address_components['street'] ?? '',
        $address_components['city'] ?? '',
        $address_components['state'] ?? '',
        $address_components['postalcode'] ?? '',
        $country_value
    ]);

    if (!empty($parts)) {
        $query_params = [
            'q' => implode(', ', $parts),
            'format' => 'json',
            'limit' => 1
        ];
        if (!empty($country_value)) {
            $query_params['countrycodes'] = sb_get_country_code($country_value);
        }

        $api_url = "https://nominatim.openstreetmap.org/search?" . http_build_query($query_params);
        $response = wp_remote_get($api_url, ['headers'=>['User-Agent'=>'Speakers Bureau Plugin']]);

        if (!is_wp_error($response)) {
            $data = json_decode(wp_remote_retrieve_body($response), true);
            if (!empty($data[0])) {
                return ['lat'=>$data[0]['lat'], 'lng'=>$data[0]['lon']];
            }
        }
    }

    return false;
}

/**
 * Main geocoding function for a speaker
 * Separated so it can be called from multiple hooks
 */
function sb_geocode_speaker($post_id) {
    // Skip if autosave
    if (defined('DOING_AUTOSAVE') && DOING_AUTOSAVE) return;

    // Verify it's a speaker post
    if (get_post_type($post_id) !== 'speaker') return;

    // Get address components - check multiple possible field names
    // Street address (optional, may not exist)
    $address = get_post_meta($post_id, 'address', true);
    if (empty($address)) $address = get_post_meta($post_id, 'sb_address', true);

    // City - check multiple possible field names
    $city = get_post_meta($post_id, 'city', true);
    if (empty($city)) $city = get_post_meta($post_id, 'sb_city', true);

    // State - check multiple possible field names
    $state = get_post_meta($post_id, 'state', true);
    if (empty($state)) $state = get_post_meta($post_id, 'sb_state', true);

    // Zip - check multiple possible field names
    $zip = get_post_meta($post_id, 'zip', true);
    if (empty($zip)) $zip = get_post_meta($post_id, 'zip_code', true);
    if (empty($zip)) $zip = get_post_meta($post_id, 'sb_zip', true);
    if (empty($zip)) $zip = get_post_meta($post_id, 'postal_code', true);

    // Try to get country from various possible field names
    $country = get_post_meta($post_id, 'country', true);
    if (empty($country)) $country = get_post_meta($post_id, 'sb_country', true);

    // If no country specified, use default from settings
    if (empty($country)) {
        $settings = get_option('sb_settings', []);
        $country = isset($settings['geo_default_country']) ? $settings['geo_default_country'] : 'US';
    }

    // Check if we have any address data
    if (empty($address) && empty($city) && empty($state) && empty($zip)) {
        return;
    }

    // Build structured address components for better geocoding precision
    $address_components = [
        'street' => $address,
        'city' => $city,
        'state' => $state,
        'postalcode' => $zip,
        'country' => $country
    ];

    // Geocode with structured query for better accuracy (especially for zip codes)
    $coords = sb_geocode_address($address_components, $country);
    if ($coords) {
        update_post_meta($post_id, 'geo_lat', $coords['lat']);
        update_post_meta($post_id, 'geo_lng', $coords['lng']);
        // Also update old format for backward compatibility
        update_post_meta($post_id, '_sb_lat', $coords['lat']);
        update_post_meta($post_id, '_sb_lng', $coords['lng']);
    }
}

// Geocode on save - use priority 50 to run AFTER meta boxes save their data (which run at priority 10)
add_action('save_post_speaker', 'sb_geocode_speaker', 50);

// Also geocode when address fields are updated directly
add_action('updated_post_meta', function($meta_id, $post_id, $meta_key, $meta_value) {
    // Check if this is an address-related field
    $address_fields = ['city', 'sb_city', 'state', 'sb_state', 'zip', 'zip_code', 'sb_zip', 'postal_code', 'address', 'sb_address', 'country', 'sb_country'];

    if (in_array($meta_key, $address_fields)) {
        // Make sure it's a speaker post
        if (get_post_type($post_id) === 'speaker') {
            // Prevent infinite loop by checking if we're not already updating geo coordinates
            if (!in_array($meta_key, ['geo_lat', 'geo_lng', '_sb_lat', '_sb_lng'])) {
                sb_geocode_speaker($post_id);
            }
        }
    }
}, 10, 4);

/** END OF FILE: includes/sb-geocoding.php */
