<?php
/**
 * File: includes/sb-form-builder.php
 * Purpose: Speakers Bureau — Form Builder
 * Sortable table + add/remove/reorder rows.
 * Each row: label, key, type, placeholder/options, section, enabled, required, hide label, searchable
 */
if (!defined('ABSPATH')) exit;

class SB_Form_Builder {

    private $option_name = 'sb_form_fields';


    private function default_row() {
        return [
            'label'       => '',
            'key'         => '',
            'type'        => 'text',
            'placeholder' => '',
            'enabled'     => 1,
            'required'    => 0,
            'hide_label'  => 0,
            'section'     => 2,
            'searchable'  => 1,
            'validate_url' => 0,
            'fa_icon'     => '',
        ];
    }

    private function get_fields() {
        $rows = get_option($this->option_name, []);
        if (is_string($rows)) {
            $d = json_decode($rows, true);
            if (is_array($d)) $rows = $d;
        }
        if (!is_array($rows)) $rows = [];

        $out = [];
        foreach ($rows as $row) {
            $row = wp_parse_args($row, $this->default_row());

            // Normalize field types on load to handle legacy data
            $row['type'] = $this->normalize_field_type($row['type']);

            $out[] = $row;
        }
        return $out;
    }

    private function normalize_field_type($type) {
        $valid_types = ['text', 'email', 'tel', 'url', 'textarea', 'select', 'checkbox', 'image', 'edit_link'];

        // Ensure type is a string to prevent str_starts_with warnings
        $type = (string)$type;
        if (empty($type)) {
            return 'text';
        }

        // Map legacy/alternative type names to valid types
        $type_mapping = [
            'phone' => 'tel',
            'telephone' => 'tel',
            'mobile' => 'tel',
            'cell' => 'tel',
            'cellphone' => 'tel',
            'cell_phone' => 'tel',
            'website' => 'url',
            'link' => 'url'
        ];

        $type_lower = strtolower($type);

        // Normalize the type
        if (isset($type_mapping[$type_lower])) {
            return $type_mapping[$type_lower];
        } elseif (in_array($type, $valid_types)) {
            return $type;
        } else {
            return 'text'; // fallback for invalid types
        }
    }

    /**
     * Validate URL for security - checks for malicious patterns
     */
    public static function validate_url_security($url) {
        if (empty($url)) {
            return ['valid' => true, 'url' => ''];
        }

        // Basic URL validation
        $url = trim($url);

        // Add protocol if missing
        if (!preg_match('/^https?:\/\//', $url)) {
            $url = 'https://' . $url;
        }

        // Validate URL format
        if (!filter_var($url, FILTER_VALIDATE_URL)) {
            return ['valid' => false, 'error' => 'Invalid URL format', 'url' => $url];
        }

        // Security checks - block dangerous protocols and patterns
        $parsed = parse_url($url);

        if (!$parsed || !isset($parsed['scheme']) || !isset($parsed['host'])) {
            return ['valid' => false, 'error' => 'Invalid URL structure', 'url' => $url];
        }

        // Only allow HTTP/HTTPS
        if (!in_array(strtolower($parsed['scheme']), ['http', 'https'])) {
            return ['valid' => false, 'error' => 'Only HTTP and HTTPS URLs are allowed', 'url' => $url];
        }

        // Block localhost and internal IPs for security
        $host = strtolower($parsed['host']);
        if ($host === 'localhost' ||
            preg_match('/^127\./', $host) ||
            preg_match('/^192\.168\./', $host) ||
            preg_match('/^10\./', $host) ||
            preg_match('/^172\.(1[6-9]|2[0-9]|3[01])\./', $host)) {
            return ['valid' => false, 'error' => 'Local/internal URLs are not allowed', 'url' => $url];
        }

        // Block suspicious patterns
        if (preg_match('/[<>"\']/', $url)) {
            return ['valid' => false, 'error' => 'URL contains invalid characters', 'url' => $url];
        }

        return ['valid' => true, 'url' => esc_url($url)];
    }

    private function save_fields() {
        if (empty($_POST['sb_fb_nonce']) || !wp_verify_nonce($_POST['sb_fb_nonce'], 'sb_fb_save')) {
            return;
        }

        $rows = [];
        $labels       = isset($_POST['fb_label'])       ? (array) $_POST['fb_label']       : [];
        $keys         = isset($_POST['fb_key'])         ? (array) $_POST['fb_key']         : [];
        $types        = isset($_POST['fb_type'])        ? (array) $_POST['fb_type']        : [];
        $placeholders = isset($_POST['fb_placeholder']) ? (array) $_POST['fb_placeholder'] : [];
        $sections     = isset($_POST['fb_section'])     ? (array) $_POST['fb_section']     : [];

        $enabled      = isset($_POST['fb_enabled'])     ? (array) $_POST['fb_enabled']     : [];
        $required     = isset($_POST['fb_required'])    ? (array) $_POST['fb_required']    : [];
        $hide_label   = isset($_POST['fb_hide_label'])  ? (array) $_POST['fb_hide_label']  : [];
        $searchable   = isset($_POST['fb_searchable'])  ? (array) $_POST['fb_searchable']  : [];
        $validate_url = isset($_POST['fb_validate_url']) ? (array) $_POST['fb_validate_url'] : [];
        $fa_icons     = isset($_POST['fb_fa_icon'])     ? (array) $_POST['fb_fa_icon']     : [];

        $count = max(count($labels), count($keys), count($types), count($placeholders), count($sections), count($fa_icons));


        for ($i = 0; $i < $count; $i++) {
            $row = [
                'label'       => sanitize_text_field($labels[$i] ?? ''),
                'key'         => sanitize_key($keys[$i] ?? ''),
                'type'        => $this->normalize_field_type(sanitize_text_field($types[$i] ?? 'text')),
                'placeholder' => sanitize_text_field($placeholders[$i] ?? ''),
                'enabled'     => in_array($i, array_keys($enabled)) ? 1 : 0,
                'required'    => in_array($i, array_keys($required)) ? 1 : 0,
                'hide_label'  => in_array($i, array_keys($hide_label)) ? 1 : 0,
                'section'     => (int) ($sections[$i] ?? 2),
                'searchable'  => in_array($i, array_keys($searchable)) ? 1 : 0,
                'validate_url' => in_array($i, array_keys($validate_url)) ? 1 : 0,
                'fa_icon'     => sanitize_text_field($fa_icons[$i] ?? ''),
            ];
            if ($row['label'] === '' && $row['key'] === '') continue;

            // Don't use wp_parse_args to avoid overwriting valid field types
            $default = $this->default_row();
            foreach ($default as $key => $default_value) {
                if (!isset($row[$key]) || $row[$key] === '') {
                    // Only use default if the field is truly missing or empty string
                    // Don't override valid values (especially 'type' field)
                    if ($key !== 'type' || $row[$key] === '') {
                        $row[$key] = $default_value;
                    }
                }
            }

            $rows[] = $row;
        }


        update_option($this->option_name, $rows);
    }


    /** Build section options from Profile Layout; exclude blank titles */
    private function section_options() {
        $defaults = function_exists('sb_default_profile_layout') ? sb_default_profile_layout() : [];
        $layout   = get_option('sb_profile_layout', $defaults);
        $opts     = [];
        foreach (range(1,8) as $i) {
            $row = wp_parse_args($layout[$i] ?? [], $defaults[$i] ?? []);
            $title = isset($row['title']) ? trim($row['title']) : '';
            if ($title === '') continue;
            $opts[$i] = $i . ' — ' . ($title ?: ('Section '.$i));
        }
        if (empty($opts)) { foreach (range(1,8) as $i) $opts[$i] = (string)$i; }
        return $opts;
    }

    public function render_form_builder() {
        // Handle form submission directly
        if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['sb_fb_nonce']) && wp_verify_nonce($_POST['sb_fb_nonce'], 'sb_fb_save')) {
            $this->save_fields();
            echo '<div class="updated notice is-dismissible"><p>'.esc_html__('Form builder settings saved successfully.', 'speakers-bureau').'</p></div>';
        }

        $rows     = $this->get_fields();
        $sections = $this->section_options();
        ?>
        <form method="post" action="">
            <?php wp_nonce_field('sb_fb_save', 'sb_fb_nonce'); ?>

            <p class="description">
                <?php esc_html_e('Reorder with arrows or drag-and-drop (if available). “Section” places the field into profile sections. “Search” includes the field in the simple search.', 'speakers-bureau'); ?>
            </p>

            <style>
                .sb-fb-table input[type="text"], .sb-fb-table select { width:100%; }
                .sb-fb-table .col-order { width:60px; text-align:center; }
                .sb-fb-table .col-type { width:160px; }
                .sb-fb-table .col-section { width:180px; }
                .sb-fb-table .col-chk { width:70px; text-align:center; }
                .sb-fb-table .col-fa-icon { width:120px; }
                .sb-fb-row { cursor:move; }
                .sb-fb-add { margin-top:10px; }
                .sb-fb-actions button { color:#b32d2e; border:none; background:none; cursor:pointer; }
                .sb-fb-table { font-size: 13px; }
                .sb-fb-table th, .sb-fb-table td { padding: 8px 4px; }

                /* Icon Picker Styles */
                .sb-icon-picker-wrapper { display:flex; align-items:center; gap:5px; }
                .sb-icon-picker-btn { padding:5px 8px !important; min-height:28px; }
                .sb-icon-picker-btn i { font-size:14px; }
                .sb-icon-label { font-size:11px; color:#666; }

                .sb-icon-picker-backdrop { position:fixed; top:0; left:0; width:100%; height:100%; background:rgba(0,0,0,0.5); z-index:9999; display:flex; align-items:center; justify-content:center; }
                .sb-icon-picker-content { background:white; width:600px; max-width:90vw; max-height:80vh; border-radius:4px; overflow:hidden; }
                .sb-icon-picker-header { padding:15px 20px; border-bottom:1px solid #ddd; display:flex; justify-content:space-between; align-items:center; }
                .sb-icon-picker-header h3 { margin:0; }
                .sb-icon-picker-close { background:none; border:none; font-size:24px; cursor:pointer; color:#666; }
                .sb-icon-picker-search { padding:15px 20px; border-bottom:1px solid #ddd; }
                .sb-icon-search { width:100%; padding:8px 12px; border:1px solid #ddd; border-radius:3px; }
                .sb-icon-picker-grid { padding:20px; max-height:400px; overflow-y:auto; display:grid; grid-template-columns:repeat(auto-fill, minmax(60px, 1fr)); gap:10px; }
                .sb-icon-option { padding:10px; text-align:center; cursor:pointer; border:1px solid #ddd; border-radius:3px; transition:all 0.2s; }
                .sb-icon-option:hover { background:#f0f0f0; border-color:#0073aa; }
                .sb-icon-option i { font-size:20px; display:block; margin-bottom:5px; }
                .sb-icon-option .icon-name { font-size:10px; color:#666; }
                .sb-icon-picker-footer { padding:15px 20px; border-top:1px solid #ddd; text-align:right; }
                .sb-icon-picker-footer .sb-icon-none { float:left; }
            </style>

            <table class="widefat striped sb-fb-table" id="sb-fb-table">
                <thead>
                <tr>
                    <th class="col-order"><?php esc_html_e('Order', 'speakers-bureau'); ?></th>
                    <th><?php esc_html_e('Label', 'speakers-bureau'); ?></th>
                    <th><?php esc_html_e('Key', 'speakers-bureau'); ?></th>
                    <th class="col-type"><?php esc_html_e('Type', 'speakers-bureau'); ?></th>
                    <th><?php esc_html_e('Placeholder / Options', 'speakers-bureau'); ?></th>
                    <th class="col-section"><?php esc_html_e('Section', 'speakers-bureau'); ?></th>
                    <th class="col-chk"><?php esc_html_e('Enabled', 'speakers-bureau'); ?></th>
                    <th class="col-chk"><?php esc_html_e('Required', 'speakers-bureau'); ?></th>
                    <th class="col-chk"><?php esc_html_e('Hide Label', 'speakers-bureau'); ?></th>
                    <th class="col-chk"><?php esc_html_e('Search', 'speakers-bureau'); ?></th>
                    <th class="col-chk"><?php esc_html_e('URL Valid', 'speakers-bureau'); ?></th>
                    <th><?php esc_html_e('FA Icon', 'speakers-bureau'); ?></th>
                    <th class="col-chk"><?php esc_html_e('Actions', 'speakers-bureau'); ?></th>
                </tr>
                </thead>
                <tbody id="sb-fb-body">
                <?php $i = 0; foreach ($rows as $row): ?>
                    <tr class="sb-fb-row">
                        <td class="col-order">
                            <button type="button" class="button-link sb-move-up" title="Move Up">⬆️</button><br>
                            <button type="button" class="button-link sb-move-down" title="Move Down">⬇️</button>
                        </td>
                        <td><input type="text" name="fb_label[]" value="<?php echo esc_attr($row['label']); ?>"></td>
                        <td><input type="text" name="fb_key[]" value="<?php echo esc_attr($row['key']); ?>"></td>
                        <td class="col-type">
                            <select name="fb_type[]">
                                <?php
                                $types = [
                                    'text'      => 'Text',
                                    'email'     => 'Email',
                                    'tel'       => 'Phone',
                                    'url'       => 'URL',
                                    'textarea'  => 'Textarea',
                                    'select'    => 'Select (comma options)',
                                    'checkbox'  => 'Checkbox (comma options)',
                                    'image'     => 'Image (Attachment)',
                                    'edit_link' => 'Edit Link (admin-only)',
                                ];
                                foreach ($types as $val => $label) {
                                    printf('<option value="%s"%s>%s</option>',
                                        esc_attr($val),
                                        selected($row['type'], $val, false),
                                        esc_html($label)
                                    );
                                }
                                ?>
                            </select>
                        </td>
                        <td>
                            <input type="text" name="fb_placeholder[]" value="<?php echo esc_attr($row['placeholder']); ?>">
                            <p class="description" style="margin:2px 0 0;">
                                <?php esc_html_e('For select/checkbox: comma-separated options (e.g., "Option 1, Option 2, Option 3"). For Edit Link: use as the button label.', 'speakers-bureau'); ?>
                            </p>
                        </td>
                        <td class="col-section">
                            <select name="fb_section[]">
                                <?php foreach ($sections as $num => $label): ?>
                                    <option value="<?php echo esc_attr($num); ?>" <?php selected((int)$row['section'], (int)$num); ?>>
                                        <?php echo esc_html($label); ?>
                                    </option>
                                <?php endforeach; ?>
                            </select>
                        </td>
                        <td class="col-chk"><input type="checkbox" name="fb_enabled[<?php echo $i; ?>]"  <?php checked($row['enabled'], 1); ?>></td>
                        <td class="col-chk"><input type="checkbox" name="fb_required[<?php echo $i; ?>]" <?php checked($row['required'], 1); ?>></td>
                        <td class="col-chk"><input type="checkbox" name="fb_hide_label[<?php echo $i; ?>]" <?php checked($row['hide_label'], 1); ?>></td>
                        <td class="col-chk"><input type="checkbox" name="fb_searchable[<?php echo $i; ?>]" <?php checked($row['searchable'], 1); ?>></td>
                        <td class="col-chk">
                            <input type="checkbox" name="fb_validate_url[<?php echo $i; ?>]" <?php checked($row['validate_url'] ?? 0, 1); ?> title="Validate URL for security">
                        </td>
                        <td>
                            <div class="sb-icon-picker-wrapper">
                                <input type="hidden" name="fb_fa_icon[]" value="<?php echo esc_attr($row['fa_icon'] ?? ''); ?>" class="sb-icon-value">
                                <button type="button" class="button sb-icon-picker-btn" title="Choose Icon">
                                    <?php if (!empty($row['fa_icon'])): ?>
                                        <i class="fa <?php echo esc_attr($row['fa_icon']); ?>"></i>
                                    <?php else: ?>
                                        <i class="fa fa-plus"></i>
                                    <?php endif; ?>
                                </button>
                                <span class="sb-icon-label"><?php echo !empty($row['fa_icon']) ? esc_html($row['fa_icon']) : 'None'; ?></span>
                            </div>
                        </td>
                        <td class="col-chk sb-fb-actions">
                            <button type="button" class="sb-fb-del" title="Delete">🗑️</button>
                        </td>
                    </tr>
                <?php $i++; endforeach; ?>
                </tbody>
            </table>

            <p>
                <a href="#" class="button sb-fb-add"><?php esc_html_e('Add field', 'speakers-bureau'); ?></a>
            </p>

            <p>
                <button type="submit" class="button button-primary"><?php esc_html_e('Save Form', 'speakers-bureau'); ?></button>
            </p>
        </form>

        <!-- Icon Picker Modal -->
        <div id="sb-icon-picker-modal" style="display:none;">
            <div class="sb-icon-picker-backdrop">
                <div class="sb-icon-picker-content">
                    <div class="sb-icon-picker-header">
                        <h3><?php esc_html_e('Choose Icon', 'speakers-bureau'); ?></h3>
                        <button type="button" class="sb-icon-picker-close">&times;</button>
                    </div>
                    <div class="sb-icon-picker-search">
                        <input type="text" placeholder="<?php esc_attr_e('Search icons...', 'speakers-bureau'); ?>" class="sb-icon-search">
                    </div>
                    <div class="sb-icon-picker-grid" id="sb-icon-grid">
                        <!-- Icons will be populated by JavaScript -->
                    </div>
                    <div class="sb-icon-picker-footer">
                        <button type="button" class="button sb-icon-none"><?php esc_html_e('Remove Icon', 'speakers-bureau'); ?></button>
                        <button type="button" class="button button-primary sb-icon-picker-close"><?php esc_html_e('Close', 'speakers-bureau'); ?></button>
                    </div>
                </div>
            </div>
        </div>

        <script>
        (function($){
            var $body = $('#sb-fb-body');
            var currentIconInput = null;

            // Popular Font Awesome icons for speakers/contacts
            var iconList = [
                'fa-globe', 'fa-envelope', 'fa-phone', 'fa-mobile-alt', 'fa-fax',
                'fa-linkedin', 'fa-facebook', 'fa-twitter', 'fa-instagram', 'fa-youtube',
                'fa-user', 'fa-users', 'fa-building', 'fa-home', 'fa-map-marker-alt',
                'fa-calendar', 'fa-clock', 'fa-star', 'fa-heart', 'fa-thumbs-up',
                'fa-link', 'fa-external-link-alt', 'fa-download', 'fa-upload', 'fa-file',
                'fa-file-pdf', 'fa-file-word', 'fa-image', 'fa-video', 'fa-music',
                'fa-microphone', 'fa-bullhorn', 'fa-graduation-cap', 'fa-book', 'fa-certificate',
                'fa-briefcase', 'fa-handshake', 'fa-trophy', 'fa-medal', 'fa-award'
            ];

            function renumber(){
                $body.find('tr').each(function(i){
                    $(this).attr('data-index', i);
                });
            }

            // Initialize icon picker
            function initIconPicker() {
                var $grid = $('#sb-icon-grid');
                $grid.empty();

                iconList.forEach(function(icon) {
                    var $option = $('<div class="sb-icon-option" data-icon="' + icon + '">' +
                        '<i class="fa ' + icon + '"></i>' +
                        '<div class="icon-name">' + icon.replace('fa-', '') + '</div>' +
                    '</div>');
                    $grid.append($option);
                });
            }

            // Open icon picker
            $(document).on('click', '.sb-icon-picker-btn', function() {
                currentIconInput = $(this).closest('.sb-icon-picker-wrapper');
                $('#sb-icon-picker-modal').show();
                initIconPicker();
            });

            // Close icon picker
            $(document).on('click', '.sb-icon-picker-close', function() {
                $('#sb-icon-picker-modal').hide();
                currentIconInput = null;
            });

            // Select icon
            $(document).on('click', '.sb-icon-option', function() {
                var icon = $(this).data('icon');
                if (currentIconInput) {
                    currentIconInput.find('.sb-icon-value').val(icon);
                    currentIconInput.find('.sb-icon-picker-btn i').attr('class', 'fa ' + icon);
                    currentIconInput.find('.sb-icon-label').text(icon);
                }
                $('#sb-icon-picker-modal').hide();
                currentIconInput = null;
            });

            // Remove icon
            $(document).on('click', '.sb-icon-none', function() {
                if (currentIconInput) {
                    currentIconInput.find('.sb-icon-value').val('');
                    currentIconInput.find('.sb-icon-picker-btn i').attr('class', 'fa fa-plus');
                    currentIconInput.find('.sb-icon-label').text('None');
                }
                $('#sb-icon-picker-modal').hide();
                currentIconInput = null;
            });

            // Icon search
            $(document).on('input', '.sb-icon-search', function() {
                var search = $(this).val().toLowerCase();
                $('.sb-icon-option').each(function() {
                    var iconName = $(this).data('icon').toLowerCase();
                    if (iconName.indexOf(search) >= 0) {
                        $(this).show();
                    } else {
                        $(this).hide();
                    }
                });
            });

            // Up/down buttons
            $body.on('click', '.sb-move-up', function(){
                var $row = $(this).closest('tr');
                $row.prev().before($row);
                renumber();
            });
            $body.on('click', '.sb-move-down', function(){
                var $row = $(this).closest('tr');
                $row.next().after($row);
                renumber();
            });

            // Delete row
            $body.on('click', '.sb-fb-del', function(e){
                e.preventDefault();
                if (confirm('Delete this field?')) {
                    $(this).closest('tr').remove();
                    renumber();
                }
            });

            // Add row
            $('.sb-fb-add').on('click', function(e){
                e.preventDefault();
                var idx = $body.find('tr').length;
                var row = `
<tr class="sb-fb-row" data-index="${idx}">
  <td class="col-order">
    <button type="button" class="button-link sb-move-up">⬆️</button><br>
    <button type="button" class="button-link sb-move-down">⬇️</button>
  </td>
  <td><input type="text" name="fb_label[]" value=""></td>
  <td><input type="text" name="fb_key[]" value=""></td>
  <td class="col-type">
    <select name="fb_type[]">
      <option value="text">Text</option>
      <option value="email">Email</option>
      <option value="tel">Phone</option>
      <option value="url">URL</option>
      <option value="textarea">Textarea</option>
      <option value="select">Select (comma options)</option>
      <option value="checkbox">Checkbox (comma options)</option>
      <option value="image">Image (Attachment)</option>
      <option value="edit_link">Edit Link (admin-only)</option>
    </select>
  </td>
  <td><input type="text" name="fb_placeholder[]" value=""></td>
  <td class="col-section"><?php foreach ($sections as $num => $label): ?><option value="<?php echo esc_attr($num); ?>"><?php echo esc_html($label); ?></option><?php endforeach; ?></td>
  <td class="col-chk"><input type="checkbox" name="fb_enabled[${idx}]" checked></td>
  <td class="col-chk"><input type="checkbox" name="fb_required[${idx}]"></td>
  <td class="col-chk"><input type="checkbox" name="fb_hide_label[${idx}]"></td>
  <td class="col-chk"><input type="checkbox" name="fb_searchable[${idx}]" checked></td>
  <td class="col-chk"><input type="checkbox" name="fb_validate_url[${idx}]" title="Validate URL for security"></td>
  <td>
    <div class="sb-icon-picker-wrapper">
      <input type="hidden" name="fb_fa_icon[]" value="" class="sb-icon-value">
      <button type="button" class="button sb-icon-picker-btn" title="Choose Icon">
        <i class="fa fa-plus"></i>
      </button>
      <span class="sb-icon-label">None</span>
    </div>
  </td>
  <td class="col-chk sb-fb-actions"><button type="button" class="sb-fb-del">🗑️</button></td>
</tr>`;
                $body.append(row);
                renumber();
            });

            renumber();
        })(jQuery);
        </script>
        <?php
    }
}


/** END OF FILE: includes/sb-form-builder.php */
