<?php
/**
 * Contact Form Functionality
 *
 * Handles the speaker contact form displayed when email/phone is private or missing.
 *
 * @package SpeakersBureau
 */

if (!defined('ABSPATH')) {
    exit;
}

/**
 * Determine if the contact button should be shown for a speaker
 *
 * @param int $post_id Speaker post ID
 * @return bool True if button should be shown
 */
function sb_should_show_contact_button($post_id) {
    // Check if email is private
    $email_private = get_post_meta($post_id, 'email_private', true);
    if ($email_private == '1') {
        return true;
    }

    // Check if phone is private
    $phone_private = get_post_meta($post_id, 'phone_private', true);
    if ($phone_private == '1') {
        return true;
    }

    // Check if phone number is missing
    $phone = get_post_meta($post_id, 'phone', true);
    if (empty($phone)) {
        return true;
    }

    return false;
}

/**
 * Generate a math CAPTCHA challenge
 *
 * @return array ['question' => string, 'answer' => int, 'hash' => string]
 */
function sb_generate_contact_captcha() {
    $operations = ['+', '-'];
    $op = $operations[array_rand($operations)];

    $num1 = rand(1, 20);
    $num2 = rand(1, 20);

    if ($op === '+') {
        $answer = $num1 + $num2;
        $question = "$num1 + $num2";
    } else {
        // For subtraction, ensure result is positive
        if ($num1 < $num2) {
            $temp = $num1;
            $num1 = $num2;
            $num2 = $temp;
        }
        $answer = $num1 - $num2;
        $question = "$num1 - $num2";
    }

    // Create a hash of the answer for verification
    $hash = wp_hash($answer . wp_salt());

    return [
        'question' => $question,
        'answer' => $answer,
        'hash' => $hash
    ];
}

/**
 * Verify a CAPTCHA answer
 *
 * @param int $user_answer The user's submitted answer
 * @param string $hash The expected hash
 * @return bool True if answer is correct
 */
function sb_verify_contact_captcha($user_answer, $hash) {
    $expected_hash = wp_hash($user_answer . wp_salt());
    return hash_equals($hash, $expected_hash);
}

/**
 * AJAX handler to get a new CAPTCHA challenge
 */
function sb_get_contact_captcha_ajax() {
    $captcha = sb_generate_contact_captcha();

    wp_send_json_success([
        'question' => $captcha['question'],
        'hash' => $captcha['hash']
    ]);
}

/**
 * AJAX handler for contact form submission
 */
function sb_handle_contact_form_ajax() {
    // Verify nonce
    if (!isset($_POST['nonce']) || !wp_verify_nonce($_POST['nonce'], 'sb_contact_form')) {
        wp_send_json_error(['message' => 'Security verification failed. Please refresh the page and try again.']);
        return;
    }

    // Validate required fields
    if (empty($_POST['sender_name'])) {
        wp_send_json_error(['message' => 'Please enter your name.']);
        return;
    }

    if (empty($_POST['sender_email']) || !is_email($_POST['sender_email'])) {
        wp_send_json_error(['message' => 'Please enter a valid email address.']);
        return;
    }

    if (empty($_POST['message'])) {
        wp_send_json_error(['message' => 'Please enter a message.']);
        return;
    }

    if (empty($_POST['speaker_id']) || !is_numeric($_POST['speaker_id'])) {
        wp_send_json_error(['message' => 'Invalid speaker ID.']);
        return;
    }

    // Verify CAPTCHA
    if (empty($_POST['captcha_answer']) || empty($_POST['captcha_hash'])) {
        wp_send_json_error(['message' => 'Please answer the security question.']);
        return;
    }

    $captcha_answer = intval($_POST['captcha_answer']);
    $captcha_hash = sanitize_text_field($_POST['captcha_hash']);

    if (!sb_verify_contact_captcha($captcha_answer, $captcha_hash)) {
        wp_send_json_error(['message' => 'Incorrect answer to the security question. Please try again.']);
        return;
    }

    // Sanitize inputs
    $speaker_id = intval($_POST['speaker_id']);
    $sender_data = [
        'name' => sanitize_text_field($_POST['sender_name']),
        'email' => sanitize_email($_POST['sender_email']),
        'phone' => sanitize_text_field($_POST['sender_phone'] ?? ''),
        'message' => sanitize_textarea_field($_POST['message'])
    ];

    // Send email
    $result = sb_send_contact_inquiry_email($speaker_id, $sender_data);

    if ($result) {
        wp_send_json_success(['message' => 'Your message has been sent successfully! The speaker will receive your inquiry and can reply directly to your email.']);
    } else {
        wp_send_json_error(['message' => 'There was an error sending your message. Please try again or contact the speaker directly if their information is available.']);
    }
}

/**
 * Send contact inquiry email to speaker
 *
 * @param int $speaker_id Speaker post ID
 * @param array $sender_data Sender information
 * @return bool True if email sent successfully
 */
function sb_send_contact_inquiry_email($speaker_id, $sender_data) {
    // Get speaker email
    $speaker_email = get_post_meta($speaker_id, 'email', true);
    if (empty($speaker_email) || !is_email($speaker_email)) {
        return false;
    }

    // Get speaker name
    $speaker_post = get_post($speaker_id);
    if (!$speaker_post) {
        return false;
    }
    $speaker_name = $speaker_post->post_title;

    // Get email templates from settings
    $email_templates = get_option('sb_email_templates', []);

    // Default subject and body if not configured
    $default_subject = 'Speakers Bureau Inquiry from {sender_name}';
    $default_body = '<p>Hello {speaker_name},</p>
<p>You have received a new inquiry through the Speakers Bureau:</p>
<div style="border-left: 3px solid #0073aa; padding-left: 15px; margin: 20px 0;">
  <p><strong>From:</strong> {sender_name}<br>
  <strong>Email:</strong> <a href="mailto:{sender_email}">{sender_email}</a><br>
  <strong>Phone:</strong> {sender_phone}</p>
  <p><strong>Message:</strong><br>{message}</p>
</div>
<p>You can reply directly to this email to respond to {sender_name}.</p>
<hr style="margin: 30px 0; border: none; border-top: 1px solid #ddd;">
<p style="font-size: 12px; color: #666;">This inquiry was submitted from your speaker profile:<br><a href="{profile_url}">{profile_url}</a></p>';

    $subject_template = isset($email_templates['contact_subject']) ? $email_templates['contact_subject'] : $default_subject;
    $body_template = isset($email_templates['contact_body']) ? $email_templates['contact_body'] : $default_body;

    // Get speaker profile URL
    $profile_url = get_permalink($speaker_id);

    // Template variables
    $variables = [
        '{sender_name}' => $sender_data['name'],
        '{sender_email}' => $sender_data['email'],
        '{sender_phone}' => $sender_data['phone'] ?: 'Not provided',
        '{message}' => nl2br($sender_data['message']),
        '{speaker_name}' => $speaker_name,
        '{profile_url}' => $profile_url
    ];

    // Replace template variables
    $subject = str_replace(array_keys($variables), array_values($variables), $subject_template);
    $body = str_replace(array_keys($variables), array_values($variables), $body_template);

    // Get BCC email (configurable, defaults to admin email)
    $bcc_email = get_option('sb_contact_form_bcc_email', get_option('admin_email'));

    // Set up email headers
    $headers = [
        'Content-Type: text/html; charset=UTF-8',
        'Reply-To: ' . $sender_data['email']
    ];

    // Add BCC if configured
    if (!empty($bcc_email) && is_email($bcc_email)) {
        $headers[] = 'Bcc: ' . $bcc_email;
    }

    // Send email
    return wp_mail($speaker_email, $subject, $body, $headers);
}

// Register AJAX hooks
add_action('wp_ajax_sb_contact_speaker', 'sb_handle_contact_form_ajax');
add_action('wp_ajax_nopriv_sb_contact_speaker', 'sb_handle_contact_form_ajax');
add_action('wp_ajax_sb_get_contact_captcha', 'sb_get_contact_captcha_ajax');
add_action('wp_ajax_nopriv_sb_get_contact_captcha', 'sb_get_contact_captcha_ajax');
