<?php
/**
 * File: includes/sb-bulk-geocode.php
 * Purpose: Part of the Speakers Bureau plugin.
 */

if (!defined('ABSPATH')) exit;

function sb_register_bulk_geocode_page() {
    add_submenu_page(
        'edit.php?post_type=speaker',
        __('Bulk Geocode','sb'),
        __('Bulk Geocode','sb'),
        'manage_options',
        'sb-bulk-geocode',
        'sb_render_bulk_geocode_page'
    );
}
add_action('admin_menu', 'sb_register_bulk_geocode_page');

function sb_render_bulk_geocode_page() {
    if (isset($_POST['sb_bulk_geocode']) && check_admin_referer('sb_bulk_geocode')) {
        $count = 0;

        // Optimized query to get speakers with all needed meta in one go
        global $wpdb;
        $query = "
            SELECT p.ID,
                   MAX(CASE WHEN pm.meta_key = 'sb_address' THEN pm.meta_value END) as address,
                   MAX(CASE WHEN pm.meta_key = 'sb_city' THEN pm.meta_value END) as city,
                   MAX(CASE WHEN pm.meta_key = 'sb_state' THEN pm.meta_value END) as state,
                   MAX(CASE WHEN pm.meta_key = 'sb_zip' THEN pm.meta_value END) as zip
            FROM {$wpdb->posts} p
            LEFT JOIN {$wpdb->postmeta} pm ON p.ID = pm.post_id
                AND pm.meta_key IN ('sb_address', 'sb_city', 'sb_state', 'sb_zip')
            WHERE p.post_type = 'speaker'
            AND p.post_status = 'publish'
            GROUP BY p.ID
            HAVING (address IS NOT NULL AND address != '')
                OR (city IS NOT NULL AND city != '')
                OR (state IS NOT NULL AND state != '')
                OR (zip IS NOT NULL AND zip != '')
        ";

        $speakers = $wpdb->get_results($query);

        // Get default country from settings
        $settings = get_option('sb_settings', []);
        $default_country = isset($settings['geo_default_country']) ? $settings['geo_default_country'] : 'US';

        foreach ($speakers as $speaker) {
            // Check if we have any address data
            if (empty($speaker->address) && empty($speaker->city) && empty($speaker->state) && empty($speaker->zip)) {
                continue;
            }

            // Get country for this speaker
            $country = get_post_meta($speaker->ID, 'country', true);
            if (empty($country)) $country = get_post_meta($speaker->ID, 'sb_country', true);
            if (empty($country)) {
                $country = $default_country;
            }

            // Build structured address components for better geocoding precision
            $address_components = [
                'street' => $speaker->address,
                'city' => $speaker->city,
                'state' => $speaker->state,
                'postalcode' => $speaker->zip,
                'country' => $country
            ];

            // Use the geocoding function with structured query
            $coords = sb_geocode_address($address_components, $country);

            if ($coords) {
                update_post_meta($speaker->ID, 'geo_lat', $coords['lat']);
                update_post_meta($speaker->ID, 'geo_lng', $coords['lng']);
                // Also update old format for backward compatibility
                update_post_meta($speaker->ID, '_sb_lat', $coords['lat']);
                update_post_meta($speaker->ID, '_sb_lng', $coords['lng']);
                $count++;
            }
        }
        echo '<div class="updated"><p>'.sprintf(esc_html__('%d speakers geocoded.','sb'), intval($count)).'</p></div>';
    }
    ?>
    <div class="wrap"><h1><?php esc_html_e('Bulk Geocode Speakers','sb'); ?></h1>
    <form method="post">
        <?php wp_nonce_field('sb_bulk_geocode'); ?>
        <p><?php esc_html_e('This will look up latitude/longitude for existing speakers that have address information.','sb'); ?></p>
        <p><input type="submit" name="sb_bulk_geocode" class="button button-primary" value="<?php esc_attr_e('Run Bulk Geocode','sb'); ?>"></p>
    </form></div>
    <?php
}

/** END OF FILE: includes/sb-bulk-geocode.php */
