<?php
/**
 * File: includes/sb-templates.php
 * Purpose: Part of the Speakers Bureau plugin.
 */

if (!defined('ABSPATH')) exit;

function sb_template_include($template) {
    // Ensure template is a string to prevent block template warnings
    if (!is_string($template)) {
        $template = '';
    }

    if (is_singular('speaker')) {
        $theme_template = get_stylesheet_directory() . '/speakers-bureau/single-speaker.php';
        $plugin_template = plugin_dir_path(dirname(__FILE__)) . 'templates/single-speaker.php';

        $selected_template = file_exists($theme_template) ? $theme_template : (file_exists($plugin_template) ? $plugin_template : $template);

        // Ensure we return a valid string
        return is_string($selected_template) ? $selected_template : $template;
    }
    if (is_post_type_archive('speaker')) {
        $theme_template = get_stylesheet_directory() . '/speakers-bureau/archive-speaker.php';
        $plugin_template = plugin_dir_path(dirname(__FILE__)) . 'templates/archive-speaker.php';
        $selected_template = file_exists($theme_template) ? $theme_template : (file_exists($plugin_template) ? $plugin_template : $template);

        // Ensure we return a valid string
        return is_string($selected_template) ? $selected_template : $template;
    }
    return $template;
}
add_filter('template_include', 'sb_template_include', 999);


/**
 * Prevent WordPress block template warnings by ensuring template hierarchy values are strings
 */
function sb_prevent_block_template_warnings() {
    if (is_singular('speaker') || is_post_type_archive('speaker')) {
        // Ensure global template variables are strings to prevent str_starts_with warnings
        global $wp_query, $_wp_current_template_content;

        if (isset($wp_query) && is_object($wp_query)) {
            // Clean up query vars that might cause block template issues
            $problematic_vars = ['template', 'pagename', 'name', 'category_name', 'tag'];
            foreach ($problematic_vars as $var) {
                if (isset($wp_query->query_vars[$var]) && !is_string($wp_query->query_vars[$var])) {
                    $wp_query->query_vars[$var] = '';
                }
            }
        }

        // Ensure WordPress template globals are properly initialized
        if (!isset($_wp_current_template_content)) {
            $_wp_current_template_content = '';
        }
    }
}
add_action('wp', 'sb_prevent_block_template_warnings', 5);

/**
 * Early hook to prevent block template system issues
 */
function sb_early_template_safety() {
    if (is_admin()) return;

    // Check if we're on a local development environment
    $is_local = (
        isset($_SERVER['HTTP_HOST']) && (
            strpos($_SERVER['HTTP_HOST'], '.local') !== false ||
            strpos($_SERVER['HTTP_HOST'], 'localhost') !== false ||
            strpos($_SERVER['HTTP_HOST'], '127.0.0.1') !== false ||
            strpos($_SERVER['HTTP_HOST'], '.test') !== false ||
            strpos($_SERVER['HTTP_HOST'], '.dev') !== false
        )
    );

    // Apply targeted fixes for local environments (removed aggressive theme.json disabling)
    if ($is_local) {
        // Only disable block editor for speaker post type in admin
        add_filter('use_block_editor_for_post_type', function($enabled, $post_type) {
            if ($post_type === 'speaker') {
                return false;
            }
            return $enabled;
        }, 10, 2);
    }

    // More aggressive block template prevention for speaker pages
    if ((isset($_SERVER['REQUEST_URI']) && strpos($_SERVER['REQUEST_URI'], '/speaker/') !== false) ||
        (isset($_GET['post_type']) && $_GET['post_type'] === 'speaker')) {

        // Prevent WordPress from trying to load block templates for speaker pages
        add_filter('pre_get_block_template', function($template, $id, $template_type) {
            if ($template_type === 'wp_template') {
                return false; // Force use of PHP templates instead of block templates
            }
            return $template;
        }, 1, 3);

        // Additional local environment protection
        if ($is_local) {
            // Prevent all block template queries for speaker pages
            add_filter('get_block_templates', function($templates, $query) {
                return []; // Return empty array to force PHP template usage
            }, 1, 2);

            // Also prevent theme template resolution
            add_filter('get_block_template', function($template, $id, $template_type) {
                if ($template_type === 'wp_template') {
                    return null;
                }
                return $template;
            }, 1, 3);
        }
    }
}
add_action('init', 'sb_early_template_safety', 1);

/**
 * Additional safety for template hierarchy to prevent null values in WordPress core
 */
function sb_sanitize_template_hierarchy($templates) {
    if (is_array($templates)) {
        $templates = array_filter($templates, function($template) {
            return is_string($template) && !empty($template);
        });
    }
    return $templates;
}
add_filter('template_hierarchy', 'sb_sanitize_template_hierarchy', 10);

/** END OF FILE: includes/sb-templates.php */
