<?php
/**
 * File: includes/sb-settings.php
 * Purpose: Admin settings page (Directory Options, Form Builder, Email Templates, Profile Layout)
 *
 * UPDATE (2025-09-19):
 * - Keep ALL existing functionality from prior file.
 * - Fixed sanitize_profile_layout() to normalize to exactly 8 sections (no duplication).
 * - Do not call migrations on admin_init; migrations should run on activation or behind a guard elsewhere.
 */

if (!defined('ABSPATH')) exit;

// Optional: keep available for callers, but do NOT hook migrations here.
require_once plugin_dir_path(__FILE__) . 'sb-migrations.php';

class SB_Settings_Page {
    private $option_group_general = 'sb_general_settings';
    private $option_group_email   = 'sb_email_settings';
    private $option_group_layout  = 'sb_layout_settings';
    private $option_group_display = 'sb_display_settings_group';

    private $option_name_general = 'sb_settings';         // Directory options
    private $option_name_email   = 'sb_email_templates';  // Email templates
    private $option_name_layout  = 'sb_profile_layout';   // Profile layout
    private $option_name_form    = 'sb_form_fields';      // Form builder fields
    private $option_name_display = 'sb_display_settings'; // Display settings

    public function __construct() {
        add_action('admin_menu',  [$this, 'add_settings_page']);
        add_action('admin_init',  [$this, 'register_settings']);

        // Export/Import handlers
        add_action('admin_post_sb_export_settings', [$this, 'handle_export_settings']);
        add_action('admin_post_sb_import_settings', [$this, 'handle_import_settings']);

        // NOTE: per your directive, DO NOT run migrations here.
        // They must run only on plugin activation or behind an explicit guard.
        // add_action('admin_init', ['SB_Migrations', 'fix_profile_layout_option']); // <- intentionally removed

        // Admin actions (maintenance buttons)
        add_action('admin_post_sb_rebuild_geo',  [$this, 'handle_rebuild_geo']);
        add_action('admin_post_sb_create_pages', [$this, 'handle_create_pages']);
        add_action('admin_post_sb_test_email',   [$this, 'handle_test_email']);

        // AJAX handlers for batch geocoding with progress
        add_action('wp_ajax_sb_get_geocode_count', [$this, 'ajax_get_geocode_count']);
        add_action('wp_ajax_sb_geocode_batch', [$this, 'ajax_geocode_batch']);
    }

    public function add_settings_page() {
        add_menu_page(
            __('Speakers Bureau', 'speakers-bureau'),
            __('Speakers Bureau', 'speakers-bureau'),
            'manage_options',
            'sb-settings',
            [$this, 'render_settings_page'],
            'dashicons-microphone',
            58
        );
    }

    public function register_settings() {
        // Directory Options
        register_setting($this->option_group_general, $this->option_name_general, [
            'sanitize_callback' => [$this, 'sanitize_general'],
            'default'           => $this->default_general(),
        ]);

        // Email templates
        register_setting($this->option_group_email, $this->option_name_email, [
            'sanitize_callback' => [$this, 'sanitize_email_templates'],
        ]);

        // Profile layout (sanitizer normalizes to exactly 8 sections)
        register_setting($this->option_group_layout, $this->option_name_layout, [
            'sanitize_callback' => [$this, 'sanitize_profile_layout'],
            'default'           => $this->default_profile_layout(),
        ]);

        // Display settings
        register_setting($this->option_group_display, $this->option_name_display, [
            'sanitize_callback' => [$this, 'sanitize_display_settings'],
            'default'           => $this->default_display_settings(),
        ]);

        // Seed defaults for form builder
        $current = get_option($this->option_name_form);
        if (!$current || !is_array($current)) {
            update_option($this->option_name_form, $this->default_form_fields());
        }
    }

    /* ================= Defaults & Sanitizers ================ */

    private function default_general() {
        return [
            'enable_geo'           => 1,
            'geo_default_country'  => 'US',
            'geo_default_radius'   => 50,
            'enable_nearest_sort'  => 1,
        ];
    }

    public function sanitize_general($input) {
        $out = $this->default_general();
        if (!is_array($input)) return $out;

        $out['enable_geo']          = !empty($input['enable_geo']) ? 1 : 0;
        $out['geo_default_country'] = isset($input['geo_default_country']) ? strtoupper(sanitize_text_field($input['geo_default_country'])) : $out['geo_default_country'];
        $out['geo_default_radius']  = isset($input['geo_default_radius']) ? max(1, intval($input['geo_default_radius'])) : $out['geo_default_radius'];
        $out['enable_nearest_sort'] = !empty($input['enable_nearest_sort']) ? 1 : 0;

        return $out;
    }

    public function default_profile_layout() {
        return [
            1 => ['title'=>'Profile','heading'=>'h4','display'=>1,'class'=>'sb-section-1','hidden'=>0],
            2 => ['title'=>'Overview','heading'=>'h2','display'=>1,'class'=>'sb-section-2','hidden'=>0],
            3 => ['title'=>'Topics','heading'=>'h3','display'=>1,'class'=>'sb-section-3','hidden'=>0],
            4 => ['title'=>'Background','heading'=>'h3','display'=>1,'class'=>'sb-section-4','hidden'=>0],
            5 => ['title'=>'','heading'=>'h3','display'=>0,'class'=>'sb-section-5','hidden'=>1],
            6 => ['title'=>'','heading'=>'h3','display'=>0,'class'=>'sb-section-6','hidden'=>1],
            7 => ['title'=>'More Information','heading'=>'h2','display'=>1,'class'=>'sb-section-7','hidden'=>0],
            8 => ['title'=>'Private Fields','heading'=>'h3','display'=>0,'class'=>'sb-section-8','hidden'=>1],
        ];
    }

    /**
     * IMPORTANT: Normalize to exactly 8 sections (indexes 1..8) every save.
     * - Prevents duplication/append issues previously seen in the DB.
     * - No reliance on prior DB shape; missing rows fall back to defaults.
     */
    public function sanitize_profile_layout($input) {
        $defaults = $this->default_profile_layout();
        $output   = [];

        if (!is_array($input)) {
            // If the raw input isn't an array, just return clean defaults
            foreach (range(1, 8) as $i) {
                $output[$i] = $defaults[$i];
            }
            return $output;
        }

        foreach (range(1, 8) as $i) {
            $row = isset($input[$i]) && is_array($input[$i]) ? $input[$i] : [];

            // Title
            $title = array_key_exists('title', $row)
                ? wp_kses_post($row['title'])
                : $defaults[$i]['title'];

            // Heading
            $heading = array_key_exists('heading', $row)
                ? strtolower(sanitize_text_field($row['heading']))
                : $defaults[$i]['heading'];
            if (!in_array($heading, ['h1','h2','h3','h4','h5','h6'], true)) {
                $heading = $defaults[$i]['heading'];
            }

            // Display & Hidden (checkboxes)
            $display = !empty($row['display']) ? 1 : 0;
            $hidden  = !empty($row['hidden'])  ? 1 : 0;

            // CSS class
            $class = array_key_exists('class', $row)
                ? sanitize_html_class($row['class'])
                : $defaults[$i]['class'];

            $output[$i] = [
                'title'   => $title,
                'heading' => $heading,
                'display' => $display,
                'class'   => $class,
                'hidden'  => $hidden,
            ];
        }

        return $output;
    }

    public function sanitize_email_templates($input) {
        if (!is_array($input)) return [];

        $sanitized = [];
        $allowed_keys = ['speaker_subject', 'speaker_body', 'admin_subject', 'admin_body', 'update_subject', 'update_body'];

        foreach ($allowed_keys as $key) {
            if (isset($input[$key])) {
                if (in_array($key, ['speaker_body', 'admin_body', 'update_body'])) {
                    // Allow HTML for email bodies
                    $sanitized[$key] = wp_kses_post($input[$key]);
                } else {
                    // Plain text for subjects
                    $sanitized[$key] = sanitize_text_field($input[$key]);
                }
            }
        }

        return $sanitized;
    }

    private function default_form_fields() {
        return [
            [
                "label" => "Profile Image",
                "key" => "profile_image",
                "type" => "image",
                "placeholder" => "",
                "enabled" => 1,
                "required" => 0,
                "hide_label" => 0,
                "section" => 1,
                "searchable" => 0
            ],
            [
                "label" => "Headline",
                "key" => "headline",
                "type" => "text",
                "placeholder" => "e.g., Keynote Speaker on Leadership",
                "enabled" => 1,
                "required" => 0,
                "hide_label" => 0,
                "section" => 1,
                "searchable" => 1
            ],
            [
                "label" => "Organization",
                "key" => "organization",
                "type" => "text",
                "placeholder" => "Your company name",
                "enabled" => 1,
                "required" => 0,
                "hide_label" => 0,
                "section" => 1,
                "searchable" => 1
            ],
            [
                "label" => "Topics",
                "key" => "topics",
                "type" => "text",
                "placeholder" => "Youth, Camping, Topics (Comma Separated)",
                "enabled" => 1,
                "required" => 0,
                "hide_label" => 0,
                "section" => 2,
                "searchable" => 1
            ],
            [
                "label" => "Bio",
                "key" => "bio",
                "type" => "textarea",
                "placeholder" => "The speaker's personal biography or background",
                "enabled" => 1,
                "required" => 0,
                "hide_label" => 0,
                "section" => 2,
                "searchable" => 1
            ],
            [
                "label" => "Email",
                "key" => "email",
                "type" => "email",
                "placeholder" => "Your personal email address (for registering on this website)",
                "enabled" => 1,
                "required" => 0,
                "hide_label" => 0,
                "section" => 2,
                "searchable" => 1
            ],
            [
                "label" => "Phone",
                "key" => "phone",
                "type" => "tel",
                "placeholder" => "(123) 456-7890",
                "enabled" => 1,
                "required" => 0,
                "hide_label" => 0,
                "section" => 2,
                "searchable" => 1
            ],
            [
                "label" => "Your Website",
                "key" => "website",
                "type" => "url",
                "placeholder" => "Speaker's website - url starts with https://",
                "enabled" => 1,
                "required" => 0,
                "hide_label" => 0,
                "section" => 2,
                "searchable" => 1
            ],
            [
                "label" => "Cell Phone",
                "key" => "cell-phone",
                "type" => "text",
                "placeholder" => "Your Cell phonumber (if you want this to be public)",
                "enabled" => 1,
                "required" => 0,
                "hide_label" => 0,
                "section" => 2,
                "searchable" => 0
            ],
            [
                "label" => "Edit This Profile",
                "key" => "placeholder",
                "type" => "edit_link",
                "placeholder" => "",
                "enabled" => 1,
                "required" => 0,
                "hide_label" => 0,
                "section" => 1,
                "searchable" => 0
            ],
            [
                "label" => "Topic Image",
                "key" => "topic-image",
                "type" => "image",
                "placeholder" => "",
                "enabled" => 1,
                "required" => 0,
                "hide_label" => 0,
                "section" => 2,
                "searchable" => 0
            ],
            [
                "label" => "Presentation Summary",
                "key" => "summary",
                "type" => "textarea",
                "placeholder" => "Summary or overview of the presentation topic",
                "enabled" => 1,
                "required" => 0,
                "hide_label" => 0,
                "section" => 2,
                "searchable" => 1
            ],
            [
                "label" => "City",
                "key" => "city",
                "type" => "text",
                "placeholder" => "The city where you are located",
                "enabled" => 1,
                "required" => 0,
                "hide_label" => 0,
                "section" => 1,
                "searchable" => 1
            ],
            [
                "label" => "State",
                "key" => "state",
                "type" => "text",
                "placeholder" => "The 2-character state code where you are located eg \"CA\" or \"NV\" etc.",
                "enabled" => 1,
                "required" => 0,
                "hide_label" => 0,
                "section" => 1,
                "searchable" => 0
            ],
            [
                "label" => "Zip Code",
                "key" => "zip",
                "type" => "text",
                "placeholder" => "5-Digit zip code",
                "enabled" => 1,
                "required" => 0,
                "hide_label" => 0,
                "section" => 1,
                "searchable" => 1
            ],
            [
                "label" => "Available for Online",
                "key" => "online-mtg",
                "type" => "select",
                "placeholder" => "YES, NO",
                "enabled" => 1,
                "required" => 0,
                "hide_label" => 0,
                "section" => 1,
                "searchable" => 0
            ],
            [
                "label" => "Country",
                "key" => "country",
                "type" => "select",
                "placeholder" => "USA, CANADA, OTHER",
                "enabled" => 1,
                "required" => 0,
                "hide_label" => 0,
                "section" => 1,
                "searchable" => 0
            ],
            [
                "label" => "Facebook URL",
                "key" => "facebookurl",
                "type" => "url",
                "placeholder" => "url starts with https://www.facebook.com/",
                "enabled" => 1,
                "required" => 0,
                "hide_label" => 0,
                "section" => 7,
                "searchable" => 1
            ],
            [
                "label" => "X/Twitter URL",
                "key" => "xurl",
                "type" => "url",
                "placeholder" => "url starts with https://x.com/",
                "enabled" => 1,
                "required" => 0,
                "hide_label" => 0,
                "section" => 7,
                "searchable" => 1
            ],
            [
                "label" => "Youtube",
                "key" => "youtubeurl",
                "type" => "url",
                "placeholder" => "url starts with https://",
                "enabled" => 1,
                "required" => 0,
                "hide_label" => 0,
                "section" => 7,
                "searchable" => 1
            ]
        ];
    }

    private function default_display_settings() {
        return [
            'columns' => 3,
            'per_page' => 12,
            'card_fields' => ['bio', 'topics'],
            'max_words' => 25,
            'card_style' => 'default',
            'image_style' => 'square',
            'show_excerpt' => true,
            'excerpt_length' => 20,
            'primary_color' => '#0073aa',
            'card_border_radius' => '8',
            'card_shadow' => 'light',
            'show_date_created' => false,
            'show_date_updated' => false,
            'date_format' => 'm/d/Y g:ia',
            'auto_publish_profiles' => false
        ];
    }

    public function sanitize_display_settings($input) {
        if (!is_array($input)) return $this->default_display_settings();

        $defaults = $this->default_display_settings();
        $sanitized = [];

        $sanitized['columns'] = max(1, min(6, intval($input['columns'] ?? $defaults['columns'])));
        $sanitized['per_page'] = max(1, min(100, intval($input['per_page'] ?? $defaults['per_page'])));
        $sanitized['max_words'] = max(5, min(100, intval($input['max_words'] ?? $defaults['max_words'])));
        $sanitized['excerpt_length'] = max(5, min(50, intval($input['excerpt_length'] ?? $defaults['excerpt_length'])));
        $sanitized['card_border_radius'] = max(0, min(20, intval($input['card_border_radius'] ?? $defaults['card_border_radius'])));

        $sanitized['card_fields'] = isset($input['card_fields']) && is_array($input['card_fields']) ?
            array_map('sanitize_key', $input['card_fields']) : $defaults['card_fields'];

        $sanitized['card_style'] = in_array($input['card_style'] ?? '', ['default', 'compact', 'detailed']) ?
            $input['card_style'] : $defaults['card_style'];

        $sanitized['image_style'] = in_array($input['image_style'] ?? '', ['square', 'circle', 'rounded']) ?
            $input['image_style'] : $defaults['image_style'];

        $sanitized['card_shadow'] = in_array($input['card_shadow'] ?? '', ['none', 'light', 'medium', 'strong']) ?
            $input['card_shadow'] : $defaults['card_shadow'];

        $sanitized['show_excerpt'] = !empty($input['show_excerpt']);
        $sanitized['show_date_created'] = !empty($input['show_date_created']);
        $sanitized['show_date_updated'] = !empty($input['show_date_updated']);
        $sanitized['auto_publish_profiles'] = !empty($input['auto_publish_profiles']);

        $sanitized['primary_color'] = sanitize_hex_color($input['primary_color'] ?? $defaults['primary_color']) ?: $defaults['primary_color'];

        $sanitized['date_format'] = sanitize_text_field($input['date_format'] ?? $defaults['date_format']);

        return $sanitized;
    }

    /* ================= Tabs & Rendering ================= */

    private function current_tab() {
        return isset($_GET['tab']) ? sanitize_key($_GET['tab']) : 'general';
    }

    private function tabs(){ return [
        'docs'            => __('Docs', 'speakers-bureau'),
        'general'         => __('Directory Options', 'speakers-bureau'),
        'form_builder'    => __('Form Builder', 'speakers-bureau'),
        'email_templates' => __('Email Templates', 'speakers-bureau'),
        'profile_layout'  => __('Profile Layout', 'speakers-bureau'),
        'display'         => __('Display', 'speakers-bureau'),
    ]; }

    public function render_settings_page() {
        if (!current_user_can('manage_options')) return;

        $active_tab = $this->current_tab();
        $tabs = $this->tabs();
        ?>
        <div class="wrap sb-wrap">
            <h1 class="sb-title"><?php esc_html_e('Speakers Bureau — Settings', 'speakers-bureau'); ?></h1>
            <?php $this->render_admin_messages($_GET['message'] ?? ''); ?>

            <h2 class="nav-tab-wrapper">
                <?php foreach ($tabs as $slug => $label): ?>
                    <a class="nav-tab <?php echo $active_tab === $slug ? 'nav-tab-active' : ''; ?>"
                       href="<?php echo esc_url(add_query_arg(['page'=>'sb-settings','tab'=>$slug], admin_url('admin.php'))); ?>">
                        <?php echo esc_html($label); ?>
                    </a>
                <?php endforeach; ?>
            </h2>

            <?php if (isset($_GET['sb_notice'])): ?>
                <div class="notice notice-success is-dismissible"><p><?php echo esc_html(wp_unslash($_GET['sb_notice'])); ?></p></div>
            <?php endif; ?>
            <?php if (isset($_GET['sb_error'])): ?>
                <div class="notice notice-error is-dismissible"><p><?php echo esc_html(wp_unslash($_GET['sb_error'])); ?></p></div>
            <?php endif; ?>

            <?php if ($active_tab === 'form_builder'): ?>
                <?php
                if (class_exists('SB_Form_Builder')) {
                    $builder = new SB_Form_Builder();
                    $builder->render_form_builder();
                } else {
                    echo '<p>'.esc_html__('Form Builder not available.', 'speakers-bureau').'</p>';
                }
                ?>
            <?php else: ?>
                <form method="post" action="options.php">
                    <?php
                    // Use the appropriate option group based on the active tab
                    switch ($active_tab) {
                        case 'docs':
                            // Docs don't need settings fields
                            $this->render_docs_tab();
                            break;
                        case 'email_templates':
                            settings_fields($this->option_group_email);
                            $this->render_email_templates_tab();
                            break;
                        case 'profile_layout':
                            settings_fields($this->option_group_layout);
                            $this->render_profile_layout_tab();
                            break;
                        case 'display':
                            settings_fields($this->option_group_display);
                            $this->render_display_tab();
                            break;
                        case 'general':
                        default:
                            settings_fields($this->option_group_general);
                            $this->render_general_tab();
                            break;
                    }

                    // Only show save button for general tab above export/import
                    if ($active_tab !== 'general') {
                        submit_button(__('Save Changes', 'speakers-bureau'));
                    }
                    ?>
                </form>

                <?php if ($active_tab === 'general'): ?>
                    <?php submit_button(__('Save Changes', 'speakers-bureau'), 'primary', 'submit', false); ?>

                    <hr style="margin: 30px 0;">
                    <h2><?php esc_html_e('Configuration Export/Import', 'speakers-bureau'); ?></h2>
                    <p class="description"><?php esc_html_e('Export your current form builder and profile layout settings to backup or share with other installations. Import settings to restore or apply configurations from another site.', 'speakers-bureau'); ?></p>

                    <div style="display: flex; gap: 30px; margin: 20px 0;">
                        <!-- Export Section -->
                        <div style="flex: 1; padding: 20px; border: 1px solid #ddd; border-radius: 4px; background: #f9f9f9;">
                            <h3 style="margin-top: 0;"><?php esc_html_e('Export Settings', 'speakers-bureau'); ?></h3>
                            <p><?php esc_html_e('Download your current configuration as a JSON file.', 'speakers-bureau'); ?></p>

                            <form method="post" action="<?php echo esc_url(admin_url('admin-post.php')); ?>" style="margin-bottom: 15px;">
                                <input type="hidden" name="action" value="sb_export_settings">
                                <?php wp_nonce_field('sb_export_settings'); ?>
                                <label style="display: block; margin-bottom: 10px;">
                                    <input type="checkbox" name="export_form_fields" value="1" checked>
                                    <?php esc_html_e('Form Builder Fields', 'speakers-bureau'); ?>
                                </label>
                                <label style="display: block; margin-bottom: 10px;">
                                    <input type="checkbox" name="export_profile_layout" value="1" checked>
                                    <?php esc_html_e('Profile Layout', 'speakers-bureau'); ?>
                                </label>
                                <label style="display: block; margin-bottom: 10px;">
                                    <input type="checkbox" name="export_email_templates" value="1">
                                    <?php esc_html_e('Email Templates', 'speakers-bureau'); ?>
                                </label>
                                <label style="display: block; margin-bottom: 15px;">
                                    <input type="checkbox" name="export_display_settings" value="1">
                                    <?php esc_html_e('Display Settings', 'speakers-bureau'); ?>
                                </label>
                                <button type="submit" class="button button-primary">
                                    <?php esc_html_e('Download Export File', 'speakers-bureau'); ?>
                                </button>
                            </form>
                        </div>

                        <!-- Import Section -->
                        <div style="flex: 1; padding: 20px; border: 1px solid #ddd; border-radius: 4px; background: #f9f9f9;">
                            <h3 style="margin-top: 0;"><?php esc_html_e('Import Settings', 'speakers-bureau'); ?></h3>
                            <p><?php esc_html_e('Upload a JSON configuration file to import settings.', 'speakers-bureau'); ?></p>

                            <form method="post" action="<?php echo esc_url(admin_url('admin-post.php')); ?>" enctype="multipart/form-data">
                                <input type="hidden" name="action" value="sb_import_settings">
                                <?php wp_nonce_field('sb_import_settings'); ?>

                                <div style="margin-bottom: 15px;">
                                    <label for="import_file" style="display: block; margin-bottom: 5px; font-weight: 600;">
                                        <?php esc_html_e('Select Import File:', 'speakers-bureau'); ?>
                                    </label>
                                    <input type="file" name="import_file" id="import_file" accept=".json" required style="width: 100%;">
                                    <p class="description"><?php esc_html_e('Choose a JSON file exported from Speakers Bureau.', 'speakers-bureau'); ?></p>
                                </div>

                                <div style="margin-bottom: 15px;">
                                    <label style="display: block; margin-bottom: 5px; font-weight: 600;">
                                        <?php esc_html_e('Import Options:', 'speakers-bureau'); ?>
                                    </label>
                                    <label style="display: block; margin-bottom: 8px;">
                                        <input type="radio" name="import_mode" value="merge" checked>
                                        <?php esc_html_e('Merge with existing settings', 'speakers-bureau'); ?>
                                    </label>
                                    <label style="display: block; margin-bottom: 15px;">
                                        <input type="radio" name="import_mode" value="replace">
                                        <?php esc_html_e('Replace existing settings', 'speakers-bureau'); ?>
                                    </label>
                                </div>

                                <button type="submit" class="button button-secondary">
                                    <?php esc_html_e('Import Settings', 'speakers-bureau'); ?>
                                </button>
                            </form>
                        </div>
                    </div>

                    <div style="padding: 15px; background: #fff3cd; border: 1px solid #ffeaa7; border-radius: 4px; margin-top: 20px;">
                        <strong><?php esc_html_e('Important Notes:', 'speakers-bureau'); ?></strong>
                        <ul style="margin: 10px 0; padding-left: 20px;">
                            <li><?php esc_html_e('Always backup your current settings before importing.', 'speakers-bureau'); ?></li>
                            <li><?php esc_html_e('Import will only affect the selected configuration areas.', 'speakers-bureau'); ?></li>
                            <li><?php esc_html_e('Existing speaker data and content will not be affected.', 'speakers-bureau'); ?></li>
                            <li><?php esc_html_e('Test imported configurations on a staging site first.', 'speakers-bureau'); ?></li>
                        </ul>
                    </div>
                <?php endif; ?>

                <?php if ($active_tab === 'email_templates'): ?>
                    <hr style="margin:24px 0;">
                    <h2><?php esc_html_e('Test Email Templates', 'speakers-bureau'); ?></h2>
                    <p class="description"><?php esc_html_e('Send test emails to verify your templates are working correctly.', 'speakers-bureau'); ?></p>

                    <div style="display: flex; gap: 20px; margin: 20px 0;">
                        <div style="flex: 1; padding: 15px; border: 1px solid #ddd; border-radius: 4px;">
                            <h4 style="margin-top: 0;"><?php esc_html_e('Speaker Confirmation Email', 'speakers-bureau'); ?></h4>
                            <p class="description"><?php esc_html_e('Test the email that speakers receive when they register.', 'speakers-bureau'); ?></p>
                            <form method="post" action="<?php echo esc_url(admin_url('admin-post.php')); ?>">
                                <input type="hidden" name="action" value="sb_test_email">
                                <input type="hidden" name="email_type" value="speaker">
                                <?php wp_nonce_field('sb_test_email'); ?>
                                <button type="submit" class="button button-secondary">
                                    <span class="dashicons dashicons-email" style="margin-right: 4px; font-size: 16px; vertical-align: middle;"></span>
                                    <?php esc_html_e('Send Test Email', 'speakers-bureau'); ?>
                                </button>
                            </form>
                            <p class="description" style="margin-top: 8px;">
                                <?php printf(esc_html__('Test email will be sent to: %s', 'speakers-bureau'), '<strong>' . esc_html(wp_get_current_user()->user_email) . '</strong>'); ?>
                            </p>
                        </div>

                        <div style="flex: 1; padding: 15px; border: 1px solid #ddd; border-radius: 4px;">
                            <h4 style="margin-top: 0;"><?php esc_html_e('Admin Notification Email', 'speakers-bureau'); ?></h4>
                            <p class="description"><?php esc_html_e('Test the email that admins receive when a speaker registers.', 'speakers-bureau'); ?></p>
                            <form method="post" action="<?php echo esc_url(admin_url('admin-post.php')); ?>">
                                <input type="hidden" name="action" value="sb_test_email">
                                <input type="hidden" name="email_type" value="admin">
                                <?php wp_nonce_field('sb_test_email'); ?>
                                <button type="submit" class="button button-secondary">
                                    <span class="dashicons dashicons-email" style="margin-right: 4px; font-size: 16px; vertical-align: middle;"></span>
                                    <?php esc_html_e('Send Test Email', 'speakers-bureau'); ?>
                                </button>
                            </form>
                            <p class="description" style="margin-top: 8px;">
                                <?php printf(esc_html__('Test email will be sent to: %s', 'speakers-bureau'), '<strong>' . esc_html(wp_get_current_user()->user_email) . '</strong>'); ?>
                            </p>
                        </div>
                    </div>
                <?php endif; ?>

                <?php if ($active_tab === 'general'): ?>
                    <hr style="margin:24px 0;">
                    <h2><?php esc_html_e('Maintenance', 'speakers-bureau'); ?></h2>

                    <div style="display:inline-block;margin-right:12px;">
                        <p class="description" style="max-width:720px;">
                            <?php esc_html_e('Rebuild speaker coordinates. Useful if you just enabled geo search or edited addresses.', 'speakers-bureau'); ?>
                        </p>
                        <button type="button" id="sb-rebuild-coordinates-btn" class="button button-secondary">
                            <?php esc_html_e('Rebuild Coordinates', 'speakers-bureau'); ?>
                        </button>
                        <div id="sb-geocode-progress" style="display:none; margin-top:15px;">
                            <div style="background:#f0f0f1; border:1px solid #c3c4c7; border-radius:3px; padding:3px; width:100%; max-width:500px;">
                                <div id="sb-geocode-progress-bar" style="background:#2271b1; height:24px; border-radius:2px; width:0%; transition: width 0.3s ease; display:flex; align-items:center; justify-content:center;">
                                    <span id="sb-geocode-progress-text" style="color:#fff; font-size:11px; font-weight:600;"></span>
                                </div>
                            </div>
                            <p id="sb-geocode-status" style="margin:8px 0 0 0; font-size:13px;"></p>
                        </div>
                    </div>

                    <form method="post" action="<?php echo esc_url(admin_url('admin-post.php')); ?>" style="display:inline-block;margin-left:12px;">
                        <?php wp_nonce_field('sb_create_pages'); ?>
                        <input type="hidden" name="action" value="sb_create_pages">
                        <p class="description" style="max-width:720px;">
                            <?php esc_html_e('Create/repair default pages (Speakers, Register Speaker, My Profile).', 'speakers-bureau'); ?>
                        </p>
                        <?php submit_button(__('Create Default Pages'), 'secondary', 'submit', false); ?>
                    </form>

                    <form method="post" action="<?php echo esc_url(admin_url('admin-post.php')); ?>" style="display:inline-block;margin-left:12px;">
                        <?php wp_nonce_field('sb_seed_samples'); ?>
                        <input type="hidden" name="action" value="sb_seed_samples">
                        <p class="description" style="max-width:720px;">
                            <?php esc_html_e('Create 5 sample speakers (only if 0 or 1 speakers exist).', 'speakers-bureau'); ?>
                        </p>
                        <?php submit_button(__('Create Sample Speakers'), 'secondary', 'submit', false); ?>
                    </form>

                    <form method="post" action="<?php echo esc_url(admin_url('admin-post.php')); ?>" style="display:inline-block;margin-left:12px;">
                        <?php wp_nonce_field('sb_refresh_topics_cache'); ?>
                        <input type="hidden" name="action" value="sb_refresh_topics_cache">
                        <p class="description" style="max-width:720px;">
                            <?php esc_html_e('Refresh the cached topic index data. Use this if topics are not showing correctly in the topics index block.', 'speakers-bureau'); ?>
                        </p>
                        <?php submit_button(__('Refresh Topics Cache'), 'secondary', 'submit', false); ?>
                    </form>
                <?php endif; ?>
            <?php endif; ?>
        </div>
        <?php
    }

    /* ================= Tab: Directory Options ================= */

    private function render_general_tab() {
        $opts = get_option($this->option_name_general, $this->default_general());
        if (!is_array($opts)) $opts = $this->default_general();
        ?>
        <table class="form-table" role="presentation">
            <tbody>
            <tr>
                <th scope="row"><?php esc_html_e('Enable geo/radius search', 'speakers-bureau'); ?></th>
                <td>
                    <label>
                        <input type="checkbox" name="<?php echo esc_attr($this->option_name_general); ?>[enable_geo]" value="1" <?php checked(!empty($opts['enable_geo'])); ?>>
                        <?php esc_html_e('Allow ZIP + Radius search and nearest sorting.', 'speakers-bureau'); ?>
                    </label>
                </td>
            </tr>
            <tr>
                <th scope="row"><?php esc_html_e('Default country code', 'speakers-bureau'); ?></th>
                <td>
                    <input type="text" name="<?php echo esc_attr($this->option_name_general); ?>[geo_default_country]" value="<?php echo esc_attr($opts['geo_default_country']); ?>" class="regular-text" style="max-width:90px;">
                    <p class="description"><?php esc_html_e('Two-letter ISO code (e.g., US, CA, GB).', 'speakers-bureau'); ?></p>
                </td>
            </tr>
            <tr>
                <th scope="row"><?php esc_html_e('Default radius (miles)', 'speakers-bureau'); ?></th>
                <td>
                    <input type="number" min="1" step="1" name="<?php echo esc_attr($this->option_name_general); ?>[geo_default_radius]" value="<?php echo esc_attr($opts['geo_default_radius']); ?>" class="small-text">
                </td>
            </tr>
            <tr>
                <th scope="row"><?php esc_html_e('Enable “Nearest” sort', 'speakers-bureau'); ?></th>
                <td>
                    <label>
                        <input type="checkbox" name="<?php echo esc_attr($this->option_name_general); ?>[enable_nearest_sort]" value="1" <?php checked(!empty($opts['enable_nearest_sort'])); ?>>
                        <?php esc_html_e('Show “Nearest” in sort options (requires ZIP + Radius).', 'speakers-bureau'); ?>
                    </label>
                </td>
            </tr>
            </tbody>
        </table>


        <?php
    }

    /* ================= Tab: Email Templates ================= */

    private function render_email_templates_tab() {
        $emails = get_option($this->option_name_email, []);
        if (!is_array($emails)) $emails = [];

        $speaker_subject = $emails['speaker_subject'] ?? 'Thank you for registering, {speaker_name}';
        $speaker_body    = $emails['speaker_body'] ?? "<p>Hi {speaker_name},</p><p>Thank you for registering with our Speakers Bureau.</p><p>You can <a href=\"{public_profile_url}\">view your public profile here</a><br>You can <a href=\"{profile_url}\">edit your profile here</a></p><p>Best regards,<br>{site_name}</p>";
        $admin_subject   = $emails['admin_subject'] ?? 'New Speaker Registration: {speaker_name}';
        $admin_body      = $emails['admin_body'] ?? "<p>A new speaker has registered:</p><p><strong>Name:</strong> {speaker_name}<br><strong>Profile:</strong> <a href=\"{profile_url}\">{profile_url}</a></p><p>Login to review.</p>";

        // New profile update email template
        $update_subject  = $emails['update_subject'] ?? 'Your speaker profile has been updated';
        $update_body     = $emails['update_body'] ?? "<p>Hi {speaker_name},</p><p>Your speaker profile has been updated on {site_name}.</p><h3>Your Current Profile:</h3>{speaker_listing}<p><a href=\"{my_profile_url}\">Login to view or edit your profile</a></p><p>Best regards,<br>{site_name}</p>";
        ?>
        <h2><?php esc_html_e('Email Templates', 'speakers-bureau'); ?></h2>
        <div style="background: #f0f8ff; padding: 12px; border-radius: 4px; margin-bottom: 20px;">
            <strong><?php esc_html_e('Available Shortcodes:', 'speakers-bureau'); ?></strong>
            <ul style="margin: 8px 0 0 20px;">
                <li><code>{speaker_name}</code> - <?php esc_html_e('Speaker name/title', 'speakers-bureau'); ?></li>
                <li><code>{public_profile_url}</code> - <?php esc_html_e('Public speaker profile page (no login required)', 'speakers-bureau'); ?></li>
                <li><code>{profile_url}</code> - <?php esc_html_e('Private profile edit page (requires login)', 'speakers-bureau'); ?></li>
                <li><code>{my_profile_url}</code> - <?php esc_html_e('My-profile page URL for login', 'speakers-bureau'); ?></li>
                <li><code>{speaker_listing}</code> - <?php esc_html_e('Complete speaker profile listing (HTML formatted)', 'speakers-bureau'); ?></li>
                <li><code>{site_name}</code> - <?php esc_html_e('Website name', 'speakers-bureau'); ?></li>
            </ul>
        </div>

        <table class="form-table">
            <tr>
                <th scope="row"><?php esc_html_e('Speaker Confirmation Subject', 'speakers-bureau'); ?></th>
                <td><input type="text" class="large-text" name="<?php echo esc_attr($this->option_name_email); ?>[speaker_subject]" value="<?php echo esc_attr($speaker_subject); ?>"></td>
            </tr>
            <tr>
                <th scope="row"><?php esc_html_e('Speaker Confirmation Body', 'speakers-bureau'); ?></th>
                <td>
                    <?php wp_editor($speaker_body, 'speaker_body', [
                        'textarea_name' => $this->option_name_email . '[speaker_body]',
                        'textarea_rows' => 10,
                        'media_buttons' => false,
                        'teeny' => false,
                        'quicktags' => true
                    ]); ?>
                    <p class="description"><?php esc_html_e('Use the "Text" tab to edit HTML code directly.', 'speakers-bureau'); ?></p>
                </td>
            </tr>
            <tr>
                <th scope="row"><?php esc_html_e('Admin Notification Subject', 'speakers-bureau'); ?></th>
                <td><input type="text" class="large-text" name="<?php echo esc_attr($this->option_name_email); ?>[admin_subject]" value="<?php echo esc_attr($admin_subject); ?>"></td>
            </tr>
            <tr>
                <th scope="row"><?php esc_html_e('Admin Notification Body', 'speakers-bureau'); ?></th>
                <td>
                    <?php wp_editor($admin_body, 'admin_body', [
                        'textarea_name' => $this->option_name_email . '[admin_body]',
                        'textarea_rows' => 10,
                        'media_buttons' => false,
                        'teeny' => false,
                        'quicktags' => true
                    ]); ?>
                    <p class="description"><?php esc_html_e('Use the "Text" tab to edit HTML code directly.', 'speakers-bureau'); ?></p>
                </td>
            </tr>
            <tr>
                <th scope="row"><?php esc_html_e('Profile Update Subject', 'speakers-bureau'); ?></th>
                <td><input type="text" class="large-text" name="<?php echo esc_attr($this->option_name_email); ?>[update_subject]" value="<?php echo esc_attr($update_subject); ?>"></td>
            </tr>
            <tr>
                <th scope="row"><?php esc_html_e('Profile Update Body', 'speakers-bureau'); ?></th>
                <td>
                    <?php wp_editor($update_body, 'update_body', [
                        'textarea_name' => $this->option_name_email . '[update_body]',
                        'textarea_rows' => 10,
                        'media_buttons' => false,
                        'teeny' => false,
                        'quicktags' => true
                    ]); ?>
                    <p class="description"><?php esc_html_e('Use the "Text" tab to edit HTML code directly.', 'speakers-bureau'); ?></p>
                </td>
            </tr>
        </table>
        <?php
    }

    /* ================= Tab: Profile Layout ================= */

    private function render_profile_layout_tab() {
        $layout = get_option($this->option_name_layout, $this->default_profile_layout());
        if (!is_array($layout)) $layout = $this->default_profile_layout();
        ?>
        <table class="widefat striped" style="max-width: 1100px;">
            <thead>
            <tr>
                <th style="width:70px;"><?php esc_html_e('Section', 'speakers-bureau'); ?></th>
                <th><?php esc_html_e('Title', 'speakers-bureau'); ?></th>
                <th style="width:120px;"><?php esc_html_e('Heading', 'speakers-bureau'); ?></th>
                <th style="width:140px;"><?php esc_html_e('Display Title', 'speakers-bureau'); ?></th>
                <th style="width:140px;"><?php esc_html_e('Hide Section', 'speakers-bureau'); ?></th>
                <th style="width:220px;"><?php esc_html_e('CSS Class', 'speakers-bureau'); ?></th>
            </tr>
            </thead>
            <tbody>
            <?php foreach (range(1,8) as $i):
                $row     = $layout[$i] ?? $this->default_profile_layout()[$i];
                $title   = $row['title'] ?? '';
                $heading = $row['heading'] ?? 'h3';
                $display = !empty($row['display']);
                $hidden  = !empty($row['hidden']);
                $class   = $row['class'] ?? 'sb-section-' . $i;
                ?>
                <tr>
                    <td><strong><?php echo esc_html($i); ?></strong></td>
                    <td><input type="text" name="<?php echo esc_attr($this->option_name_layout); ?>[<?php echo esc_attr($i); ?>][title]" value="<?php echo esc_attr($title); ?>" class="regular-text"></td>
                    <td>
                        <select name="<?php echo esc_attr($this->option_name_layout); ?>[<?php echo esc_attr($i); ?>][heading]">
                            <?php foreach (['h1','h2','h3','h4','h5','h6'] as $h): ?>
                                <option value="<?php echo esc_attr($h); ?>" <?php selected($heading, $h); ?>><?php echo esc_html(strtoupper($h)); ?></option>
                            <?php endforeach; ?>
                        </select>
                    </td>
                    <td>
                        <label><input type="checkbox" name="<?php echo esc_attr($this->option_name_layout); ?>[<?php echo esc_attr($i); ?>][display]" value="1" <?php checked($display, true); ?>> <?php esc_html_e('Show title', 'speakers-bureau'); ?></label>
                    </td>
                    <td>
                        <label><input type="checkbox" name="<?php echo esc_attr($this->option_name_layout); ?>[<?php echo esc_attr($i); ?>][hidden]" value="1" <?php checked($hidden, true); ?>> <?php esc_html_e('Hide section on profile page', 'speakers-bureau'); ?></label>
                    </td>
                    <td><input type="text" name="<?php echo esc_attr($this->option_name_layout); ?>[<?php echo esc_attr($i); ?>][class]" value="<?php echo esc_attr($class); ?>" class="regular-text"></td>
                </tr>
            <?php endforeach; ?>
            </tbody>
        </table>

        <p class="description" style="margin-top:8px;">
            <?php esc_html_e('Layout rules: Section 1 = left column; Sections 2–4 = right column; Sections 5–6 = optional; Section 7 = full width bottom; Section 8 = private (never public).', 'speakers-bureau'); ?>
        </p>
        <?php
    }

    /* ================= Tab: Display Settings ================= */

    private function render_display_tab() {
        $display = get_option($this->option_name_display, $this->default_display_settings());
        if (!is_array($display)) $display = $this->default_display_settings();

        // Merge with defaults to ensure all keys exist (for newly added settings)
        $display = array_merge($this->default_display_settings(), $display);

        // Get available form fields for card display options
        $form_fields = get_option($this->option_name_form, $this->default_form_fields());
        $available_fields = [];
        foreach ($form_fields as $field) {
            if (!empty($field['enabled']) && !in_array($field['type'] ?? '', ['image', 'edit_link', 'section_header'])) {
                $available_fields[$field['key']] = $field['label'] ?? $field['key'];
            }
        }
        ?>
        <h2><?php esc_html_e('Display Settings', 'speakers-bureau'); ?></h2>
        <p class="description"><?php esc_html_e('Customize how the speakers directory appears on your website.', 'speakers-bureau'); ?></p>

        <table class="form-table">
            <tr>
                <th scope="row"><?php esc_html_e('Grid Columns', 'speakers-bureau'); ?></th>
                <td>
                    <select name="<?php echo esc_attr($this->option_name_display); ?>[columns]">
                        <?php for ($i = 1; $i <= 6; $i++): ?>
                            <option value="<?php echo $i; ?>" <?php selected($display['columns'], $i); ?>><?php echo $i; ?></option>
                        <?php endfor; ?>
                    </select>
                    <p class="description"><?php esc_html_e('Number of speaker cards per row.', 'speakers-bureau'); ?></p>
                </td>
            </tr>
            <tr>
                <th scope="row"><?php esc_html_e('Speakers Per Page', 'speakers-bureau'); ?></th>
                <td>
                    <input type="number" min="1" max="100" name="<?php echo esc_attr($this->option_name_display); ?>[per_page]" value="<?php echo esc_attr($display['per_page']); ?>" class="small-text">
                    <p class="description"><?php esc_html_e('Number of speakers to show per page.', 'speakers-bureau'); ?></p>
                </td>
            </tr>
            <tr>
                <th scope="row"><?php esc_html_e('Card Fields', 'speakers-bureau'); ?></th>
                <td>
                    <fieldset>
                        <legend class="screen-reader-text"><?php esc_html_e('Select fields to display on speaker cards', 'speakers-bureau'); ?></legend>
                        <?php foreach ($available_fields as $key => $label): ?>
                            <label style="display: block; margin: 4px 0;">
                                <input type="checkbox" name="<?php echo esc_attr($this->option_name_display); ?>[card_fields][]" value="<?php echo esc_attr($key); ?>" <?php checked(in_array($key, $display['card_fields'])); ?>>
                                <?php echo esc_html($label); ?>
                            </label>
                        <?php endforeach; ?>
                        <p class="description"><?php esc_html_e('Choose which fields to display below the speaker image on directory cards.', 'speakers-bureau'); ?></p>
                    </fieldset>
                </td>
            </tr>
            <tr>
                <th scope="row"><?php esc_html_e('Max Words Per Field', 'speakers-bureau'); ?></th>
                <td>
                    <input type="number" min="5" max="100" name="<?php echo esc_attr($this->option_name_display); ?>[max_words]" value="<?php echo esc_attr($display['max_words']); ?>" class="small-text">
                    <p class="description"><?php esc_html_e('Maximum number of words to show for each field on cards.', 'speakers-bureau'); ?></p>
                </td>
            </tr>
            <tr>
                <th scope="row"><?php esc_html_e('Card Style', 'speakers-bureau'); ?></th>
                <td>
                    <select name="<?php echo esc_attr($this->option_name_display); ?>[card_style]">
                        <option value="default" <?php selected($display['card_style'], 'default'); ?>><?php esc_html_e('Default', 'speakers-bureau'); ?></option>
                        <option value="compact" <?php selected($display['card_style'], 'compact'); ?>><?php esc_html_e('Compact', 'speakers-bureau'); ?></option>
                        <option value="detailed" <?php selected($display['card_style'], 'detailed'); ?>><?php esc_html_e('Detailed', 'speakers-bureau'); ?></option>
                    </select>
                    <p class="description"><?php esc_html_e('Choose the overall style for speaker cards.', 'speakers-bureau'); ?></p>
                </td>
            </tr>
            <tr>
                <th scope="row"><?php esc_html_e('Image Style', 'speakers-bureau'); ?></th>
                <td>
                    <select name="<?php echo esc_attr($this->option_name_display); ?>[image_style]">
                        <option value="square" <?php selected($display['image_style'], 'square'); ?>><?php esc_html_e('Square', 'speakers-bureau'); ?></option>
                        <option value="circle" <?php selected($display['image_style'], 'circle'); ?>><?php esc_html_e('Circle', 'speakers-bureau'); ?></option>
                        <option value="rounded" <?php selected($display['image_style'], 'rounded'); ?>><?php esc_html_e('Rounded', 'speakers-bureau'); ?></option>
                    </select>
                    <p class="description"><?php esc_html_e('Shape of speaker images on cards.', 'speakers-bureau'); ?></p>
                </td>
            </tr>
            <tr>
                <th scope="row"><?php esc_html_e('Show Excerpt', 'speakers-bureau'); ?></th>
                <td>
                    <label>
                        <input type="checkbox" name="<?php echo esc_attr($this->option_name_display); ?>[show_excerpt]" value="1" <?php checked($display['show_excerpt']); ?>>
                        <?php esc_html_e('Show speaker bio excerpt on cards', 'speakers-bureau'); ?>
                    </label>
                    <p class="description"><?php esc_html_e('When enabled, shows a brief bio excerpt on each speaker card.', 'speakers-bureau'); ?></p>
                </td>
            </tr>
            <tr>
                <th scope="row"><?php esc_html_e('Excerpt Length', 'speakers-bureau'); ?></th>
                <td>
                    <input type="number" min="5" max="50" name="<?php echo esc_attr($this->option_name_display); ?>[excerpt_length]" value="<?php echo esc_attr($display['excerpt_length']); ?>" class="small-text">
                    <p class="description"><?php esc_html_e('Number of words in bio excerpt.', 'speakers-bureau'); ?></p>
                </td>
            </tr>
            <tr>
                <th scope="row"><?php esc_html_e('Primary Color', 'speakers-bureau'); ?></th>
                <td>
                    <input type="color" name="<?php echo esc_attr($this->option_name_display); ?>[primary_color]" value="<?php echo esc_attr($display['primary_color']); ?>">
                    <p class="description"><?php esc_html_e('Primary color for links and accents.', 'speakers-bureau'); ?></p>
                </td>
            </tr>
            <tr>
                <th scope="row"><?php esc_html_e('Card Border Radius', 'speakers-bureau'); ?></th>
                <td>
                    <input type="range" min="0" max="20" name="<?php echo esc_attr($this->option_name_display); ?>[card_border_radius]" value="<?php echo esc_attr($display['card_border_radius']); ?>" oninput="this.nextElementSibling.value = this.value + 'px'">
                    <output><?php echo esc_html($display['card_border_radius']); ?>px</output>
                    <p class="description"><?php esc_html_e('Roundness of card corners (0 = square, 20 = very rounded).', 'speakers-bureau'); ?></p>
                </td>
            </tr>
            <tr>
                <th scope="row"><?php esc_html_e('Card Shadow', 'speakers-bureau'); ?></th>
                <td>
                    <select name="<?php echo esc_attr($this->option_name_display); ?>[card_shadow]">
                        <option value="none" <?php selected($display['card_shadow'], 'none'); ?>><?php esc_html_e('None', 'speakers-bureau'); ?></option>
                        <option value="light" <?php selected($display['card_shadow'], 'light'); ?>><?php esc_html_e('Light', 'speakers-bureau'); ?></option>
                        <option value="medium" <?php selected($display['card_shadow'], 'medium'); ?>><?php esc_html_e('Medium', 'speakers-bureau'); ?></option>
                        <option value="strong" <?php selected($display['card_shadow'], 'strong'); ?>><?php esc_html_e('Strong', 'speakers-bureau'); ?></option>
                    </select>
                    <p class="description"><?php esc_html_e('Drop shadow effect for speaker cards.', 'speakers-bureau'); ?></p>
                </td>
            </tr>
            <tr>
                <th scope="row"><?php esc_html_e('Date Display', 'speakers-bureau'); ?></th>
                <td>
                    <fieldset>
                        <legend class="screen-reader-text"><?php esc_html_e('Date display options', 'speakers-bureau'); ?></legend>
                        <label style="display: block; margin: 8px 0;">
                            <input type="checkbox" name="<?php echo esc_attr($this->option_name_display); ?>[show_date_created]" value="1" <?php checked($display['show_date_created']); ?>>
                            <?php esc_html_e('Show date added in bottom right corner', 'speakers-bureau'); ?>
                        </label>
                        <label style="display: block; margin: 8px 0;">
                            <input type="checkbox" name="<?php echo esc_attr($this->option_name_display); ?>[show_date_updated]" value="1" <?php checked($display['show_date_updated']); ?>>
                            <?php esc_html_e('Show last updated date in bottom right corner', 'speakers-bureau'); ?>
                        </label>
                        <p class="description"><?php esc_html_e('Display creation or last updated date on speaker cards. Last updated takes priority if both are enabled.', 'speakers-bureau'); ?></p>
                    </fieldset>
                </td>
            </tr>
            <tr>
                <th scope="row"><?php esc_html_e('Date Format', 'speakers-bureau'); ?></th>
                <td>
                    <input type="text" name="<?php echo esc_attr($this->option_name_display); ?>[date_format]" value="<?php echo esc_attr($display['date_format']); ?>" class="regular-text">
                    <p class="description">
                        <?php
                        echo sprintf(
                            __('PHP date format for timestamps. Preview: %s', 'speakers-bureau'),
                            '<strong>' . esc_html(date($display['date_format'])) . '</strong>'
                        );
                        ?>
                        <br>
                        <?php esc_html_e('Common formats: m/d/Y g:ia (09/23/2025 9:53am), Y-m-d H:i (2025-09-23 21:53), M j, Y (Sep 23, 2025)', 'speakers-bureau'); ?>
                    </p>
                </td>
            </tr>
            <tr>
                <th scope="row"><?php esc_html_e('Auto-Publish Profiles', 'speakers-bureau'); ?></th>
                <td>
                    <label>
                        <input type="checkbox" name="<?php echo esc_attr($this->option_name_display); ?>[auto_publish_profiles]" value="1" <?php checked($display['auto_publish_profiles']); ?>>
                        <?php esc_html_e('Automatically publish speaker profiles when users complete registration with all required fields', 'speakers-bureau'); ?>
                    </label>
                    <p class="description">
                        <?php esc_html_e('When enabled, new speaker profiles will be published immediately if all required fields are completed. When disabled, profiles remain as drafts requiring admin approval.', 'speakers-bureau'); ?>
                    </p>
                </td>
            </tr>
        </table>
        <?php
    }

    /* ================= Maintenance ================= */

    private function geocode_zip_cached($zip, $country = 'US') {
        $zip = trim((string)$zip);
        $country = strtoupper(trim((string)$country));
        if ($zip === '') return [null, null];

        $cache = get_option('sb_geo_cache', []);
        if (!is_array($cache)) $cache = [];
        $key = $country . '|' . $zip;

        if (isset($cache[$key]['lat'], $cache[$key]['lng'])) {
            return [$cache[$key]['lat'], $cache[$key]['lng']];
        }

        $url = add_query_arg([
            'format'       => 'json',
            'postalcode'   => $zip,
            'countrycodes' => strtolower($country),
            'limit'        => 1
        ], 'https://nominatim.openstreetmap.org/search');

        $resp = wp_remote_get($url, [
            'timeout' => 10,
            'headers' => ['User-Agent' => 'Speakers-Bureau-Plugin/2.0 (+wordpress)']
        ]);
        if (!is_wp_error($resp) && wp_remote_retrieve_response_code($resp) === 200) {
            $body = json_decode(wp_remote_retrieve_body($resp), true);
            if (is_array($body) && !empty($body[0]['lat']) && !empty($body[0]['lon'])) {
                $lat = (float)$body[0]['lat'];
                $lng = (float)$body[0]['lon'];
                $cache[$key] = ['lat'=>$lat,'lng'=>$lng,'t'=>time()];
                if (count($cache) > 5000) $cache = array_slice($cache, -4000, null, true);
                update_option('sb_geo_cache', $cache, false);
                return [$lat, $lng];
            }
        }
        return [null, null];
    }

    public function handle_rebuild_geo() {
        if (!current_user_can('manage_options')) wp_die('Forbidden');
        check_admin_referer('sb_rebuild_geo');

        $updated = 0; $skipped = 0; $errors = 0;

        // Get default country from settings
        $settings = get_option('sb_settings', []);
        $default_country = isset($settings['geo_default_country']) ? $settings['geo_default_country'] : 'US';

        // Get all speakers
        $speakers = get_posts([
            'post_type' => 'speaker',
            'post_status' => ['publish', 'private', 'draft'],
            'posts_per_page' => -1,
            'fields' => 'ids'
        ]);

        foreach ($speakers as $speaker_id) {
            // Get address components - check multiple possible field names (same logic as save hook)
            // Street address (optional, may not exist)
            $address = get_post_meta($speaker_id, 'address', true);
            if (empty($address)) $address = get_post_meta($speaker_id, 'sb_address', true);

            // City - check multiple possible field names
            $city = get_post_meta($speaker_id, 'city', true);
            if (empty($city)) $city = get_post_meta($speaker_id, 'sb_city', true);

            // State - check multiple possible field names
            $state = get_post_meta($speaker_id, 'state', true);
            if (empty($state)) $state = get_post_meta($speaker_id, 'sb_state', true);

            // Zip - check multiple possible field names
            $zip = get_post_meta($speaker_id, 'zip', true);
            if (empty($zip)) $zip = get_post_meta($speaker_id, 'zip_code', true);
            if (empty($zip)) $zip = get_post_meta($speaker_id, 'sb_zip', true);
            if (empty($zip)) $zip = get_post_meta($speaker_id, 'postal_code', true);

            // Try to get country from various possible field names
            $country = get_post_meta($speaker_id, 'country', true);
            if (empty($country)) $country = get_post_meta($speaker_id, 'sb_country', true);
            if (empty($country)) {
                $country = $default_country;
            }

            // Build address string (do NOT include country in address - only use as filter parameter)
            $parts = array_filter([$address, $city, $state, $zip]);
            if (empty($parts)) {
                $skipped++;
                continue;
            }
            $full_address = implode(', ', $parts);

            // Use the same geocoding function as the save hook
            $coords = sb_geocode_address($full_address, $country);
            if ($coords && isset($coords['lat']) && isset($coords['lng'])) {
                update_post_meta($speaker_id, 'geo_lat', $coords['lat']);
                update_post_meta($speaker_id, 'geo_lng', $coords['lng']);
                // Also update old format for backward compatibility
                update_post_meta($speaker_id, '_sb_lat', $coords['lat']);
                update_post_meta($speaker_id, '_sb_lng', $coords['lng']);
                $updated++;
            } else {
                $errors++;
            }
        }

        wp_safe_redirect(add_query_arg([
            'page'      => 'sb-settings',
            'tab'       => 'general',
            'sb_notice' => rawurlencode(sprintf(__('Rebuild complete: %d updated, %d skipped, %d errors.', 'speakers-bureau'), $updated, $skipped, $errors)),
        ], admin_url('admin.php')));
        exit;
    }

    public function handle_create_pages() {
        if (!current_user_can('manage_options')) wp_die('Forbidden');
        check_admin_referer('sb_create_pages');

        $created = 0; $existing = 0;

        $pages = [
            ['title' => 'Browse Speakers', 'slug' => 'speakers',         'content' => '[speaker_list]'],
            ['title' => 'Register Speaker', 'slug' => 'register-speaker','content' => '[speaker_register]'],
            ['title' => 'My Profile',       'slug' => 'my-profile',      'content' => '[speaker_edit]'],
        ];

        foreach ($pages as $p) {
            $page = get_page_by_path($p['slug']);
            if ($page && $page->post_type === 'page') { $existing++; continue; }
            $id = wp_insert_post([
                'post_title'   => $p['title'],
                'post_name'    => $p['slug'],
                'post_content' => $p['content'],
                'post_status'  => 'publish',
                'post_type'    => 'page',
            ]);
            if (!is_wp_error($id)) $created++;
        }

        wp_safe_redirect(add_query_arg([
            'page'      => 'sb-settings',
            'tab'       => 'general',
            'sb_notice' => rawurlencode(sprintf(__('Pages created: %d (existing: %d).', 'speakers-bureau'), $created, $existing)),
        ], admin_url('admin.php')));
        exit;
    }

    public function handle_test_email() {
        if (!current_user_can('manage_options')) wp_die('Forbidden');
        check_admin_referer('sb_test_email');

        $email_type = sanitize_key($_POST['email_type'] ?? '');
        $user = wp_get_current_user();
        $test_email = $user->user_email;

        if (!$test_email || !is_email($test_email)) {
            wp_safe_redirect(add_query_arg([
                'page'      => 'sb-settings',
                'tab'       => 'email_templates',
                'sb_notice' => rawurlencode(__('Error: No valid email address found for current user.', 'speakers-bureau')),
            ], admin_url('admin.php')));
            exit;
        }

        // Find a published speaker post to use for testing
        $test_speaker = get_posts([
            'post_type' => 'speaker',
            'post_status' => 'publish',
            'posts_per_page' => 1,
            'orderby' => 'date',
            'order' => 'DESC'
        ]);

        $test_post_id = $test_speaker ? $test_speaker[0]->ID : 0;
        $test_speaker_name = $test_speaker ? get_the_title($test_speaker[0]) : 'Test Speaker';

        $success = false;
        $email_sent_to = '';

        try {
            if ($email_type === 'speaker') {
                if ($test_post_id && function_exists('sb_send_speaker_confirmation_email')) {
                    $success = sb_send_speaker_confirmation_email($test_post_id, $test_email, $test_speaker_name);
                    $email_sent_to = 'Speaker Confirmation';
                }
            } elseif ($email_type === 'admin') {
                if ($test_post_id && function_exists('sb_send_admin_notification_email')) {
                    $success = sb_send_admin_notification_email($test_post_id, $test_speaker_name);
                    $email_sent_to = 'Admin Notification';
                }
            }
        } catch (Exception $e) {
            $success = false;
        }

        $message = $success
            ? sprintf(__('Test email (%s) sent successfully to %s', 'speakers-bureau'), $email_sent_to, $test_email)
            : __('Failed to send test email. Please check your email configuration.', 'speakers-bureau');

        wp_safe_redirect(add_query_arg([
            'page'      => 'sb-settings',
            'tab'       => 'email_templates',
            'sb_notice' => rawurlencode($message),
        ], admin_url('admin.php')));
        exit;
    }

    private function render_docs_tab() {
        $current_view = isset($_GET['view']) ? sanitize_key($_GET['view']) : 'overview';

        $docs = [
            'overview' => ['title' => 'Plugin Overview', 'file' => 'README.md'],
            'install' => ['title' => 'Installation Guide', 'file' => 'INSTALLATION.md'],
            'user' => ['title' => 'User Guide', 'file' => 'USER-GUIDE.md'],
            'admin' => ['title' => 'Administrator Guide', 'file' => 'ADMIN-GUIDE.md'],
            'speaker' => ['title' => 'Speaker Walkthrough', 'file' => 'SPEAKER-WALKTHROUGH.md'],
        ];

        echo '<div class="sb-docs-navigation" style="margin-bottom: 20px;">';
        echo '<h3>Documentation</h3>';
        echo '<p class="description">Select a guide to view detailed information about plugin features and usage.</p>';

        foreach ($docs as $view => $info) {
            $url = add_query_arg(['tab' => 'docs', 'view' => $view], admin_url('admin.php?page=sb-settings'));
            $class = ($current_view === $view) ? 'button-primary' : 'button-secondary';
            echo '<a class="button ' . $class . '" href="' . esc_url($url) . '" style="margin-right: 8px; margin-bottom: 8px;">' . esc_html($info['title']) . '</a>';
        }
        echo '</div>';

        if (isset($docs[$current_view])) {
            $this->render_markdown_content($docs[$current_view]['file']);
        } else {
            echo '<p>' . esc_html__('Documentation not found.', 'speakers-bureau') . '</p>';
        }
    }

    private function render_markdown_content($filename) {
        $filepath = plugin_dir_path(dirname(__FILE__)) . $filename;

        if (!file_exists($filepath)) {
            echo '<div class="notice notice-error"><p>' . sprintf(esc_html__('Documentation file %s not found.', 'speakers-bureau'), esc_html($filename)) . '</p></div>';
            return;
        }

        $content = file_get_contents($filepath);
        if ($content === false) {
            echo '<div class="notice notice-error"><p>' . esc_html__('Could not read documentation file.', 'speakers-bureau') . '</p></div>';
            return;
        }

        // Convert markdown to basic HTML
        $html_content = $this->simple_markdown_to_html($content);

        echo '<div class="sb-docs-content" style="background: #fff; padding: 24px; border: 1px solid #ccd0d4; border-radius: 4px; max-height: 70vh; overflow-y: auto;">';
        echo $html_content;
        echo '</div>';
    }

    private function simple_markdown_to_html($markdown) {
        // Basic markdown conversion - headers, lists, links, code blocks, bold, italic
        $html = $markdown;

        // Code blocks
        $html = preg_replace('/```([a-z]*)\n(.*?)\n```/s', '<pre><code class="language-$1">$2</code></pre>', $html);

        // Inline code
        $html = preg_replace('/`([^`]+)`/', '<code>$1</code>', $html);

        // Headers
        $html = preg_replace('/^### (.+)$/m', '<h3>$1</h3>', $html);
        $html = preg_replace('/^## (.+)$/m', '<h2>$1</h2>', $html);
        $html = preg_replace('/^# (.+)$/m', '<h1>$1</h1>', $html);

        // Bold and italic
        $html = preg_replace('/\*\*(.+?)\*\*/', '<strong>$1</strong>', $html);
        $html = preg_replace('/\*(.+?)\*/', '<em>$1</em>', $html);

        // Links
        $html = preg_replace('/\[(.+?)\]\((.+?)\)/', '<a href="$2" target="_blank">$1</a>', $html);

        // Lists
        $html = preg_replace('/^- (.+)$/m', '<li>$1</li>', $html);
        $html = preg_replace('/^(\d+)\. (.+)$/m', '<li>$2</li>', $html);

        // Wrap consecutive <li> elements in <ul>
        $html = preg_replace('/(<li>.*<\/li>)/s', '<ul>$1</ul>', $html);
        $html = preg_replace('/<\/ul>\s*<ul>/', '', $html);

        // Checkboxes
        $html = preg_replace('/- ☑️/', '<li style="list-style: none;">✅', $html);
        $html = preg_replace('/- ✅/', '<li style="list-style: none;">✅', $html);

        // Paragraphs
        $html = preg_replace('/\n\n/', '</p><p>', $html);
        $html = '<p>' . $html . '</p>';
        $html = preg_replace('/<p><\/p>/', '', $html);
        $html = preg_replace('/<p>(<h[1-6]>)/i', '$1', $html);
        $html = preg_replace('/(<\/h[1-6]>)<\/p>/i', '$1', $html);
        $html = preg_replace('/<p>(<ul>)/i', '$1', $html);
        $html = preg_replace('/(<\/ul>)<\/p>/i', '$1', $html);
        $html = preg_replace('/<p>(<pre>)/i', '$1', $html);
        $html = preg_replace('/(<\/pre>)<\/p>/i', '$1', $html);

        // Add some basic styling
        $html = str_replace('<h1>', '<h1 style="border-bottom: 2px solid #0073aa; padding-bottom: 8px; margin-top: 24px;">', $html);
        $html = str_replace('<h2>', '<h2 style="color: #0073aa; margin-top: 24px; margin-bottom: 12px;">', $html);
        $html = str_replace('<h3>', '<h3 style="margin-top: 20px; margin-bottom: 10px;">', $html);
        $html = str_replace('<code>', '<code style="background: #f1f1f1; padding: 2px 4px; border-radius: 3px; font-family: monospace;">', $html);
        $html = str_replace('<pre>', '<pre style="background: #f8f8f8; padding: 12px; border-radius: 4px; overflow-x: auto; border-left: 4px solid #0073aa;">', $html);
        $html = str_replace('<ul>', '<ul style="margin: 12px 0; padding-left: 24px;">', $html);
        $html = str_replace('<li>', '<li style="margin: 4px 0;">', $html);

        return $html;
    }

    /* ================= Admin Messages ================= */

    private function render_admin_messages($message) {
        if (!$message) return;

        $class = 'notice';
        $text = '';

        switch ($message) {
            case 'import_success':
                $class = 'notice notice-success is-dismissible';
                $imported = urldecode($_GET['imported'] ?? '');
                $text = sprintf(__('Settings imported successfully: %s', 'speakers-bureau'), $imported);
                break;
            case 'no_data_selected':
                $class = 'notice notice-warning is-dismissible';
                $text = __('Please select at least one type of data to export.', 'speakers-bureau');
                break;
            case 'upload_error':
                $class = 'notice notice-error is-dismissible';
                $text = __('File upload failed. Please try again.', 'speakers-bureau');
                break;
            case 'invalid_json':
                $class = 'notice notice-error is-dismissible';
                $text = __('Invalid JSON file. Please upload a valid Speakers Bureau export file.', 'speakers-bureau');
                break;
            case 'no_valid_data':
                $class = 'notice notice-warning is-dismissible';
                $text = __('No valid configuration data found in the uploaded file.', 'speakers-bureau');
                break;
        }

        if ($text) {
            echo '<div class="' . esc_attr($class) . '"><p>' . esc_html($text) . '</p></div>';
        }
    }

    /* ================= Export/Import Handlers ================= */

    public function handle_export_settings() {
        if (!current_user_can('manage_options')) {
            wp_die(__('Access denied.', 'speakers-bureau'));
        }

        if (!wp_verify_nonce($_POST['_wpnonce'], 'sb_export_settings')) {
            wp_die(__('Security check failed.', 'speakers-bureau'));
        }

        $export_data = [];

        // Export Form Builder Fields
        if (!empty($_POST['export_form_fields'])) {
            $export_data['form_fields'] = get_option($this->option_name_form, []);
        }

        // Export Profile Layout
        if (!empty($_POST['export_profile_layout'])) {
            $export_data['profile_layout'] = get_option($this->option_name_layout, []);
        }

        // Export Email Templates
        if (!empty($_POST['export_email_templates'])) {
            $export_data['email_templates'] = get_option($this->option_name_email, []);
        }

        // Export Display Settings
        if (!empty($_POST['export_display_settings'])) {
            $export_data['display_settings'] = get_option($this->option_name_display, []);
        }

        if (empty($export_data)) {
            wp_redirect(add_query_arg([
                'page' => 'sb-settings',
                'tab' => 'general',
                'message' => 'no_data_selected'
            ], admin_url('admin.php')));
            exit;
        }

        // Add metadata
        $export_data['_export_info'] = [
            'plugin_version' => defined('SB_PLUGIN_VER') ? SB_PLUGIN_VER : '3.3.0',
            'export_date' => current_time('mysql'),
            'site_url' => home_url(),
            'wp_version' => get_bloginfo('version')
        ];

        // Generate filename
        $timestamp = current_time('Y-m-d_H-i-s');
        $filename = "speakers-bureau-settings_{$timestamp}.json";

        // Set headers for download
        header('Content-Type: application/json');
        header('Content-Disposition: attachment; filename="' . $filename . '"');
        header('Content-Length: ' . strlen(json_encode($export_data, JSON_PRETTY_PRINT)));

        // Output JSON
        echo json_encode($export_data, JSON_PRETTY_PRINT);
        exit;
    }

    public function handle_import_settings() {
        if (!current_user_can('manage_options')) {
            wp_die(__('Access denied.', 'speakers-bureau'));
        }

        if (!wp_verify_nonce($_POST['_wpnonce'], 'sb_import_settings')) {
            wp_die(__('Security check failed.', 'speakers-bureau'));
        }

        $redirect_args = [
            'page' => 'sb-settings',
            'tab' => 'general'
        ];

        // Check if file was uploaded
        if (empty($_FILES['import_file']) || $_FILES['import_file']['error'] !== UPLOAD_ERR_OK) {
            $redirect_args['message'] = 'upload_error';
            wp_redirect(add_query_arg($redirect_args, admin_url('admin.php')));
            exit;
        }

        // Read and validate JSON
        $file_content = file_get_contents($_FILES['import_file']['tmp_name']);
        $import_data = json_decode($file_content, true);

        if (json_last_error() !== JSON_ERROR_NONE || !is_array($import_data)) {
            $redirect_args['message'] = 'invalid_json';
            wp_redirect(add_query_arg($redirect_args, admin_url('admin.php')));
            exit;
        }

        $import_mode = $_POST['import_mode'] ?? 'merge';
        $imported_items = [];

        // Import Form Builder Fields
        if (isset($import_data['form_fields']) && is_array($import_data['form_fields'])) {
            if ($import_mode === 'replace') {
                update_option($this->option_name_form, $import_data['form_fields']);
            } else {
                // Merge mode
                $existing = get_option($this->option_name_form, []);
                $merged = array_merge($existing, $import_data['form_fields']);
                update_option($this->option_name_form, $merged);
            }
            $imported_items[] = __('Form Builder Fields', 'speakers-bureau');
        }

        // Import Profile Layout
        if (isset($import_data['profile_layout']) && is_array($import_data['profile_layout'])) {
            if ($import_mode === 'replace') {
                update_option($this->option_name_layout, $import_data['profile_layout']);
            } else {
                // Merge mode
                $existing = get_option($this->option_name_layout, []);
                $merged = array_merge($existing, $import_data['profile_layout']);
                update_option($this->option_name_layout, $merged);
            }
            $imported_items[] = __('Profile Layout', 'speakers-bureau');
        }

        // Import Email Templates
        if (isset($import_data['email_templates']) && is_array($import_data['email_templates'])) {
            if ($import_mode === 'replace') {
                update_option($this->option_name_email, $import_data['email_templates']);
            } else {
                // Merge mode
                $existing = get_option($this->option_name_email, []);
                $merged = array_merge($existing, $import_data['email_templates']);
                update_option($this->option_name_email, $merged);
            }
            $imported_items[] = __('Email Templates', 'speakers-bureau');
        }

        // Import Display Settings
        if (isset($import_data['display_settings']) && is_array($import_data['display_settings'])) {
            if ($import_mode === 'replace') {
                update_option($this->option_name_display, $import_data['display_settings']);
            } else {
                // Merge mode
                $existing = get_option($this->option_name_display, []);
                $merged = array_merge($existing, $import_data['display_settings']);
                update_option($this->option_name_display, $merged);
            }
            $imported_items[] = __('Display Settings', 'speakers-bureau');
        }

        if (empty($imported_items)) {
            $redirect_args['message'] = 'no_valid_data';
        } else {
            $redirect_args['message'] = 'import_success';
            $redirect_args['imported'] = urlencode(implode(', ', $imported_items));
        }

        wp_redirect(add_query_arg($redirect_args, admin_url('admin.php')));
        exit;
    }

    /**
     * AJAX handler: Get count of speakers that need geocoding
     */
    public function ajax_get_geocode_count() {
        check_ajax_referer('sb_geocode_nonce', 'nonce');

        if (!current_user_can('manage_options')) {
            wp_send_json_error(['message' => 'Insufficient permissions']);
        }

        $speakers = get_posts([
            'post_type' => 'speaker',
            'post_status' => ['publish', 'private', 'draft'],
            'posts_per_page' => -1,
            'fields' => 'ids'
        ]);

        wp_send_json_success([
            'total' => count($speakers),
            'speaker_ids' => $speakers
        ]);
    }

    /**
     * AJAX handler: Geocode a batch of speakers
     */
    public function ajax_geocode_batch() {
        check_ajax_referer('sb_geocode_nonce', 'nonce');

        if (!current_user_can('manage_options')) {
            wp_send_json_error(['message' => 'Insufficient permissions']);
        }

        $speaker_ids = isset($_POST['speaker_ids']) ? array_map('intval', $_POST['speaker_ids']) : [];

        if (empty($speaker_ids)) {
            wp_send_json_error(['message' => 'No speaker IDs provided']);
        }

        // Get default country from settings
        $settings = get_option('sb_settings', []);
        $default_country = isset($settings['geo_default_country']) ? $settings['geo_default_country'] : 'US';

        $results = [];
        $updated = 0;
        $skipped = 0;
        $errors = 0;

        foreach ($speaker_ids as $speaker_id) {
            // Get address components - check multiple possible field names
            $address = get_post_meta($speaker_id, 'address', true);
            if (empty($address)) $address = get_post_meta($speaker_id, 'sb_address', true);

            $city = get_post_meta($speaker_id, 'city', true);
            if (empty($city)) $city = get_post_meta($speaker_id, 'sb_city', true);

            $state = get_post_meta($speaker_id, 'state', true);
            if (empty($state)) $state = get_post_meta($speaker_id, 'sb_state', true);

            $zip = get_post_meta($speaker_id, 'zip', true);
            if (empty($zip)) $zip = get_post_meta($speaker_id, 'zip_code', true);
            if (empty($zip)) $zip = get_post_meta($speaker_id, 'sb_zip', true);
            if (empty($zip)) $zip = get_post_meta($speaker_id, 'postal_code', true);

            $country = get_post_meta($speaker_id, 'country', true);
            if (empty($country)) $country = get_post_meta($speaker_id, 'sb_country', true);
            if (empty($country)) {
                $country = $default_country;
            }

            // Build address string
            $parts = array_filter([$address, $city, $state, $zip]);
            if (empty($parts)) {
                $skipped++;
                $results[] = [
                    'id' => $speaker_id,
                    'status' => 'skipped',
                    'message' => 'No address data'
                ];
                continue;
            }

            $full_address = implode(', ', $parts);

            // Use the geocoding function
            $coords = sb_geocode_address($full_address, $country);
            if ($coords && isset($coords['lat']) && isset($coords['lng'])) {
                update_post_meta($speaker_id, 'geo_lat', $coords['lat']);
                update_post_meta($speaker_id, 'geo_lng', $coords['lng']);
                update_post_meta($speaker_id, '_sb_lat', $coords['lat']);
                update_post_meta($speaker_id, '_sb_lng', $coords['lng']);
                $updated++;
                $results[] = [
                    'id' => $speaker_id,
                    'status' => 'updated',
                    'address' => $full_address,
                    'coords' => $coords
                ];
            } else {
                $errors++;
                $results[] = [
                    'id' => $speaker_id,
                    'status' => 'error',
                    'address' => $full_address,
                    'message' => 'Geocoding failed'
                ];
            }

            // Small delay to respect API rate limits (but much shorter than 1 second)
            usleep(250000); // 0.25 seconds = 4 requests per second
        }

        wp_send_json_success([
            'updated' => $updated,
            'skipped' => $skipped,
            'errors' => $errors,
            'results' => $results
        ]);
    }

}

if (is_admin()) {
    new SB_Settings_Page();
}

/**
 * Process email template with shortcode replacements
 */
function sb_process_email_template($template, $speaker_post_id, $speaker_name = '') {
    if (empty($speaker_name)) {
        $speaker_name = get_the_title($speaker_post_id);
    }

    // Get URLs
    $public_profile_url = get_permalink($speaker_post_id);

    // For profile_url, create URL with login redirect if user is not logged in
    $my_profile_page = get_page_by_path('my-profile');
    $profile_base_url = $my_profile_page ? get_permalink($my_profile_page) : home_url('/my-profile/');
    $profile_edit_url = add_query_arg('speaker_id', $speaker_post_id, $profile_base_url);

    // Add login redirect for profile URL
    $login_redirect_url = wp_login_url($profile_edit_url);

    // My-profile page URL (for login)
    $my_profile_url = $my_profile_page ? get_permalink($my_profile_page) : home_url('/my-profile/');

    // Generate speaker listing HTML
    $speaker_listing = sb_generate_speaker_listing_html($speaker_post_id);

    // Replace shortcodes
    $replacements = [
        '{speaker_name}' => $speaker_name,
        '{public_profile_url}' => $public_profile_url,
        '{profile_url}' => $login_redirect_url,
        '{my_profile_url}' => $my_profile_url,
        '{speaker_listing}' => $speaker_listing,
        '{site_name}' => get_bloginfo('name'),
    ];

    return str_replace(array_keys($replacements), array_values($replacements), $template);
}

/**
 * Send speaker confirmation email using template
 */
function sb_send_speaker_confirmation_email($speaker_post_id, $recipient_email, $speaker_name = '') {
    $emails = get_option('sb_email_templates', []);
    if (!is_array($emails)) $emails = [];

    $subject_template = $emails['speaker_subject'] ?? 'Thank you for registering, {speaker_name}';
    $body_template = $emails['speaker_body'] ?? "<p>Hi {speaker_name},</p><p>Thank you for registering with our Speakers Bureau.</p><p>You can <a href=\"{public_profile_url}\">view your public profile here</a><br>You can <a href=\"{profile_url}\">edit your profile here</a></p><p>Best regards,<br>{site_name}</p>";

    $subject = sb_process_email_template($subject_template, $speaker_post_id, $speaker_name);
    $body = sb_process_email_template($body_template, $speaker_post_id, $speaker_name);

    // Set HTML email headers with BCC to admin
    $admin_email = get_option('admin_email');
    $headers = [
        'Content-Type: text/html; charset=UTF-8',
        'Bcc: ' . $admin_email
    ];

    return wp_mail($recipient_email, $subject, $body, $headers);
}

/**
 * Generate HTML formatted speaker listing for emails
 */
function sb_generate_speaker_listing_html($speaker_post_id) {
    $fields = function_exists('sb_normalize_form_fields') ? sb_normalize_form_fields() : [];
    if (!is_array($fields)) $fields = [];

    $html = '<div style="border: 1px solid #ddd; padding: 20px; border-radius: 8px; background: #f9f9f9;">';
    $html .= '<h2 style="margin-top: 0; color: #0073aa;">' . esc_html(get_the_title($speaker_post_id)) . '</h2>';

    // Group fields by section (skip private section)
    $fields_by_section = [];
    foreach ($fields as $f) {
        if (empty($f['enabled'])) continue;
        $sec = isset($f['section']) ? (int) $f['section'] : 2;
        if ($sec < 1 || $sec > 8) $sec = 2;
        if ($sec === 8) continue; // Skip private fields
        $fields_by_section[$sec][] = $f;
    }

    foreach ([1,2,3,4,5,6,7] as $section) {
        $section_fields = $fields_by_section[$section] ?? [];
        if (!$section_fields) continue;

        foreach ($section_fields as $field) {
            $key = $field['key'] ?? '';
            $label = $field['label'] ?? '';
            $type = $field['type'] ?? 'text';

            if (!$key || $type === 'edit_link') continue;

            // Check if field is private
            if (in_array($type, ['email', 'phone', 'tel'])) {
                $is_private = get_post_meta($speaker_post_id, $key.'_private', true);
                if ($is_private) continue; // Skip private fields
            }

            $value = get_post_meta($speaker_post_id, $key, true);
            if (!$value) continue;

            // Format value based on type
            $formatted_value = '';
            switch ($type) {
                case 'email':
                    $formatted_value = '<a href="mailto:' . esc_attr($value) . '">' . esc_html($value) . '</a>';
                    break;
                case 'url':
                    $formatted_value = '<a href="' . esc_url($value) . '" target="_blank">' . esc_html($value) . '</a>';
                    break;
                case 'phone':
                case 'tel':
                    $digits = preg_replace('/\D+/', '', $value);
                    $formatted_value = '<a href="tel:' . esc_attr($digits) . '">' . esc_html($value) . '</a>';
                    break;
                case 'textarea':
                    $formatted_value = wpautop(esc_html($value));
                    break;
                case 'image':
                    if (is_numeric($value)) {
                        $src = wp_get_attachment_image_src((int)$value, 'medium');
                        if ($src && !empty($src[0])) {
                            $formatted_value = '<img src="' . esc_url($src[0]) . '" alt="" style="max-width: 200px; height: auto;">';
                        }
                    }
                    break;
                default:
                    if (is_array($value)) {
                        $formatted_value = esc_html(implode(', ', $value));
                    } else {
                        $formatted_value = esc_html($value);
                    }
                    break;
            }

            if ($formatted_value) {
                $html .= '<p><strong>' . esc_html($label) . ':</strong> ' . $formatted_value . '</p>';
            }
        }
    }

    $html .= '</div>';
    return $html;
}

/**
 * Send admin notification email using template
 */
function sb_send_admin_notification_email($speaker_post_id, $speaker_name = '') {
    $emails = get_option('sb_email_templates', []);
    if (!is_array($emails)) $emails = [];

    $subject_template = $emails['admin_subject'] ?? 'New Speaker Registration: {speaker_name}';
    $body_template = $emails['admin_body'] ?? "<p>A new speaker has registered:</p><p><strong>Name:</strong> {speaker_name}<br><strong>Profile:</strong> <a href=\"{profile_url}\">{profile_url}</a></p><p>Login to review.</p>";

    $subject = sb_process_email_template($subject_template, $speaker_post_id, $speaker_name);
    $body = sb_process_email_template($body_template, $speaker_post_id, $speaker_name);

    // Set HTML email headers
    $headers = ['Content-Type: text/html; charset=UTF-8'];

    $admin_email = get_option('admin_email');
    return wp_mail($admin_email, $subject, $body, $headers);
}

/**
 * Send profile update notification email
 */
function sb_send_profile_update_email($speaker_post_id, $recipient_email, $speaker_name = '') {
    $emails = get_option('sb_email_templates', []);
    if (!is_array($emails)) $emails = [];

    $subject_template = $emails['update_subject'] ?? 'Your speaker profile has been updated';
    $body_template = $emails['update_body'] ?? "<p>Hi {speaker_name},</p><p>Your speaker profile has been updated on {site_name}.</p><h3>Your Current Profile:</h3>{speaker_listing}<p><a href=\"{my_profile_url}\">Login to view or edit your profile</a></p><p>Best regards,<br>{site_name}</p>";

    $subject = sb_process_email_template($subject_template, $speaker_post_id, $speaker_name);
    $body = sb_process_email_template($body_template, $speaker_post_id, $speaker_name);

    // Set HTML email headers with BCC to admin
    $admin_email = get_option('admin_email');
    $headers = [
        'Content-Type: text/html; charset=UTF-8',
        'Bcc: ' . $admin_email
    ];

    return wp_mail($recipient_email, $subject, $body, $headers);
}

/** END OF FILE: includes/sb-settings.php */
