<?php
/**
 * File: includes/sb-rotary-import.php
 * Purpose: Import speakers from Rotary 5320 website with preview
 */

if (!defined('ABSPATH')) exit;

class SB_Rotary_Import {

    public function __construct() {
        add_action('admin_menu', [$this, 'add_import_page']);
        add_action('wp_ajax_sb_preview_rotary_speaker', [$this, 'handle_preview']);
        add_action('wp_ajax_sb_import_rotary_speaker', [$this, 'handle_import']);
        add_action('wp_ajax_sb_scan_rotary_range', [$this, 'handle_scan_range']);
        add_action('wp_ajax_sb_preview_csv_import', [$this, 'handle_csv_preview']);
        add_action('wp_ajax_sb_check_website_compatibility', [$this, 'handle_website_check']);
        add_action('wp_ajax_sb_import_csv_speaker', [$this, 'handle_csv_import']);
    }

    public function add_import_page() {
        add_submenu_page(
            'sb-settings',
            __('Import Speakers from Another Source', 'speakers-bureau'),
            __('Import Speakers', 'speakers-bureau'),
            'manage_options',
            'sb-import-speakers',
            [$this, 'import_page']
        );
    }

    public function import_page() {
        ?>
        <div class="wrap">
            <h1><?php esc_html_e('Import Speakers from Another Source', 'speakers-bureau'); ?></h1>

            <div class="notice notice-info">
                <p><strong><?php esc_html_e('Multiple Import Options Available:', 'speakers-bureau'); ?></strong></p>
                <ul style="margin-left: 20px;">
                    <li><?php esc_html_e('Import from CSV file with custom speaker data', 'speakers-bureau'); ?></li>
                    <li><?php esc_html_e('Import from website (speakers.rotary5320.org supported)', 'speakers-bureau'); ?></li>
                    <li><?php esc_html_e('Preview and review data before importing', 'speakers-bureau'); ?></li>
                    <li><?php esc_html_e('Custom import scripts available for other websites', 'speakers-bureau'); ?></li>
                </ul>
            </div>

            <!-- Step 1: Choose Import Source -->
            <div class="card" style="max-width: 800px; margin-bottom: 20px;">
                <h2><?php esc_html_e('Step 1: Choose Import Source', 'speakers-bureau'); ?></h2>

                <form id="sb-import-source-form">
                    <table class="form-table">
                        <tr>
                            <th scope="row"><?php esc_html_e('Import Source', 'speakers-bureau'); ?></th>
                            <td>
                                <fieldset>
                                    <label>
                                        <input type="radio" name="import_source" value="csv" checked>
                                        <?php esc_html_e('CSV File Upload', 'speakers-bureau'); ?>
                                    </label><br>
                                    <label style="margin-top: 10px; display: block;">
                                        <input type="radio" name="import_source" value="website">
                                        <?php esc_html_e('Website Import', 'speakers-bureau'); ?>
                                    </label>
                                </fieldset>
                                <p class="description">
                                    <?php esc_html_e('Choose whether to import from a CSV file or scrape from a website.', 'speakers-bureau'); ?>
                                </p>
                            </td>
                        </tr>
                    </table>
                    <p class="submit">
                        <button type="button" id="sb-continue-import" class="button button-primary">
                            <?php esc_html_e('Continue', 'speakers-bureau'); ?>
                        </button>
                    </p>
                </form>
            </div>

            <!-- CSV Import Section (initially hidden) -->
            <div id="csv-import-section" class="card" style="max-width: 800px; margin-bottom: 20px; display: none;">
                <h2><?php esc_html_e('CSV File Import', 'speakers-bureau'); ?></h2>

                <div class="notice notice-warning">
                    <p><strong><?php esc_html_e('CSV Format Requirements:', 'speakers-bureau'); ?></strong></p>
                    <p><?php esc_html_e('Your CSV file should include columns for: name, email, phone, bio, topics, website, etc.', 'speakers-bureau'); ?></p>
                    <p>
                        <a href="#" id="download-sample-csv" class="button button-secondary">
                            <?php esc_html_e('Download Sample CSV Template', 'speakers-bureau'); ?>
                        </a>
                    </p>
                </div>

                <form id="csv-upload-form" enctype="multipart/form-data">
                    <table class="form-table">
                        <tr>
                            <th scope="row"><?php esc_html_e('CSV File', 'speakers-bureau'); ?></th>
                            <td>
                                <input type="file" name="csv_file" id="csv_file" accept=".csv" required>
                                <p class="description">
                                    <?php esc_html_e('Upload a CSV file containing speaker information.', 'speakers-bureau'); ?>
                                </p>
                            </td>
                        </tr>
                    </table>
                    <p class="submit">
                        <button type="submit" class="button button-primary">
                            <?php esc_html_e('Preview CSV Data', 'speakers-bureau'); ?>
                        </button>
                    </p>
                </form>
            </div>

            <!-- Website Import Section (initially hidden) -->
            <div id="website-import-section" class="card" style="max-width: 800px; margin-bottom: 20px; display: none;">
                <h2><?php esc_html_e('Website Import', 'speakers-bureau'); ?></h2>

                <form id="website-import-form">
                    <table class="form-table">
                        <tr>
                            <th scope="row"><?php esc_html_e('Website URL', 'speakers-bureau'); ?></th>
                            <td>
                                <input type="url" name="website_url" id="website_url" class="regular-text"
                                       placeholder="https://speakers.rotary5320.org" required>
                                <p class="description">
                                    <?php esc_html_e('Enter the website URL to import speakers from.', 'speakers-bureau'); ?>
                                </p>
                            </td>
                        </tr>
                    </table>
                    <p class="submit">
                        <button type="submit" class="button button-primary">
                            <?php esc_html_e('Check Website Compatibility', 'speakers-bureau'); ?>
                        </button>
                    </p>
                </form>
            </div>

            <!-- Step 1: Scan for Available Speakers -->
            <div class="card" style="max-width: 800px; margin-bottom: 20px;">
                <h2><?php esc_html_e('Step 1: Scan Available Speakers', 'speakers-bureau'); ?></h2>

                <p><?php esc_html_e('First, let\'s scan the Rotary website to find which speaker IDs actually exist:', 'speakers-bureau'); ?></p>

                <table class="form-table">
                    <tr>
                        <th scope="row"><?php esc_html_e('Scan Range', 'speakers-bureau'); ?></th>
                        <td>
                            <input type="number" id="scan_start" value="4" min="1" max="200" style="width: 80px;">
                            <?php esc_html_e('to', 'speakers-bureau'); ?>
                            <input type="number" id="scan_end" value="20" min="1" max="200" style="width: 80px;">
                            <p class="description"><?php esc_html_e('Start with a small range (e.g., 4-20) to test', 'speakers-bureau'); ?></p>
                        </td>
                    </tr>
                </table>

                <p class="submit">
                    <button type="button" id="scan-speakers" class="button button-secondary">
                        <?php esc_html_e('Scan for Available Speakers', 'speakers-bureau'); ?>
                    </button>
                </p>

                <div id="scan-status" style="display: none;">
                    <div style="background: #f1f1f1; border: 1px solid #ccc; height: 20px; border-radius: 10px; overflow: hidden; margin: 10px 0;">
                        <div id="scan-progress" style="background: #2271b1; height: 100%; width: 0%; transition: width 0.3s;"></div>
                    </div>
                    <p id="scan-message"><?php esc_html_e('Scanning...', 'speakers-bureau'); ?></p>
                </div>

                <div id="scan-results" style="display: none; margin-top: 20px;">
                    <h3><?php esc_html_e('Available Speakers Found:', 'speakers-bureau'); ?></h3>
                    <div id="available-speakers"></div>

                    <!-- Auto-Import Controls -->
                    <div id="auto-import-controls" style="display: none; margin-top: 20px; padding: 15px; background: #f8f9fa; border: 1px solid #ddd; border-radius: 5px;">
                        <h4><?php esc_html_e('Auto-Import Options', 'speakers-bureau'); ?></h4>
                        <p><?php esc_html_e('Import all found speakers automatically (no manual clicking required):', 'speakers-bureau'); ?></p>

                        <label style="display: block; margin: 10px 0;">
                            <input type="checkbox" id="skip-existing" checked>
                            <?php esc_html_e('Skip speakers that already exist in the database', 'speakers-bureau'); ?>
                        </label>

                        <label style="display: block; margin: 10px 0;">
                            <input type="checkbox" id="import-delay" checked>
                            <?php esc_html_e('Add delay between imports (recommended to avoid server overload)', 'speakers-bureau'); ?>
                        </label>

                        <p class="submit" style="margin-top: 15px;">
                            <button type="button" id="auto-import-all" class="button button-primary">
                                <?php esc_html_e('Auto-Import All Speakers', 'speakers-bureau'); ?>
                            </button>
                            <button type="button" id="stop-auto-import" class="button button-secondary" style="display: none; margin-left: 10px;">
                                <?php esc_html_e('Stop Auto-Import', 'speakers-bureau'); ?>
                            </button>
                        </p>

                        <div id="auto-import-status" style="display: none; margin-top: 15px;">
                            <div style="background: #f1f1f1; border: 1px solid #ccc; height: 20px; border-radius: 10px; overflow: hidden; margin: 10px 0;">
                                <div id="auto-import-progress" style="background: #00a32a; height: 100%; width: 0%; transition: width 0.3s;"></div>
                            </div>
                            <p id="auto-import-message"><?php esc_html_e('Starting auto-import...', 'speakers-bureau'); ?></p>
                            <div id="auto-import-results" style="margin-top: 10px; font-family: monospace; font-size: 12px; max-height: 200px; overflow-y: auto; background: #fff; padding: 10px; border: 1px solid #ddd;"></div>
                        </div>
                    </div>
                </div>
            </div>

            <!-- Step 2: Preview Individual Speakers -->
            <div class="card" style="max-width: 1200px;" id="preview-section" style="display: none;">
                <h2><?php esc_html_e('Step 2: Preview Speaker Data', 'speakers-bureau'); ?></h2>

                <p><?php esc_html_e('Enter a speaker ID to preview how the data will be mapped:', 'speakers-bureau'); ?></p>

                <div style="margin-bottom: 20px;">
                    <label for="speaker_id_input"><?php esc_html_e('Speaker ID:', 'speakers-bureau'); ?></label>
                    <input type="number" id="speaker_id_input" min="1" max="200" style="width: 80px; margin: 0 10px;">
                    <button type="button" id="preview-speaker" class="button">
                        <?php esc_html_e('Preview Speaker', 'speakers-bureau'); ?>
                    </button>
                </div>

                <div id="preview-loading" style="display: none;">
                    <p><?php esc_html_e('Loading speaker data...', 'speakers-bureau'); ?></p>
                </div>

                <div id="preview-content" style="display: none;">
                    <!-- Speaker preview will be loaded here -->
                </div>
            </div>
        </div>

        <script>
        jQuery(document).ready(function($) {
            // Handle source selection
            $('#sb-continue-import').click(function() {
                const selectedSource = $('input[name="import_source"]:checked').val();

                // Hide both sections first
                $('#csv-import-section, #website-import-section').hide();

                // Show the appropriate section
                if (selectedSource === 'csv') {
                    $('#csv-import-section').show();
                } else if (selectedSource === 'website') {
                    $('#website-import-section').show();
                }
            });

            // Handle CSV upload and preview
            $('#csv-upload-form').on('submit', function(e) {
                e.preventDefault();

                const fileInput = $('#csv_file')[0];
                const file = fileInput.files[0];

                if (!file) {
                    alert('<?php esc_js(_e('Please select a CSV file', 'speakers-bureau')); ?>');
                    return;
                }

                const formData = new FormData();
                formData.append('action', 'sb_preview_csv_import');
                formData.append('csv_file', file);
                formData.append('nonce', '<?php echo wp_create_nonce('sb_csv_preview'); ?>');

                const submitBtn = $(this).find('button[type="submit"]');
                submitBtn.prop('disabled', true).text('<?php esc_js(_e('Processing CSV...', 'speakers-bureau')); ?>');

                $.ajax({
                    url: ajaxurl,
                    type: 'POST',
                    data: formData,
                    processData: false,
                    contentType: false,
                    success: function(response) {
                        if (response.success) {
                            $('#preview-content').html(response.data.html).show();
                        } else {
                            $('#preview-content').html('<div class="notice notice-error"><p>' + response.data + '</p></div>').show();
                        }
                    },
                    error: function() {
                        $('#preview-content').html('<div class="notice notice-error"><p><?php esc_js(_e('Error processing CSV file', 'speakers-bureau')); ?></p></div>').show();
                    },
                    complete: function() {
                        submitBtn.prop('disabled', false).text('<?php esc_js(_e('Preview CSV Data', 'speakers-bureau')); ?>');
                    }
                });
            });

            // Handle website import form
            $('#website-import-form').on('submit', function(e) {
                e.preventDefault();

                const websiteUrl = $('#website_url').val().trim();

                if (!websiteUrl) {
                    alert('<?php esc_js(_e('Please enter a website URL', 'speakers-bureau')); ?>');
                    return;
                }

                const submitBtn = $(this).find('button[type="submit"]');
                submitBtn.prop('disabled', true).text('<?php esc_js(_e('Checking...', 'speakers-bureau')); ?>');

                $.ajax({
                    url: ajaxurl,
                    type: 'POST',
                    data: {
                        action: 'sb_check_website_compatibility',
                        website_url: websiteUrl,
                        nonce: '<?php echo wp_create_nonce('sb_website_check'); ?>'
                    },
                    success: function(response) {
                        if (response.success) {
                            if (response.data.compatible) {
                                // Website is compatible, proceed with existing functionality
                                $('#preview-section').show();
                                window.location.hash = '#preview-section';
                            } else {
                                // Show incompatible website message
                                $('#preview-content').html(response.data.html).show();
                            }
                        } else {
                            $('#preview-content').html('<div class="notice notice-error"><p>' + response.data + '</p></div>').show();
                        }
                    },
                    error: function() {
                        $('#preview-content').html('<div class="notice notice-error"><p><?php esc_js(_e('Error checking website compatibility', 'speakers-bureau')); ?></p></div>').show();
                    },
                    complete: function() {
                        submitBtn.prop('disabled', false).text('<?php esc_js(_e('Check Website Compatibility', 'speakers-bureau')); ?>');
                    }
                });
            });

            // Handle sample CSV download
            $('#download-sample-csv').click(function(e) {
                e.preventDefault();

                // Create CSV content
                const csvContent = `name,email,phone,headline,organization,website,topics,summary,city,state,country,zip,facebookurl,linkedinurl,instagramurl,xurl,youtubeurl,online-mtg
"John Doe","john.doe@example.com","(555) 123-4567","Marketing Director","ABC Company","https://johndoe.com","Marketing, Leadership, Innovation","John discusses innovative marketing strategies and leadership principles that have helped grow successful businesses.","Orange","CA","USA","92867","https://facebook.com/johndoe","https://linkedin.com/in/johndoe","","","","YES"
"Jane Smith","jane.smith@example.com","(555) 987-6543","CEO & Founder","Tech Startup Inc","https://techstartup.com","Technology, Entrepreneurship, Women in Business","Jane shares her journey from startup founder to successful CEO, focusing on overcoming challenges in the tech industry.","Irvine","CA","USA","92618","","https://linkedin.com/in/janesmith","https://instagram.com/janesmith","https://twitter.com/janesmith","","NO"`;

                // Create and download the file
                const blob = new Blob([csvContent], { type: 'text/csv' });
                const url = window.URL.createObjectURL(blob);
                const a = document.createElement('a');
                a.style.display = 'none';
                a.href = url;
                a.download = 'speaker-import-template.csv';
                document.body.appendChild(a);
                a.click();
                window.URL.revokeObjectURL(url);
                document.body.removeChild(a);
            });

            // Show preview section immediately (for manual speaker ID entry)
            $('#preview-section').show();

            // Scan for available speakers
            $('#scan-speakers').click(function() {
                const startId = parseInt($('#scan_start').val()) || 4;
                const endId = parseInt($('#scan_end').val()) || 20;
                const button = $(this);

                button.prop('disabled', true).text('<?php esc_js(_e('Scanning...', 'speakers-bureau')); ?>');
                $('#scan-status').show();
                $('#scan-results').hide();

                scanSpeakers(startId, endId);
            });

            function scanSpeakers(startId, endId) {
                const total = endId - startId + 1;
                let current = 0;
                let found = [];

                function scanNext(id) {
                    if (id > endId) {
                        $('#scan-message').text('<?php esc_js(_e('Scan completed!', 'speakers-bureau')); ?>');
                        $('#scan-speakers').prop('disabled', false).text('<?php esc_js(_e('Scan for Available Speakers', 'speakers-bureau')); ?>');

                        if (found.length > 0) {
                            let html = '<div style="display: grid; grid-template-columns: repeat(auto-fill, minmax(100px, 1fr)); gap: 10px; margin-top: 10px;">';
                            found.forEach(function(speaker) {
                                html += `<button type="button" class="button speaker-preview-btn" data-id="${speaker.id}" style="padding: 8px;">${speaker.id}: ${speaker.name}</button>`;
                            });
                            html += '</div>';
                            html += `<p style="margin-top: 15px;"><strong><?php esc_js(_e('Found:', 'speakers-bureau')); ?></strong> ${found.length} <?php esc_js(_e('speakers', 'speakers-bureau')); ?></p>`;
                            $('#available-speakers').html(html);
                            $('#scan-results').show();

                            // Show auto-import controls
                            $('#auto-import-controls').show();

                            // Store found speakers for auto-import
                            window.foundSpeakers = found;
                        } else {
                            $('#available-speakers').html('<p style="color: #d63638;"><?php esc_js(_e('No speakers found in this range.', 'speakers-bureau')); ?></p>');
                            $('#scan-results').show();
                            $('#auto-import-controls').hide();
                        }
                        return;
                    }

                    current++;
                    const progress = (current / total) * 100;
                    $('#scan-progress').css('width', progress + '%');
                    $('#scan-message').text('<?php esc_js(_e('Checking ID:', 'speakers-bureau')); ?> ' + id + ' (' + current + '/' + total + ')');

                    $.ajax({
                        url: ajaxurl,
                        type: 'POST',
                        data: {
                            action: 'sb_scan_rotary_range',
                            speaker_id: id,
                            nonce: '<?php echo wp_create_nonce('sb_rotary_scan'); ?>'
                        },
                        success: function(response) {
                            if (response.success && response.data.exists) {
                                found.push({id: id, name: response.data.name});
                            }
                        },
                        complete: function() {
                            setTimeout(() => scanNext(id + 1), 200); // Small delay
                        }
                    });
                }

                scanNext(startId);
            }

            // Preview individual speaker
            $('#preview-speaker, body').on('click', '#preview-speaker, .speaker-preview-btn', function() {
                let speakerId;
                if ($(this).hasClass('speaker-preview-btn')) {
                    speakerId = $(this).data('id');
                    $('#speaker_id_input').val(speakerId);
                } else {
                    speakerId = parseInt($('#speaker_id_input').val());
                }

                if (!speakerId) {
                    alert('<?php esc_js(_e('Please enter a speaker ID', 'speakers-bureau')); ?>');
                    return;
                }

                $('#preview-loading').show();
                $('#preview-content').hide();

                $.ajax({
                    url: ajaxurl,
                    type: 'POST',
                    data: {
                        action: 'sb_preview_rotary_speaker',
                        speaker_id: speakerId,
                        nonce: '<?php echo wp_create_nonce('sb_rotary_preview'); ?>'
                    },
                    success: function(response) {
                        $('#preview-loading').hide();
                        if (response.success) {
                            $('#preview-content').html(response.data.html).show();
                        } else {
                            $('#preview-content').html('<div class="notice notice-error"><p>' + response.data + '</p></div>').show();
                        }
                    },
                    error: function() {
                        $('#preview-loading').hide();
                        $('#preview-content').html('<div class="notice notice-error"><p><?php esc_js(_e('Error loading speaker data', 'speakers-bureau')); ?></p></div>').show();
                    }
                });
            });

            // Handle import/skip buttons
            $('body').on('click', '.import-speaker-btn', function() {
                const button = $(this);
                const speakerId = button.data('id');
                const action = button.data('action'); // 'import' or 'skip'

                if (action === 'skip') {
                    button.closest('.speaker-preview-card').find('.import-result').html('<div class="notice notice-info"><p><?php esc_js(_e('Skipped by user', 'speakers-bureau')); ?></p></div>');
                    return;
                }

                button.prop('disabled', true).text('<?php esc_js(_e('Importing...', 'speakers-bureau')); ?>');

                $.ajax({
                    url: ajaxurl,
                    type: 'POST',
                    data: {
                        action: 'sb_import_rotary_speaker',
                        speaker_id: speakerId,
                        nonce: '<?php echo wp_create_nonce('sb_rotary_import'); ?>'
                    },
                    success: function(response) {
                        if (response.success) {
                            const result = `<div class="notice notice-success"><p><strong><?php esc_js(_e('Imported successfully!', 'speakers-bureau')); ?></strong><br><?php esc_js(_e('Post ID:', 'speakers-bureau')); ?> ${response.data.post_id}</p></div>`;
                            button.closest('.speaker-preview-card').find('.import-result').html(result);
                        } else {
                            const result = `<div class="notice notice-error"><p><strong><?php esc_js(_e('Import failed:', 'speakers-bureau')); ?></strong><br>${response.data}</p></div>`;
                            button.closest('.speaker-preview-card').find('.import-result').html(result);
                        }
                    },
                    error: function() {
                        const result = `<div class="notice notice-error"><p><?php esc_js(_e('Network error during import', 'speakers-bureau')); ?></p></div>`;
                        button.closest('.speaker-preview-card').find('.import-result').html(result);
                    },
                    complete: function() {
                        button.prop('disabled', false).text('<?php esc_js(_e('Import Speaker', 'speakers-bureau')); ?>');
                    }
                });
            });

            // Auto-import functionality
            let autoImportRunning = false;
            let autoImportStopped = false;

            $('#auto-import-all').click(function() {
                if (!window.foundSpeakers || window.foundSpeakers.length === 0) {
                    alert('<?php esc_js(_e('No speakers found to import', 'speakers-bureau')); ?>');
                    return;
                }

                const skipExisting = $('#skip-existing').is(':checked');
                const useDelay = $('#import-delay').is(':checked');

                autoImportRunning = true;
                autoImportStopped = false;
                $('#auto-import-all').hide();
                $('#stop-auto-import').show();
                $('#auto-import-status').show();
                $('#auto-import-results').empty();

                const speakers = [...window.foundSpeakers]; // Copy array
                const total = speakers.length;
                let current = 0;
                let imported = 0;
                let skipped = 0;
                let errors = 0;

                function logResult(message, type = 'info') {
                    const timestamp = new Date().toLocaleTimeString();
                    const color = type === 'success' ? '#00a32a' : type === 'error' ? '#d63638' : type === 'skip' ? '#dba617' : '#666';
                    $('#auto-import-results').append(`<div style="color: ${color}">[${timestamp}] ${message}</div>`);
                    $('#auto-import-results').scrollTop($('#auto-import-results')[0].scrollHeight);
                }

                function importNext() {
                    if (autoImportStopped || current >= speakers.length) {
                        // Auto-import completed
                        autoImportRunning = false;
                        $('#auto-import-all').show();
                        $('#stop-auto-import').hide();

                        const finalMessage = autoImportStopped ?
                            '<?php esc_js(_e('Auto-import stopped by user', 'speakers-bureau')); ?>' :
                            `<?php esc_js(_e('Auto-import completed!', 'speakers-bureau')); ?> ${imported} <?php esc_js(_e('imported', 'speakers-bureau')); ?>, ${skipped} <?php esc_js(_e('skipped', 'speakers-bureau')); ?>, ${errors} <?php esc_js(_e('errors', 'speakers-bureau')); ?>`;

                        $('#auto-import-message').text(finalMessage);
                        logResult(finalMessage, 'info');
                        return;
                    }

                    const speaker = speakers[current];
                    const progress = ((current + 1) / total) * 100;
                    $('#auto-import-progress').css('width', progress + '%');
                    $('#auto-import-message').text(`<?php esc_js(_e('Importing:', 'speakers-bureau')); ?> ${speaker.name} (${current + 1}/${total})`);

                    $.ajax({
                        url: ajaxurl,
                        type: 'POST',
                        data: {
                            action: 'sb_import_rotary_speaker',
                            speaker_id: speaker.id,
                            nonce: '<?php echo wp_create_nonce('sb_rotary_import'); ?>'
                        },
                        success: function(response) {
                            if (response.success) {
                                imported++;
                                logResult(`✓ ${speaker.name} (ID: ${speaker.id}) - Post ID: ${response.data.post_id}`, 'success');
                            } else {
                                if (skipExisting && response.data && response.data.includes('already exists')) {
                                    skipped++;
                                    logResult(`⚠ ${speaker.name} (ID: ${speaker.id}) - Already exists (skipped)`, 'skip');
                                } else {
                                    errors++;
                                    logResult(`✗ ${speaker.name} (ID: ${speaker.id}) - Error: ${response.data}`, 'error');
                                }
                            }
                        },
                        error: function() {
                            errors++;
                            logResult(`✗ ${speaker.name} (ID: ${speaker.id}) - Network error`, 'error');
                        },
                        complete: function() {
                            current++;
                            if (useDelay && current < speakers.length && !autoImportStopped) {
                                setTimeout(importNext, 1000); // 1 second delay
                            } else {
                                importNext();
                            }
                        }
                    });
                }

                logResult(`Starting auto-import of ${total} speakers...`, 'info');
                importNext();
            });

            $('#stop-auto-import').click(function() {
                autoImportStopped = true;
                $(this).prop('disabled', true).text('<?php esc_js(_e('Stopping...', 'speakers-bureau')); ?>');
            });

            // Handle CSV speaker import buttons
            $('body').on('click', '.import-csv-speaker-btn', function() {
                const button = $(this);
                const speakerIndex = button.data('index');
                const action = button.data('action');

                if (action === 'skip') {
                    button.closest('.speaker-preview-card').find('.import-result').html('<div class="notice notice-info"><p><?php esc_js(_e('Skipped by user', 'speakers-bureau')); ?></p></div>');
                    return;
                }

                if (!window.csvSpeakersData || !window.csvSpeakersData[speakerIndex]) {
                    alert('<?php esc_js(_e('Speaker data not found', 'speakers-bureau')); ?>');
                    return;
                }

                button.prop('disabled', true).text('<?php esc_js(_e('Importing...', 'speakers-bureau')); ?>');

                const speakerData = window.csvSpeakersData[speakerIndex];

                $.ajax({
                    url: ajaxurl,
                    type: 'POST',
                    data: {
                        action: 'sb_import_csv_speaker',
                        speaker_data: JSON.stringify(speakerData),
                        nonce: '<?php echo wp_create_nonce('sb_csv_import'); ?>'
                    },
                    success: function(response) {
                        if (response.success) {
                            const result = `<div class="notice notice-success"><p><strong><?php esc_js(_e('Imported successfully!', 'speakers-bureau')); ?></strong><br><?php esc_js(_e('Post ID:', 'speakers-bureau')); ?> ${response.data.post_id}</p></div>`;
                            button.closest('.speaker-preview-card').find('.import-result').html(result);
                        } else {
                            const result = `<div class="notice notice-error"><p><strong><?php esc_js(_e('Import failed:', 'speakers-bureau')); ?></strong><br>${response.data}</p></div>`;
                            button.closest('.speaker-preview-card').find('.import-result').html(result);
                        }
                    },
                    error: function() {
                        const result = `<div class="notice notice-error"><p><?php esc_js(_e('Network error during import', 'speakers-bureau')); ?></p></div>`;
                        button.closest('.speaker-preview-card').find('.import-result').html(result);
                    },
                    complete: function() {
                        button.prop('disabled', false).text('<?php esc_js(_e('Import Speaker', 'speakers-bureau')); ?>');
                    }
                });
            });

            // Handle bulk CSV import
            $('body').on('click', '#import-all-csv', function() {
                if (!window.csvSpeakersData || window.csvSpeakersData.length === 0) {
                    alert('<?php esc_js(_e('No CSV speakers found to import', 'speakers-bureau')); ?>');
                    return;
                }

                const skipExisting = $('#csv-skip-existing').is(':checked');

                $('#import-all-csv').hide();
                $('#csv-import-status').show();
                $('#csv-import-results').empty();

                const speakers = [...window.csvSpeakersData];
                const total = speakers.length;
                let current = 0;
                let imported = 0;
                let skipped = 0;
                let errors = 0;

                function logResult(message, type = 'info') {
                    const timestamp = new Date().toLocaleTimeString();
                    const color = type === 'success' ? '#00a32a' : type === 'error' ? '#d63638' : type === 'skip' ? '#dba617' : '#666';
                    $('#csv-import-results').append(`<div style="color: ${color}">[${timestamp}] ${message}</div>`);
                    $('#csv-import-results').scrollTop($('#csv-import-results')[0].scrollHeight);
                }

                function importNext() {
                    if (current >= speakers.length) {
                        $('#import-all-csv').show();
                        const finalMessage = `<?php esc_js(_e('CSV import completed!', 'speakers-bureau')); ?> ${imported} <?php esc_js(_e('imported', 'speakers-bureau')); ?>, ${skipped} <?php esc_js(_e('skipped', 'speakers-bureau')); ?>, ${errors} <?php esc_js(_e('errors', 'speakers-bureau')); ?>`;
                        $('#csv-import-message').text(finalMessage);
                        logResult(finalMessage, 'info');
                        return;
                    }

                    const speaker = speakers[current];
                    const progress = ((current + 1) / total) * 100;
                    $('#csv-import-progress').css('width', progress + '%');
                    $('#csv-import-message').text(`<?php esc_js(_e('Importing:', 'speakers-bureau')); ?> ${speaker.name} (${current + 1}/${total})`);

                    $.ajax({
                        url: ajaxurl,
                        type: 'POST',
                        data: {
                            action: 'sb_import_csv_speaker',
                            speaker_data: JSON.stringify(speaker),
                            nonce: '<?php echo wp_create_nonce('sb_csv_import'); ?>'
                        },
                        success: function(response) {
                            if (response.success) {
                                imported++;
                                logResult(`✓ ${speaker.name} - Post ID: ${response.data.post_id}`, 'success');
                            } else {
                                if (skipExisting && response.data && response.data.includes('already exists')) {
                                    skipped++;
                                    logResult(`⚠ ${speaker.name} - Already exists (skipped)`, 'skip');
                                } else {
                                    errors++;
                                    logResult(`✗ ${speaker.name} - Error: ${response.data}`, 'error');
                                }
                            }
                        },
                        error: function() {
                            errors++;
                            logResult(`✗ ${speaker.name} - Network error`, 'error');
                        },
                        complete: function() {
                            current++;
                            setTimeout(importNext, 500); // Small delay between imports
                        }
                    });
                }

                logResult(`Starting CSV import of ${total} speakers...`, 'info');
                importNext();
            });
        });
        </script>
        <?php
    }

    public function handle_scan_range() {
        if (!current_user_can('manage_options')) {
            wp_die(__('Access denied.', 'speakers-bureau'));
        }

        if (!wp_verify_nonce($_POST['nonce'], 'sb_rotary_scan')) {
            wp_send_json_error(__('Invalid nonce.', 'speakers-bureau'));
        }

        $speaker_id = intval($_POST['speaker_id']);

        try {
            $exists = $this->check_speaker_exists($speaker_id);
            wp_send_json_success($exists);
        } catch (Exception $e) {
            wp_send_json_error($e->getMessage());
        }
    }

    public function handle_preview() {
        if (!current_user_can('manage_options')) {
            wp_die(__('Access denied.', 'speakers-bureau'));
        }

        if (!wp_verify_nonce($_POST['nonce'], 'sb_rotary_preview')) {
            wp_send_json_error(__('Invalid nonce.', 'speakers-bureau'));
        }

        $speaker_id = intval($_POST['speaker_id']);

        try {
            $preview_html = $this->get_speaker_preview($speaker_id);
            wp_send_json_success(['html' => $preview_html]);
        } catch (Exception $e) {
            wp_send_json_error($e->getMessage());
        }
    }

    public function handle_import() {
        if (!current_user_can('manage_options')) {
            wp_die(__('Access denied.', 'speakers-bureau'));
        }

        if (!wp_verify_nonce($_POST['nonce'], 'sb_rotary_import')) {
            wp_send_json_error(__('Invalid nonce.', 'speakers-bureau'));
        }

        $speaker_id = intval($_POST['speaker_id']);

        try {
            $result = $this->import_single_speaker($speaker_id);
            wp_send_json_success($result);
        } catch (Exception $e) {
            wp_send_json_error($e->getMessage());
        }
    }

    public function handle_csv_preview() {
        if (!current_user_can('manage_options')) {
            wp_die(__('Access denied.', 'speakers-bureau'));
        }

        if (!wp_verify_nonce($_POST['nonce'], 'sb_csv_preview')) {
            wp_send_json_error(__('Invalid nonce.', 'speakers-bureau'));
        }

        if (!isset($_FILES['csv_file']) || $_FILES['csv_file']['error'] !== UPLOAD_ERR_OK) {
            wp_send_json_error(__('Please upload a valid CSV file.', 'speakers-bureau'));
        }

        try {
            $csv_data = $this->process_csv_file($_FILES['csv_file']['tmp_name']);
            $preview_html = $this->render_csv_preview($csv_data);
            wp_send_json_success(['html' => $preview_html]);
        } catch (Exception $e) {
            wp_send_json_error($e->getMessage());
        }
    }

    public function handle_website_check() {
        if (!current_user_can('manage_options')) {
            wp_die(__('Access denied.', 'speakers-bureau'));
        }

        if (!wp_verify_nonce($_POST['nonce'], 'sb_website_check')) {
            wp_send_json_error(__('Invalid nonce.', 'speakers-bureau'));
        }

        $website_url = sanitize_url($_POST['website_url']);

        if (empty($website_url)) {
            wp_send_json_error(__('Please provide a website URL.', 'speakers-bureau'));
        }

        try {
            $result = $this->check_website_compatibility($website_url);
            wp_send_json_success($result);
        } catch (Exception $e) {
            wp_send_json_error($e->getMessage());
        }
    }

    public function handle_csv_import() {
        if (!current_user_can('manage_options')) {
            wp_die(__('Access denied.', 'speakers-bureau'));
        }

        if (!wp_verify_nonce($_POST['nonce'], 'sb_csv_import')) {
            wp_send_json_error(__('Invalid nonce.', 'speakers-bureau'));
        }

        $speaker_data_json = $_POST['speaker_data'] ?? '';

        if (empty($speaker_data_json)) {
            wp_send_json_error(__('No speaker data provided.', 'speakers-bureau'));
        }

        $speaker_data = json_decode($speaker_data_json, true);

        if (!$speaker_data || !isset($speaker_data['name'])) {
            wp_send_json_error(__('Invalid speaker data format.', 'speakers-bureau'));
        }

        try {
            $result = $this->import_csv_speaker($speaker_data);
            wp_send_json_success($result);
        } catch (Exception $e) {
            wp_send_json_error($e->getMessage());
        }
    }

    private function process_csv_file($file_path) {
        if (!file_exists($file_path)) {
            throw new Exception(__('CSV file not found.', 'speakers-bureau'));
        }

        $csv_data = [];
        $headers = [];

        if (($handle = fopen($file_path, 'r')) !== FALSE) {
            $row_number = 0;
            while (($data = fgetcsv($handle, 1000, ',')) !== FALSE) {
                $row_number++;

                if ($row_number === 1) {
                    // First row is headers
                    $headers = array_map('trim', $data);
                    continue;
                }

                if (count($data) !== count($headers)) {
                    throw new Exception(sprintf(__('Row %d has %d columns but expected %d columns.', 'speakers-bureau'), $row_number, count($data), count($headers)));
                }

                $row_data = array_combine($headers, array_map('trim', $data));

                // Validate required fields
                if (empty($row_data['name'])) {
                    throw new Exception(sprintf(__('Row %d is missing required field: name', 'speakers-bureau'), $row_number));
                }

                $csv_data[] = $row_data;
            }
            fclose($handle);
        } else {
            throw new Exception(__('Could not open CSV file for reading.', 'speakers-bureau'));
        }

        if (empty($csv_data)) {
            throw new Exception(__('CSV file contains no speaker data.', 'speakers-bureau'));
        }

        return $csv_data;
    }

    private function render_csv_preview($csv_data) {
        ob_start();
        ?>
        <div class="csv-preview-container">
            <h3><?php esc_html_e('CSV Import Preview', 'speakers-bureau'); ?></h3>
            <p><?php printf(__('Found %d speakers to import:', 'speakers-bureau'), count($csv_data)); ?></p>

            <?php foreach ($csv_data as $index => $speaker_data): ?>
                <div class="speaker-preview-card" style="border: 1px solid #ddd; padding: 20px; border-radius: 8px; background: #f9f9f9; margin: 20px 0;">
                    <div style="display: flex; justify-content: space-between; align-items: flex-start; margin-bottom: 20px;">
                        <h4 style="margin: 0; color: #1d2327;">
                            <?php echo esc_html($speaker_data['name'] ?? 'Unknown Speaker'); ?>
                            <span style="font-size: 14px; color: #646970; font-weight: normal;">(Row <?php echo ($index + 2); ?>)</span>
                        </h4>

                        <?php
                        // Check if speaker already exists
                        $existing = null;
                        if (!empty($speaker_data['email'])) {
                            $existing = $this->find_existing_speaker($speaker_data['email']);
                        }
                        ?>

                        <?php if ($existing): ?>
                            <div class="notice notice-warning" style="margin: 0; padding: 5px 10px;">
                                <strong><?php esc_html_e('Already Exists!', 'speakers-bureau'); ?></strong><br>
                                <small><?php printf(__('Post ID: %d', 'speakers-bureau'), $existing->ID); ?></small>
                            </div>
                        <?php endif; ?>
                    </div>

                    <div style="display: grid; grid-template-columns: 1fr 1fr; gap: 30px; margin-bottom: 20px;">
                        <div>
                            <h5 style="margin-top: 0; color: #0073aa;"><?php esc_html_e('CSV Data', 'speakers-bureau'); ?></h5>
                            <table class="form-table" style="margin: 0;">
                                <?php foreach ($speaker_data as $key => $value): ?>
                                    <?php if (!empty($value)): ?>
                                        <tr>
                                            <td style="padding: 5px 10px 5px 0; font-weight: 600; width: 120px;"><?php echo esc_html($key); ?>:</td>
                                            <td style="padding: 5px 0;">
                                                <?php echo esc_html($value); ?>
                                            </td>
                                        </tr>
                                    <?php endif; ?>
                                <?php endforeach; ?>
                            </table>
                        </div>
                    </div>

                    <div style="border-top: 1px solid #ddd; padding-top: 20px; display: flex; justify-content: space-between; align-items: center;">
                        <div>
                            <?php if (!$existing): ?>
                                <button type="button" class="button button-primary import-csv-speaker-btn" data-index="<?php echo $index; ?>" data-action="import">
                                    <?php esc_html_e('Import Speaker', 'speakers-bureau'); ?>
                                </button>
                                <button type="button" class="button import-csv-speaker-btn" data-index="<?php echo $index; ?>" data-action="skip" style="margin-left: 10px;">
                                    <?php esc_html_e('Skip', 'speakers-bureau'); ?>
                                </button>
                            <?php else: ?>
                                <span style="color: #d63638; font-weight: 600;"><?php esc_html_e('Cannot import - speaker already exists', 'speakers-bureau'); ?></span>
                            <?php endif; ?>
                        </div>
                    </div>

                    <div class="import-result" style="margin-top: 15px;">
                        <!-- Import results will appear here -->
                    </div>
                </div>
            <?php endforeach; ?>

            <div style="margin-top: 30px; padding: 20px; background: #f8f9fa; border: 1px solid #ddd; border-radius: 5px;">
                <h4><?php esc_html_e('Bulk Import Options', 'speakers-bureau'); ?></h4>
                <p><?php esc_html_e('Import all speakers from this CSV file:', 'speakers-bureau'); ?></p>

                <label style="display: block; margin: 10px 0;">
                    <input type="checkbox" id="csv-skip-existing" checked>
                    <?php esc_html_e('Skip speakers that already exist', 'speakers-bureau'); ?>
                </label>

                <p class="submit">
                    <button type="button" id="import-all-csv" class="button button-primary">
                        <?php esc_html_e('Import All Speakers', 'speakers-bureau'); ?>
                    </button>
                </p>

                <div id="csv-import-status" style="display: none; margin-top: 15px;">
                    <div style="background: #f1f1f1; border: 1px solid #ccc; height: 20px; border-radius: 10px; overflow: hidden; margin: 10px 0;">
                        <div id="csv-import-progress" style="background: #00a32a; height: 100%; width: 0%; transition: width 0.3s;"></div>
                    </div>
                    <p id="csv-import-message"><?php esc_html_e('Starting import...', 'speakers-bureau'); ?></p>
                    <div id="csv-import-results" style="margin-top: 10px; font-family: monospace; font-size: 12px; max-height: 200px; overflow-y: auto; background: #fff; padding: 10px; border: 1px solid #ddd;"></div>
                </div>
            </div>

            <script>
            // Store CSV data for JavaScript access
            window.csvSpeakersData = <?php echo json_encode($csv_data); ?>;
            </script>
        </div>
        <?php
        return ob_get_clean();
    }

    private function check_website_compatibility($website_url) {
        // Check if it's the supported Rotary 5320 website
        if (strpos($website_url, 'speakers.rotary5320.org') !== false) {
            return [
                'compatible' => true,
                'message' => __('Website is compatible. You can now use the existing import functionality.', 'speakers-bureau')
            ];
        }

        // For any other website, show contact message
        ob_start();
        ?>
        <div class="notice notice-info" style="padding: 20px; margin: 20px 0;">
            <h3><?php esc_html_e('Custom Import Required', 'speakers-bureau'); ?></h3>
            <p><strong><?php esc_html_e('Website:', 'speakers-bureau'); ?></strong> <?php echo esc_html($website_url); ?></p>
            <p><?php esc_html_e('This website is not currently supported by the automated import system.', 'speakers-bureau'); ?></p>
            <p><?php esc_html_e('For a custom import script for this website, please contact:', 'speakers-bureau'); ?></p>
            <p>
                <strong><?php esc_html_e('Email:', 'speakers-bureau'); ?></strong>
                <a href="mailto:garryheath@gmail.com?subject=Custom Speaker Import Script Request&body=Website: <?php echo urlencode($website_url); ?>">garryheath@gmail.com</a>
            </p>
            <p><em><?php esc_html_e('Include the website URL and any specific requirements in your email.', 'speakers-bureau'); ?></em></p>
        </div>
        <?php
        $html = ob_get_clean();

        return [
            'compatible' => false,
            'html' => $html
        ];
    }

    private function import_csv_speaker($speaker_data) {
        // Check if speaker already exists
        if (!empty($speaker_data['email'])) {
            $existing = $this->find_existing_speaker($speaker_data['email']);
            if ($existing) {
                throw new Exception(sprintf(__('Speaker %s already exists (Post ID: %d)', 'speakers-bureau'), $speaker_data['name'], $existing->ID));
            }
        }

        // Import the speaker using CSV data
        $post_id = $this->create_speaker_post($speaker_data);

        return [
            'post_id' => $post_id,
            'name' => $speaker_data['name']
        ];
    }

    private function check_speaker_exists($speaker_id) {
        $url = "https://speakers.rotary5320.org/speakers/entry/{$speaker_id}/";

        $response = wp_remote_get($url, [
            'timeout' => 30,  // Increased timeout to match preview
            'user-agent' => 'WordPress Speaker Scanner'
        ]);

        if (is_wp_error($response)) {
            error_log("Scan Error for ID {$speaker_id}: " . $response->get_error_message());
            return ['exists' => false, 'error' => $response->get_error_message()];
        }

        $html = wp_remote_retrieve_body($response);

        // Check for "No Entries Found" (use same specific check as preview)
        if (stripos($html, 'no entries found') !== false) {
            return ['exists' => false];
        }

        // Use the same robust parsing method as preview
        $speaker_data = $this->parse_speaker_html($html);
        $name = $speaker_data['name'] ?? '';

        return [
            'exists' => !empty($name),
            'name' => $name ?: "Speaker {$speaker_id}"
        ];
    }

    private function extract_speaker_name($html) {
        // Create DOM for better parsing
        $dom = new DOMDocument();
        @$dom->loadHTML($html, LIBXML_HTML_NOIMPLIED | LIBXML_HTML_NODEFDTD);
        $xpath = new DOMXPath($dom);

        // First try frm9 div with h2 (the correct method)
        $frm9_h2 = $xpath->query('//div[@class="frm9"]//h2');
        if ($frm9_h2->length > 0) {
            $name = trim($frm9_h2->item(0)->textContent);
            if (!empty($name)) {
                return $name;
            }
        }

        // Fallback: Try multiple patterns to find the speaker name
        if (preg_match('/<h1[^>]*>([^<]+)<\/h1>/i', $html, $matches)) {
            $name = trim(strip_tags($matches[1]));
            if (!empty($name) && !stripos($name, 'rotary') && !stripos($name, 'speaker')) {
                return $name;
            }
        }

        // Try other patterns
        if (preg_match('/<title>([^<]*?)\s*[-|]\s*Rotary/i', $html, $matches)) {
            $name = trim(strip_tags($matches[1]));
            if (!empty($name)) {
                return $name;
            }
        }

        return '';
    }

    private function get_speaker_preview($speaker_id) {
        $url = "https://speakers.rotary5320.org/speakers/entry/{$speaker_id}/";

        $response = wp_remote_get($url, [
            'timeout' => 30,
            'user-agent' => 'WordPress Speaker Import Preview'
        ]);

        if (is_wp_error($response)) {
            throw new Exception("Failed to fetch speaker ID {$speaker_id}: " . $response->get_error_message());
        }

        $html = wp_remote_retrieve_body($response);

        // Check for "No Entries Found"
        if (stripos($html, 'no entries found') !== false) {
            throw new Exception("Speaker ID {$speaker_id} does not exist (No Entries Found)");
        }

        $speaker_data = $this->parse_speaker_html($html);

        if (empty($speaker_data['name'])) {
            throw new Exception("Could not extract speaker name from ID {$speaker_id}");
        }

        // Check if already exists
        $existing = null;
        if (!empty($speaker_data['email'])) {
            $existing = $this->find_existing_speaker($speaker_data['email']);
        }

        return $this->render_preview_card($speaker_id, $speaker_data, $existing);
    }

    private function render_preview_card($speaker_id, $data, $existing = null) {
        ob_start();
        ?>
        <div class="speaker-preview-card" style="border: 1px solid #ddd; padding: 20px; border-radius: 8px; background: #f9f9f9; margin: 20px 0;">
            <div style="display: flex; justify-content: space-between; align-items: flex-start; margin-bottom: 20px;">
                <h3 style="margin: 0; color: #1d2327;">
                    <?php echo esc_html($data['name'] ?: 'Unknown Speaker'); ?>
                    <span style="font-size: 14px; color: #646970; font-weight: normal;">(ID: <?php echo intval($speaker_id); ?>)</span>
                </h3>

                <?php if ($existing): ?>
                    <div class="notice notice-warning" style="margin: 0; padding: 5px 10px;">
                        <strong><?php esc_html_e('Already Exists!', 'speakers-bureau'); ?></strong><br>
                        <small><?php printf(__('Post ID: %d', 'speakers-bureau'), $existing->ID); ?></small>
                    </div>
                <?php endif; ?>
            </div>

            <div style="display: grid; grid-template-columns: 1fr 1fr; gap: 30px; margin-bottom: 20px;">
                <div>
                    <h4 style="margin-top: 0; color: #0073aa;"><?php esc_html_e('Extracted Data', 'speakers-bureau'); ?></h4>
                    <table class="form-table" style="margin: 0;">
                        <?php foreach ([
                            'name' => __('Name', 'speakers-bureau'),
                            'profile_image' => __('Profile Image', 'speakers-bureau'),
                            'email' => __('Email', 'speakers-bureau'),
                            'phone' => __('Phone', 'speakers-bureau'),
                            'job_title' => __('Job Title', 'speakers-bureau'),
                            'organization' => __('Organization', 'speakers-bureau'),
                            'website' => __('Website', 'speakers-bureau'),
                            'topics' => __('Topics', 'speakers-bureau'),
                            'summary' => __('Presentation Summary', 'speakers-bureau'),
                            'online_available' => __('Online Available', 'speakers-bureau'),
                            'city' => __('City', 'speakers-bureau'),
                            'state' => __('State', 'speakers-bureau'),
                            'country' => __('Country', 'speakers-bureau'),
                            'zip' => __('Zip Code', 'speakers-bureau'),
                        ] as $key => $label): ?>
                            <tr>
                                <td style="padding: 5px 10px 5px 0; font-weight: 600; width: 120px;"><?php echo esc_html($label); ?>:</td>
                                <td style="padding: 5px 0;">
                                    <?php
                                    $value = $data[$key] ?? '';
                                    if ($key === 'profile_image' && $value) {
                                        echo '<div><img src="' . esc_url($value) . '" alt="Profile" style="max-width: 80px; max-height: 80px; border-radius: 4px;"><br><small>' . esc_html($value) . '</small></div>';
                                    } elseif ($key === 'website' && $value) {
                                        echo '<a href="' . esc_url($value) . '" target="_blank">' . esc_html($value) . '</a>';
                                    } elseif ($key === 'email' && $value) {
                                        echo '<a href="mailto:' . esc_attr($value) . '">' . esc_html($value) . '</a>';
                                    } else {
                                        echo $value ? esc_html($value) : '<em style="color: #666;">' . __('Not found', 'speakers-bureau') . '</em>';
                                    }
                                    ?>
                                </td>
                            </tr>
                        <?php endforeach; ?>
                    </table>
                </div>

                <div>
                    <h4 style="margin-top: 0; color: #0073aa;"><?php esc_html_e('Field Mapping', 'speakers-bureau'); ?></h4>
                    <table class="form-table" style="margin: 0;">
                        <?php foreach ([
                            'name' => 'Post Title',
                            'profile_image' => 'profile_image',
                            'email' => 'email',
                            'phone' => 'phone',
                            'job_title' => 'headline',
                            'organization' => 'organization',
                            'website' => 'website',
                            'topics' => 'topics',
                            'summary' => 'summary',
                            'online_available' => 'online-mtg',
                            'city' => 'city',
                            'state' => 'state',
                            'country' => 'country',
                            'zip' => 'zip'
                        ] as $source => $target): ?>
                            <tr>
                                <td style="padding: 5px 10px 5px 0; font-weight: 600; width: 120px;"><?php echo esc_html($source); ?></td>
                                <td style="padding: 5px 0; color: #0073aa; font-family: monospace;">→ <?php echo esc_html($target); ?></td>
                            </tr>
                        <?php endforeach; ?>
                    </table>
                </div>
            </div>

            <?php if (!empty($data['summary'])): ?>
                <div style="margin-bottom: 20px;">
                    <h4 style="margin-bottom: 10px; color: #0073aa;"><?php esc_html_e('Presentation Summary', 'speakers-bureau'); ?></h4>
                    <div style="background: #fff; padding: 15px; border: 1px solid #ddd; border-radius: 4px; max-height: 150px; overflow-y: auto;">
                        <?php echo esc_html($data['summary']); ?>
                    </div>
                </div>
            <?php endif; ?>

            <div style="margin-bottom: 20px;">
                <h4 style="margin-bottom: 10px; color: #0073aa;"><?php esc_html_e('Social Media Links', 'speakers-bureau'); ?></h4>
                <?php if (!empty($data['facebookurl']) || !empty($data['linkedinurl']) || !empty($data['instagramurl']) || !empty($data['xurl']) || !empty($data['youtubeurl'])): ?>
                    <ul style="margin: 0; padding-left: 20px;">
                        <?php if (!empty($data['facebookurl'])): ?>
                            <li><strong>Facebook:</strong> <a href="<?php echo esc_url($data['facebookurl']); ?>" target="_blank"><?php echo esc_html($data['facebookurl']); ?></a></li>
                        <?php endif; ?>
                        <?php if (!empty($data['linkedinurl'])): ?>
                            <li><strong>LinkedIn:</strong> <a href="<?php echo esc_url($data['linkedinurl']); ?>" target="_blank"><?php echo esc_html($data['linkedinurl']); ?></a></li>
                        <?php endif; ?>
                        <?php if (!empty($data['instagramurl'])): ?>
                            <li><strong>Instagram:</strong> <a href="<?php echo esc_url($data['instagramurl']); ?>" target="_blank"><?php echo esc_html($data['instagramurl']); ?></a></li>
                        <?php endif; ?>
                        <?php if (!empty($data['xurl'])): ?>
                            <li><strong>X/Twitter:</strong> <a href="<?php echo esc_url($data['xurl']); ?>" target="_blank"><?php echo esc_html($data['xurl']); ?></a></li>
                        <?php endif; ?>
                        <?php if (!empty($data['youtubeurl'])): ?>
                            <li><strong>YouTube:</strong> <a href="<?php echo esc_url($data['youtubeurl']); ?>" target="_blank"><?php echo esc_html($data['youtubeurl']); ?></a></li>
                        <?php endif; ?>
                    </ul>
                <?php else: ?>
                    <p style="margin: 0; color: #666; font-style: italic;"><?php esc_html_e('No social media links found', 'speakers-bureau'); ?></p>
                <?php endif; ?>
            </div>

            <div style="border-top: 1px solid #ddd; padding-top: 20px; display: flex; justify-content: space-between; align-items: center;">
                <div>
                    <?php if (!$existing): ?>
                        <button type="button" class="button button-primary import-speaker-btn" data-id="<?php echo intval($speaker_id); ?>" data-action="import">
                            <?php esc_html_e('Import Speaker', 'speakers-bureau'); ?>
                        </button>
                        <button type="button" class="button import-speaker-btn" data-id="<?php echo intval($speaker_id); ?>" data-action="skip" style="margin-left: 10px;">
                            <?php esc_html_e('Skip', 'speakers-bureau'); ?>
                        </button>
                    <?php else: ?>
                        <span style="color: #d63638; font-weight: 600;"><?php esc_html_e('Cannot import - speaker already exists', 'speakers-bureau'); ?></span>
                    <?php endif; ?>
                </div>

                <a href="https://speakers.rotary5320.org/speakers/entry/<?php echo intval($speaker_id); ?>/" target="_blank" class="button button-link">
                    <?php esc_html_e('View Original', 'speakers-bureau'); ?> ↗
                </a>
            </div>

            <div class="import-result" style="margin-top: 15px;">
                <!-- Import results will appear here -->
            </div>
        </div>
        <?php
        return ob_get_clean();
    }

    private function import_single_speaker($speaker_id) {
        $url = "https://speakers.rotary5320.org/speakers/entry/{$speaker_id}/";

        // Fetch the speaker page
        $response = wp_remote_get($url, [
            'timeout' => 30,
            'user-agent' => 'WordPress Speaker Import Bot'
        ]);

        if (is_wp_error($response)) {
            throw new Exception("Failed to fetch speaker ID {$speaker_id}: " . $response->get_error_message());
        }

        $html = wp_remote_retrieve_body($response);

        // Check for "No Entries Found"
        if (stripos($html, 'no entries found') !== false) {
            throw new Exception("Speaker ID {$speaker_id} does not exist");
        }

        $speaker_data = $this->parse_speaker_html($html);

        if (empty($speaker_data['name'])) {
            throw new Exception("Could not extract speaker name from ID {$speaker_id}");
        }

        // Check if speaker already exists
        if (!empty($speaker_data['email'])) {
            $existing = $this->find_existing_speaker($speaker_data['email']);
            if ($existing) {
                throw new Exception("Speaker {$speaker_data['name']} already exists (Post ID: {$existing->ID})");
            }
        }

        // Import the speaker
        $post_id = $this->create_speaker_post($speaker_data);

        return [
            'post_id' => $post_id,
            'name' => $speaker_data['name']
        ];
    }

    private function parse_speaker_html($html) {
        $data = [];

        // Create DOMDocument for more reliable parsing
        $dom = new DOMDocument();
        @$dom->loadHTML($html, LIBXML_HTML_NOIMPLIED | LIBXML_HTML_NODEFDTD);
        $xpath = new DOMXPath($dom);

        // 1. Extract speaker name from first H2 tag (most reliable pattern)
        $h2_tags = $xpath->query('//h2');
        if ($h2_tags->length > 0) {
            $data['name'] = trim($h2_tags->item(0)->textContent);
        }

        // 2. Extract topics using multiple possible patterns
        $topics_found = false;

        // Try H4 with **Topic(s)**:
        $topic_h4 = $xpath->query('//h4[contains(text(), "**Topic(s)**:")]');
        if ($topic_h4->length > 0) {
            $topic_text = $topic_h4->item(0)->textContent;
            if (preg_match('/\*\*Topic\(s\)\*\*:\s*(.+)/i', $topic_text, $matches)) {
                $data['topics'] = trim($matches[1]);
                $topics_found = true;
            }
        }

        // Fallback: Try H4 containing just "Topic(s)"
        if (!$topics_found) {
            $topic_h4_alt = $xpath->query('//h4[contains(text(), "Topic(s)")]');
            if ($topic_h4_alt->length > 0) {
                $topic_text = $topic_h4_alt->item(0)->textContent;
                if (preg_match('/Topic\(s\)[:\s]*(.+)/i', $topic_text, $matches)) {
                    $data['topics'] = trim($matches[1]);
                    $topics_found = true;
                }
            }
        }

        // Fallback: Look for any element containing "District 5320 Presenter"
        if (!$topics_found) {
            $district_presenter = $xpath->query('//*[contains(text(), "District 5320 Presenter")]');
            if ($district_presenter->length > 0) {
                $data['topics'] = trim($district_presenter->item(0)->textContent);
                $topics_found = true;
            }
        }

        // Fallback: Look in any strong tag containing "Topic"
        if (!$topics_found) {
            $topic_strong = $xpath->query('//strong[contains(text(), "Topic")]');
            foreach ($topic_strong as $strong) {
                $parent = $strong->parentNode;
                if ($parent) {
                    $parent_text = trim($parent->textContent);
                    if (preg_match('/Topic[^:]*:\s*(.+)/i', $parent_text, $matches)) {
                        $data['topics'] = trim($matches[1]);
                        $topics_found = true;
                        break;
                    }
                }
            }
        }

        // Final fallback: Search the entire page text for topic patterns
        if (!$topics_found) {
            $full_text = $dom->textContent;
            // Look for lines that might contain topics using common patterns
            $lines = explode("\n", $full_text);
            foreach ($lines as $line) {
                $line = trim($line);
                // Look for lines that:
                // 1. Contain "District 5320 Presenter" (common starting pattern)
                // 2. Are comma-separated and look like topic lists
                // 3. Are reasonable length for topics
                if ((stripos($line, 'district 5320 presenter') !== false ||
                     (strpos($line, ',') !== false && strlen($line) > 20 && strlen($line) < 200)) &&
                    !stripos($line, 'summary') && !stripos($line, 'presentation') &&
                    !stripos($line, 'club') && !stripos($line, 'company')) {
                    // Additional check: line should look like a topic list
                    if (preg_match('/^[A-Za-z0-9\s,\(\)\-]+$/', $line)) {
                        $data['topics'] = $line;
                        break;
                    }
                }
            }
        }

        // 3. Extract fields using <strong> tag pattern
        $strong_tags = $xpath->query('//strong[contains(text(), ":")]');
        foreach ($strong_tags as $strong) {
            $label_text = trim($strong->textContent);

            // Get the content that follows this strong tag
            $next_content = '';
            $current_node = $strong;

            // Look for content in the same parent or following siblings
            while ($current_node) {
                $current_node = $current_node->nextSibling;
                if ($current_node) {
                    if ($current_node->nodeType === XML_TEXT_NODE) {
                        $next_content .= $current_node->textContent;
                    } elseif ($current_node->nodeType === XML_ELEMENT_NODE) {
                        // If it's another strong tag, stop
                        if ($current_node->nodeName === 'strong') break;
                        $next_content .= $current_node->textContent;
                    }
                }
            }

            // Clean up and extract the field value
            $field_value = trim($next_content);
            $field_value = preg_replace('/^\s*:\s*/', '', $field_value); // Remove leading colon

            // For phone fields, don't normalize spaces - preserve original formatting
            if (!in_array(strtolower($label_text), ['cell:', 'phone:', 'cell phone:', 'home phone:', 'home:', 'mobile:', 'office phone:', 'office:'])) {
                $field_value = preg_replace('/\s+/', ' ', $field_value);
            } else {
                $field_value = trim($field_value);
            }

            // Map based on the label
            switch (strtolower($label_text)) {
                case 'job title:':
                case 'title:':
                case 'position:':
                case 'role:':
                    $data['job_title'] = $field_value;
                    break;
                case 'company:':
                    $data['organization'] = $field_value;
                    break;
                case 'phone:':
                case 'home phone:':
                case 'home:':
                case 'cell:':
                case 'cell phone:':
                case 'mobile:':
                case 'office phone:':
                case 'office:':
                    // Map all phone types to the main phone field
                    if (!empty($field_value)) {
                        if (empty($data['phone'])) {
                            $data['phone'] = $field_value;
                        } else {
                            // If phone field already has a value, append additional numbers
                            $data['phone'] .= ' / ' . $field_value;
                        }
                    }
                    break;
                case 'email:':
                    $data['email'] = $field_value;
                    break;
                case 'website:':
                    $data['website'] = $field_value;
                    break;
                case 'availability:':
                    $data['availability'] = $field_value;
                    break;
                case 'contact info:':
                    // Parse contact info section for email, phone, website
                    $this->extract_contact_info($field_value, $data);
                    break;
                case 'social media:':
                    // Parse social media links
                    $this->extract_social_media($field_value, $data);
                    break;
                case 'summary of the presentation:':
                    $data['summary'] = $field_value;
                    break;
            }
        }

        // 4. Extract profile image - prioritize frm3 div structure
        $img_src = '';

        // Method 1: Primary - Look for image in div class="frm3" with data-src
        $frm3_img = $xpath->query('//div[@class="frm3"]//img[@data-src]');
        if ($frm3_img->length > 0) {
            $img_src = $frm3_img->item(0)->getAttribute('data-src');
        }

        // Method 2: Fallback - Look for image in div class="frm3" with src
        if (empty($img_src)) {
            $frm3_img_src = $xpath->query('//div[@class="frm3"]//img[@src]');
            if ($frm3_img_src->length > 0) {
                $img_src = $frm3_img_src->item(0)->getAttribute('src');
            }
        }

        // Method 3: General fallback - any image with data-src in uploads/formidable
        if (empty($img_src)) {
            $lazy_imgs = $xpath->query('//img[contains(@data-src, "/wp-content/uploads/") or contains(@data-src, "/formidable/")]');
            if ($lazy_imgs->length > 0) {
                $img_src = $lazy_imgs->item(0)->getAttribute('data-src');
            }
        }

        // Method 4: General fallback - any image with src in uploads/formidable
        if (empty($img_src)) {
            $std_imgs = $xpath->query('//img[contains(@src, "/wp-content/uploads/") or contains(@src, "/formidable/")]');
            if ($std_imgs->length > 0) {
                $img_src = $std_imgs->item(0)->getAttribute('src');
            }
        }

        if (!empty($img_src)) {
            // Convert relative URLs to absolute URLs
            if (strpos($img_src, '/') === 0) {
                $img_src = 'https://speakers.rotary5320.org' . $img_src;
            }
            $data['profile_image'] = $img_src;
        }

        // 5. Extract social media links from all anchor tags
        $links = $dom->getElementsByTagName('a');
        foreach ($links as $link) {
            $href = $link->getAttribute('href');

            if (strpos($href, 'mailto:') === 0) {
                $data['email'] = str_replace('mailto:', '', $href);
            } elseif (strpos($href, 'facebook.com') !== false) {
                $data['facebookurl'] = $href;
            } elseif (strpos($href, 'linkedin.com') !== false) {
                $data['linkedinurl'] = $href;
            } elseif (strpos($href, 'instagram.com') !== false) {
                $data['instagramurl'] = $href;
            } elseif (strpos($href, 'twitter.com') !== false || strpos($href, 'x.com') !== false) {
                $data['xurl'] = $href;
            } elseif (strpos($href, 'youtube.com') !== false) {
                $data['youtubeurl'] = $href;
            } elseif (strpos($href, 'http') === 0 && empty($data['website']) &&
                     !preg_match('/(facebook|linkedin|instagram|twitter|x\.com|youtube)/i', $href)) {
                $data['website'] = $href;
            }
        }

        // 6. Determine online availability (simple heuristic)
        if (!empty($data['availability']) &&
            (stripos($data['availability'], 'virtual') !== false ||
             stripos($data['availability'], 'online') !== false ||
             stripos($data['availability'], 'zoom') !== false)) {
            $data['online_available'] = 'YES';
        } else {
            $data['online_available'] = 'NO';
        }

        // 7. Check for Orange County location references and set location fields
        $full_text = $dom->textContent;
        if (stripos($full_text, 'south orange county') !== false ||
            stripos($full_text, 'north orange county') !== false) {
            $data['city'] = 'Orange';
            $data['state'] = 'CA';
            $data['country'] = 'USA';
            $data['zip'] = '92867';
        }

        // Clean up all text fields
        foreach ($data as $key => $value) {
            if (is_string($value)) {
                $data[$key] = trim($value);
            }
        }

        // Special cleanup for topics - remove any remaining "Topic(s):" prefix
        if (!empty($data['topics'])) {
            $data['topics'] = preg_replace('/^.*?Topic\(s\)[:\s]*/i', '', $data['topics']);
            $data['topics'] = preg_replace('/^\*\*Topic\(s\)\*\*[:\s]*/i', '', $data['topics']);
            $data['topics'] = trim($data['topics']);
        }

        // Specific job title extraction - text INSIDE the strong tag after "Job Title:"
        if (empty($data['job_title'])) {
            // Method 1: Look for <strong> containing "Job Title:" and extract what comes after the colon
            $job_title_nodes = $xpath->query('//strong[contains(text(), "Job Title:")]');
            if ($job_title_nodes->length > 0) {
                $strong_text = $job_title_nodes->item(0)->textContent;
                // Extract everything after "Job Title:" within the same strong tag
                if (preg_match('/Job Title:\s*(.+)/i', $strong_text, $matches)) {
                    $data['job_title'] = trim($matches[1]);
                }
            }
        }

        // Fallback: Try to find job title in raw HTML using regex
        if (empty($data['job_title'])) {
            // Get the original HTML content for regex matching
            $original_html = $dom->saveHTML();
            if (preg_match('/<strong[^>]*>Job Title:\s*([^<]+)<\/strong>/i', $original_html, $matches)) {
                $data['job_title'] = trim($matches[1]);
            }
        }

        return $data;
    }

    /**
     * Extract contact information from the contact info section
     */
    private function extract_contact_info($content, &$data) {
        // Look for email pattern
        if (preg_match('/Email:\s*([^\s\n]+)/i', $content, $matches)) {
            $data['email'] = trim($matches[1]);
        }

        // Look for all phone number patterns and consolidate into main phone field
        $phone_patterns = [
            '/Cell:\s*([^\n\r]+?)(?=\s*(?:Email:|Phone:|Home:|Office:|Website:|$))/i',
            '/Phone:\s*([^\n\r]+?)(?=\s*(?:Email:|Cell:|Home:|Office:|Website:|$))/i',
            '/Home:\s*([^\n\r]+?)(?=\s*(?:Email:|Phone:|Cell:|Office:|Website:|$))/i',
            '/Office:\s*([^\n\r]+?)(?=\s*(?:Email:|Phone:|Cell:|Home:|Website:|$))/i',
        ];

        foreach ($phone_patterns as $pattern) {
            if (preg_match($pattern, $content, $matches)) {
                $phone_number = trim($matches[1]);
                if (!empty($phone_number)) {
                    if (empty($data['phone'])) {
                        $data['phone'] = $phone_number;
                    } else {
                        // If phone field already has a value, append additional numbers
                        $data['phone'] .= ' / ' . $phone_number;
                    }
                }
            }
        }

        // Look for website pattern
        if (preg_match('/Website:\s*(https?:\/\/[^\s\n]+)/i', $content, $matches)) {
            $data['website'] = trim($matches[1]);
        }
    }

    /**
     * Extract social media links from the social media section
     */
    private function extract_social_media($content, &$data) {
        // Look for Facebook
        if (preg_match('/Facebook:\s*(https?:\/\/[^\s\n]+)/i', $content, $matches)) {
            $data['facebookurl'] = trim($matches[1]);
        }

        // Look for LinkedIn
        if (preg_match('/LinkedIn:\s*(https?:\/\/[^\s\n]+)/i', $content, $matches)) {
            $data['linkedinurl'] = trim($matches[1]);
        }

        // Look for Instagram
        if (preg_match('/Instagram:\s*(https?:\/\/[^\s\n]+)/i', $content, $matches)) {
            $data['instagramurl'] = trim($matches[1]);
        }
    }

    /**
     * Process topics - now just cleans up the raw topics without filtering
     */
    private function process_topics($raw_topics) {
        if (empty($raw_topics)) return '';

        // Just clean up whitespace and return as-is
        return trim($raw_topics);
    }

    /**
     * Determine online availability based on geographic regions
     */
    private function determine_online_availability($regions) {
        foreach ($regions as $region) {
            if (stripos($region, 'virtual') !== false ||
                stripos($region, 'zoom') !== false ||
                stripos($region, 'online') !== false) {
                return 'YES';
            }
        }
        return 'NO';
    }

    private function find_existing_speaker($email) {
        $args = [
            'post_type' => 'speaker',
            'meta_query' => [
                [
                    'key' => 'email',
                    'value' => $email,
                    'compare' => '='
                ]
            ],
            'posts_per_page' => 1
        ];

        $posts = get_posts($args);
        return !empty($posts) ? $posts[0] : null;
    }

    private function create_speaker_post($data) {
        // Create WordPress user for the speaker
        $user_id = $this->create_speaker_user($data);

        $post_data = [
            'post_type' => 'speaker',
            'post_title' => $data['name'],
            'post_status' => 'publish', // Import as published
            'post_content' => '', // Do not import any content into bio
            'post_author' => $user_id, // Link to created user
        ];

        $post_id = wp_insert_post($post_data);

        if (is_wp_error($post_id)) {
            throw new Exception('Failed to create speaker post: ' . $post_id->get_error_message());
        }

        // Add meta fields - using correct plugin field keys
        $meta_fields = [
            'profile_image' => $data['profile_image'] ?? '',  // Profile photo from frm3 div
            'email' => $data['email'] ?? '',
            'phone' => $data['phone'] ?? '',  // All phone types now consolidated into main phone field
            'website' => $data['website'] ?? '',
            'facebookurl' => $data['facebookurl'] ?? '',  // Updated to match plugin field names
            'linkedinurl' => $data['linkedinurl'] ?? '',
            'instagramurl' => $data['instagramurl'] ?? '',
            'xurl' => $data['xurl'] ?? '',
            'youtubeurl' => $data['youtubeurl'] ?? '',
            'organization' => $data['organization'] ?? '',
            'topics' => $data['topics'] ?? '',
            'summary' => $data['summary'] ?? '',  // Use summary instead of bio for presentation content
            'headline' => $data['job_title'] ?? '',
            'online-mtg' => $data['online_available'] ?? 'NO',  // Map online availability
            'city' => $data['city'] ?? '',  // Orange County location mapping
            'state' => $data['state'] ?? '',
            'country' => $data['country'] ?? '',
            'zip' => $data['zip'] ?? '',
        ];

        foreach ($meta_fields as $key => $value) {
            if (!empty($value)) {
                update_post_meta($post_id, $key, $value);
            }
        }

        // Link speaker post to the created user
        update_post_meta($post_id, '_speaker_user_id', $user_id);

        return $post_id;
    }

    /**
     * Create a WordPress user for the speaker
     */
    private function create_speaker_user($data) {
        $name = $data['name'] ?? '';
        $email = $data['email'] ?? '';

        if (empty($email)) {
            throw new Exception('Speaker email is required to create user account');
        }

        // Check if user already exists by email
        $existing_user = get_user_by('email', $email);
        if ($existing_user) {
            return $existing_user->ID;
        }

        // Generate username from speaker name
        $username = $this->generate_username($name);

        // Generate a random password
        $password = wp_generate_password(12, false);

        // Create the user without sending email
        add_filter('send_password_change_email', '__return_false');
        add_filter('send_email_change_email', '__return_false');

        $user_data = [
            'user_login' => $username,
            'user_email' => $email,
            'user_pass' => $password,
            'display_name' => $name,
            'first_name' => '',
            'last_name' => '',
            'role' => 'speaker', // Assign speaker role
        ];

        // Split name into first and last if possible
        $name_parts = explode(' ', $name, 2);
        if (count($name_parts) >= 2) {
            $user_data['first_name'] = $name_parts[0];
            $user_data['last_name'] = $name_parts[1];
        } else {
            $user_data['first_name'] = $name;
        }

        $user_id = wp_insert_user($user_data);

        // Remove the email filters
        remove_filter('send_password_change_email', '__return_false');
        remove_filter('send_email_change_email', '__return_false');

        if (is_wp_error($user_id)) {
            throw new Exception('Failed to create user: ' . $user_id->get_error_message());
        }

        return $user_id;
    }

    /**
     * Generate a unique username from speaker name
     */
    private function generate_username($name) {
        // Clean and sanitize the name
        $username = sanitize_user(strtolower(str_replace(' ', '', $name)));

        // Ensure it's not empty
        if (empty($username)) {
            $username = 'speaker';
        }

        // Check if username exists and make it unique
        $original_username = $username;
        $counter = 1;

        while (username_exists($username)) {
            $username = $original_username . $counter;
            $counter++;
        }

        return $username;
    }
}

// Initialize the import system
new SB_Rotary_Import();