<?php
/**
 * File: includes/sb-migrations.php
 * Purpose: One-shot migrations/utilities executed ONLY on plugin activation.
 *
 * UPDATE (2025-09-20):
 * - Adds SB_Migrations::run() which normalizes sb_profile_layout and sb_form_fields.
 * - No global hooks; nothing runs on every request. Activation-only.
 */

if (!defined('ABSPATH')) exit;

if (!class_exists('SB_Migrations')) {

class SB_Migrations {

    /** Entry point for activation hook */
    public static function run() {
        // Only run migrations during actual plugin activation, not during normal admin operations
        if (!defined('WP_ACTIVATING_PLUGIN') &&
            !doing_action('activate_plugin') &&
            !doing_action('activate_' . plugin_basename(SB_PLUGIN_FILE))) {
            error_log('SB_Migrations::run() called outside activation context - skipping');
            return;
        }

        self::fix_profile_layout_option();
        self::maybe_migrate_form_fields();
        self::maybe_seed_email_templates();
    }

    /**
     * Normalize the sb_profile_layout option so it always stores exactly 8 sections,
     * indexed 1..8, and without duplicate blocks.
     */
    public static function fix_profile_layout_option() {
        $opt_name = 'sb_profile_layout';
        $raw = get_option($opt_name);

        // If empty or not array, seed defaults and bail.
        if (!is_array($raw) || empty($raw)) {
            update_option($opt_name, self::default_profile_layout(), false);
            error_log('SB_Migrations::fix_profile_layout_option() seeded defaults for sb_profile_layout');
            return;
        }

        // Some sites accidentally accumulated 40 entries (duplicated sets of 8).
        // We’ll compress down to the first 8 blocks and overlay minimal fields.
        $defaults = self::default_profile_layout();
        $normalized = [];

        // Prefer first 8 entries from the *end* or *start*? Choose the first valid 8 in order 1..8.
        // We build a “best guess” map from the raw data (by stepping through and overlaying onto slots 1..8).
        for ($i = 1; $i <= 8; $i++) {
            // Try exact numeric key first
            if (isset($raw[$i]) && is_array($raw[$i])) {
                $row = $raw[$i];
            } else {
                // Otherwise try to pick the next available raw row that matches the default class
                $row = self::find_row_by_class($raw, $defaults[$i]['class']);
            }

            $heading = isset($row['heading']) ? strtolower(sanitize_text_field($row['heading'])) : $defaults[$i]['heading'];
            if (!in_array($heading, ['h1','h2','h3','h4','h5','h6'], true)) {
                $heading = $defaults[$i]['heading'];
            }

            $normalized[$i] = [
                'title'   => isset($row['title']) ? wp_kses_post($row['title']) : $defaults[$i]['title'],
                'heading' => $heading,
                'display' => !empty($row['display']) ? 1 : 0,
                'class'   => isset($row['class']) ? sanitize_html_class($row['class']) : $defaults[$i]['class'],
                'hidden'  => !empty($row['hidden']) ? 1 : 0,
            ];
        }

        update_option($opt_name, $normalized, false);
        error_log('SB_Migrations::fix_profile_layout_option() normalized sb_profile_layout');
    }

    /** Helper to find a row by CSS class inside a messy/duplicated array */
    private static function find_row_by_class($raw, $class) {
        if (!is_array($raw)) return [];
        foreach ($raw as $row) {
            if (is_array($row) && !empty($row['class']) && $row['class'] === $class) {
                return $row;
            }
        }
        return [];
    }

    /** Defaults copied local so we don’t depend on settings class during activation */
    private static function default_profile_layout() {
        return [
            1 => ['title'=>'Profile','heading'=>'h4','display'=>1,'class'=>'sb-section-1','hidden'=>0],
            2 => ['title'=>'Overview','heading'=>'h2','display'=>1,'class'=>'sb-section-2','hidden'=>0],
            3 => ['title'=>'Topics','heading'=>'h3','display'=>1,'class'=>'sb-section-3','hidden'=>0],
            4 => ['title'=>'Background','heading'=>'h3','display'=>1,'class'=>'sb-section-4','hidden'=>0],
            5 => ['title'=>'','heading'=>'h3','display'=>0,'class'=>'sb-section-5','hidden'=>1],
            6 => ['title'=>'','heading'=>'h3','display'=>0,'class'=>'sb-section-6','hidden'=>1],
            7 => ['title'=>'More Information','heading'=>'h2','display'=>1,'class'=>'sb-section-7','hidden'=>0],
            8 => ['title'=>'Private Fields','heading'=>'h3','display'=>0,'class'=>'sb-section-8','hidden'=>1],
        ];
    }

    /**
     * If sb_form_fields is stored as JSON string (older builds) or missing,
     * convert it to array and seed defaults if needed.
     */
    public static function maybe_migrate_form_fields() {
        $name = 'sb_form_fields';
        $val  = get_option($name);

        if (is_string($val)) {
            $decoded = json_decode($val, true);
            if (is_array($decoded)) {
                update_option($name, $decoded, false);
                error_log('SB_Migrations::maybe_migrate_form_fields() converted JSON to array');
                return;
            }
        }

        if (!is_array($val) || empty($val)) {
            update_option($name, self::default_form_fields(), false);
            error_log('SB_Migrations::maybe_migrate_form_fields() seeded default form fields');
        }
    }

    private static function default_form_fields() {
        return [
            [
                "label" => "Profile Image",
                "key" => "profile_image",
                "type" => "image",
                "placeholder" => "",
                "enabled" => 1,
                "required" => 0,
                "hide_label" => 0,
                "section" => 1,
                "searchable" => 0
            ],
            [
                "label" => "Headline",
                "key" => "headline",
                "type" => "text",
                "placeholder" => "e.g., Keynote Speaker on Leadership",
                "enabled" => 1,
                "required" => 0,
                "hide_label" => 0,
                "section" => 1,
                "searchable" => 1
            ],
            [
                "label" => "Organization",
                "key" => "organization",
                "type" => "text",
                "placeholder" => "Your company name",
                "enabled" => 1,
                "required" => 0,
                "hide_label" => 0,
                "section" => 1,
                "searchable" => 1
            ],
            [
                "label" => "Topics",
                "key" => "topics",
                "type" => "text",
                "placeholder" => "Youth, Camping, Topics (Comma Separated)",
                "enabled" => 1,
                "required" => 0,
                "hide_label" => 0,
                "section" => 2,
                "searchable" => 1
            ],
            [
                "label" => "Bio",
                "key" => "bio",
                "type" => "textarea",
                "placeholder" => "The speaker's personal biography or background",
                "enabled" => 1,
                "required" => 0,
                "hide_label" => 0,
                "section" => 2,
                "searchable" => 1
            ],
            [
                "label" => "Email",
                "key" => "email",
                "type" => "email",
                "placeholder" => "Your personal email address (for registering on this website)",
                "enabled" => 1,
                "required" => 0,
                "hide_label" => 0,
                "section" => 2,
                "searchable" => 1
            ],
            [
                "label" => "Phone",
                "key" => "phone",
                "type" => "tel",
                "placeholder" => "(123) 456-7890",
                "enabled" => 1,
                "required" => 0,
                "hide_label" => 0,
                "section" => 2,
                "searchable" => 1
            ],
            [
                "label" => "Your Website",
                "key" => "website",
                "type" => "url",
                "placeholder" => "Speaker's website - url starts with https://",
                "enabled" => 1,
                "required" => 0,
                "hide_label" => 0,
                "section" => 2,
                "searchable" => 1
            ],
            [
                "label" => "Cell Phone",
                "key" => "cell_phone",
                "type" => "text",
                "placeholder" => "Your Cell phonumber (if you want this to be public)",
                "enabled" => 1,
                "required" => 0,
                "hide_label" => 0,
                "section" => 2,
                "searchable" => 0
            ],
            [
                "label" => "Edit This Profile",
                "key" => "placeholder",
                "type" => "edit_link",
                "placeholder" => "",
                "enabled" => 1,
                "required" => 0,
                "hide_label" => 0,
                "section" => 1,
                "searchable" => 0
            ],
            [
                "label" => "Topic Image",
                "key" => "topic_image",
                "type" => "image",
                "placeholder" => "",
                "enabled" => 1,
                "required" => 0,
                "hide_label" => 0,
                "section" => 2,
                "searchable" => 0
            ],
            [
                "label" => "Presentation Summary",
                "key" => "presentation_summary",
                "type" => "textarea",
                "placeholder" => "Summary or overview of the presentation topic",
                "enabled" => 1,
                "required" => 0,
                "hide_label" => 0,
                "section" => 2,
                "searchable" => 1
            ],
            [
                "label" => "City",
                "key" => "city",
                "type" => "text",
                "placeholder" => "The city where you are located",
                "enabled" => 1,
                "required" => 0,
                "hide_label" => 0,
                "section" => 1,
                "searchable" => 1
            ],
            [
                "label" => "State",
                "key" => "state",
                "type" => "text",
                "placeholder" => "The 2-character state code where you are located eg \"CA\" or \"NV\" etc.",
                "enabled" => 1,
                "required" => 0,
                "hide_label" => 0,
                "section" => 1,
                "searchable" => 0
            ],
            [
                "label" => "Zip Code",
                "key" => "zip_code",
                "type" => "text",
                "placeholder" => "5-Digit zip code",
                "enabled" => 1,
                "required" => 0,
                "hide_label" => 0,
                "section" => 1,
                "searchable" => 1
            ],
            [
                "label" => "Available for Online",
                "key" => "available_online",
                "type" => "select",
                "placeholder" => "YES, NO",
                "enabled" => 1,
                "required" => 0,
                "hide_label" => 0,
                "section" => 1,
                "searchable" => 0
            ],
            [
                "label" => "Country",
                "key" => "country",
                "type" => "select",
                "placeholder" => "USA, CANADA, OTHER",
                "enabled" => 1,
                "required" => 0,
                "hide_label" => 0,
                "section" => 1,
                "searchable" => 0
            ],
            [
                "label" => "Facebook URL",
                "key" => "facebook_url",
                "type" => "url",
                "placeholder" => "url starts with https://www.facebook.com/",
                "enabled" => 1,
                "required" => 0,
                "hide_label" => 0,
                "section" => 7,
                "searchable" => 1
            ],
            [
                "label" => "X/Twitter URL",
                "key" => "x_twitter_url",
                "type" => "url",
                "placeholder" => "url starts with https://x.com/",
                "enabled" => 1,
                "required" => 0,
                "hide_label" => 0,
                "section" => 7,
                "searchable" => 1
            ],
            [
                "label" => "YouTube URL",
                "key" => "youtube_url",
                "type" => "url",
                "placeholder" => "url starts with https://",
                "enabled" => 1,
                "required" => 0,
                "hide_label" => 0,
                "section" => 7,
                "searchable" => 1
            ]
        ];
    }

    /**
     * Set up default email templates if they don't exist
     */
    public static function maybe_seed_email_templates() {
        $templates = get_option('sb_email_templates', []);

        if (empty($templates) || !is_array($templates)) {
            $default_templates = [
                'speaker_confirmation' => 'Hello {speaker_name},

Thank you for registering as a speaker! Your profile has been received and is now live on our speakers directory.

You can view your public profile at: {public_profile_url}

To edit your profile at any time, please visit: {profile_url}

If you need to log in to make changes, visit: {login_url}

Thank you for joining our speakers bureau!

Best regards,
{site_name} Team

🤖 Generated with Claude Code
https://claude.ai/code',

                'admin_notification' => 'New Speaker Registration

A new speaker has registered on {site_name}:

Speaker: {speaker_name}
Profile: {public_profile_url}
Edit Link: {profile_url}

Please review and verify the information is appropriate for your speakers bureau.

Admin Dashboard: ' . admin_url('edit.php?post_type=speaker') . '

🤖 Generated with Claude Code
https://claude.ai/code'
            ];

            update_option('sb_email_templates', $default_templates, false);
            error_log('SB_Migrations::maybe_seed_email_templates() seeded default email templates');
        }
    }
}

} // class_exists guard

/** END OF FILE: includes/sb-migrations.php */
