<?php
/**
 * File: includes/sb-map.php
 * Purpose: Speaker location map functionality using LeafletJS
 */

if (!defined('ABSPATH')) exit;

/**
 * Enqueue LeafletJS assets
 */
function sb_map_enqueue_scripts() {
    global $post;

    // Check if the speaker_map shortcode is present
    if (is_a($post, 'WP_Post') && has_shortcode($post->post_content, 'speaker_map')) {
        // Enqueue Leaflet CSS
        wp_enqueue_style(
            'leaflet-css',
            'https://unpkg.com/leaflet@1.9.4/dist/leaflet.css',
            [],
            '1.9.4'
        );

        // Enqueue Leaflet JS
        wp_enqueue_script(
            'leaflet-js',
            'https://unpkg.com/leaflet@1.9.4/dist/leaflet.js',
            [],
            '1.9.4',
            true
        );

        // Enqueue custom map JS
        wp_enqueue_script(
            'sb-map-js',
            SB_PLUGIN_URL . 'assets/sb-map.js',
            ['leaflet-js'],
            SB_PLUGIN_VER,
            true
        );

        // Enqueue map CSS
        wp_enqueue_style(
            'sb-map-css',
            SB_PLUGIN_URL . 'assets/sb-map.css',
            ['leaflet-css'],
            SB_PLUGIN_VER
        );
    }
}
add_action('wp_enqueue_scripts', 'sb_map_enqueue_scripts');

/**
 * Shortcode: [speaker_map]
 * Display speakers on a map
 *
 * @param array $atts Shortcode attributes
 *   - height: Map height (default: 600px)
 *   - width: Map width (default: 100%)
 *   - zoom: Initial zoom level (default: 4)
 *   - center_lat: Center latitude (default: auto-calculated)
 *   - center_lng: Center longitude (default: auto-calculated)
 */
function sb_speaker_map_shortcode($atts) {
    $atts = shortcode_atts([
        'height' => '600px',
        'width' => '100%',
        'zoom' => 4,
        'center_lat' => '',
        'center_lng' => '',
    ], $atts, 'speaker_map');

    // Generate unique ID for this map instance
    $map_id = 'sb-speaker-map-' . uniqid();

    // Query all published speakers with location data
    $speakers_query = new WP_Query([
        'post_type' => 'speaker',
        'post_status' => 'publish',
        'posts_per_page' => -1,
        'meta_query' => [
            'relation' => 'AND',
            [
                'key' => 'geo_lat',
                'compare' => 'EXISTS',
            ],
            [
                'key' => 'geo_lng',
                'compare' => 'EXISTS',
            ],
        ],
    ]);

    $markers = [];
    $location_groups = []; // Group markers by location
    $total_lat = 0;
    $total_lng = 0;
    $count = 0;

    if ($speakers_query->have_posts()) {
        while ($speakers_query->have_posts()) {
            $speakers_query->the_post();
            $speaker_id = get_the_ID();

            $lat = get_post_meta($speaker_id, 'geo_lat', true);
            $lng = get_post_meta($speaker_id, 'geo_lng', true);

            // Skip if coordinates are invalid
            if (empty($lat) || empty($lng) || !is_numeric($lat) || !is_numeric($lng)) {
                continue;
            }

            $lat = floatval($lat);
            $lng = floatval($lng);

            // Get speaker data
            $speaker_name = get_the_title();
            $speaker_url = get_permalink();

            // Get profile image
            $profile_image = sb_get_speaker_profile_image($speaker_id);
            $image_html = '';
            if ($profile_image) {
                $image_html = sprintf(
                    '<div class="sb-map-image"><img src="%s" alt="%s"></div>',
                    esc_url($profile_image),
                    esc_attr($speaker_name)
                );
            }

            // Get headline
            $headline = sb_get_speaker_headline($speaker_id);
            $headline_html = '';
            if ($headline) {
                $headline_html = '<div class="sb-map-headline">' . esc_html($headline) . '</div>';
            }

            // Get organization
            $organization = sb_get_speaker_organization($speaker_id);
            $organization_html = '';
            if ($organization) {
                $organization_html = '<div class="sb-map-organization">' . esc_html($organization) . '</div>';
            }

            // Get topics from form fields
            $topics = sb_get_speaker_topics($speaker_id);
            $topics_html = '';
            if (!empty($topics)) {
                $topics_html = '<div class="sb-map-topics"><strong>Topics:</strong> ' . esc_html(implode(', ', $topics)) . '</div>';
            }

            // Build popup content
            $popup_content = sprintf(
                '<div class="sb-map-popup">
                    %s
                    <div class="sb-map-content">
                        <h3 class="sb-map-speaker-name"><a href="%s">%s</a></h3>
                        %s
                        %s
                        %s
                        <div class="sb-map-profile-link">
                            <a href="%s" class="sb-map-button">View Profile</a>
                        </div>
                    </div>
                </div>',
                $image_html,
                esc_url($speaker_url),
                esc_html($speaker_name),
                $headline_html,
                $organization_html,
                $topics_html,
                esc_url($speaker_url)
            );

            // Create a unique key for this location (rounded to 4 decimal places)
            $location_key = round($lat, 4) . ',' . round($lng, 4);

            // Group markers by location
            if (!isset($location_groups[$location_key])) {
                $location_groups[$location_key] = [];
            }

            $location_groups[$location_key][] = [
                'lat' => $lat,
                'lng' => $lng,
                'popup' => $popup_content,
                'name' => $speaker_name,
            ];

            $total_lat += $lat;
            $total_lng += $lng;
            $count++;
        }
        wp_reset_postdata();
    }

    // Apply offset to markers at the same location
    foreach ($location_groups as $location_key => $group) {
        $group_count = count($group);

        if ($group_count === 1) {
            // Single marker, no offset needed
            $markers[] = $group[0];
        } else {
            // Multiple markers at same location - apply circular offset
            $offset_distance = 0.002; // ~200 meters offset

            foreach ($group as $index => $marker_data) {
                // Calculate offset angle for circular distribution
                $angle = (2 * M_PI * $index) / $group_count;

                // Apply offset
                $offset_lat = $marker_data['lat'] + ($offset_distance * cos($angle));
                $offset_lng = $marker_data['lng'] + ($offset_distance * sin($angle));

                $markers[] = [
                    'lat' => $offset_lat,
                    'lng' => $offset_lng,
                    'popup' => $marker_data['popup'],
                ];
            }
        }
    }

    // If no speakers with coordinates found
    if (empty($markers)) {
        return '<div class="sb-map-notice">No speakers with location data found.</div>';
    }

    // Calculate center if not provided
    $center_lat = !empty($atts['center_lat']) ? floatval($atts['center_lat']) : ($total_lat / $count);
    $center_lng = !empty($atts['center_lng']) ? floatval($atts['center_lng']) : ($total_lng / $count);

    // Prepare map data
    $map_data = [
        'markers' => $markers,
        'center' => [
            'lat' => $center_lat,
            'lng' => $center_lng,
        ],
        'zoom' => intval($atts['zoom']),
    ];

    // Sanitize dimensions
    $height = esc_attr($atts['height']);
    $width = esc_attr($atts['width']);

    // Output map container with inline data
    ob_start();
    ?>
    <div id="<?php echo esc_attr($map_id); ?>"
         class="sb-speaker-map"
         style="height: <?php echo $height; ?>; width: <?php echo $width; ?>;"
         data-map-config='<?php echo wp_json_encode($map_data); ?>'>
    </div>
    <?php
    return ob_get_clean();
}
add_shortcode('speaker_map', 'sb_speaker_map_shortcode');

/**
 * Helper function to extract speaker topics
 *
 * @param int $speaker_id Speaker post ID
 * @return array Array of topics
 */
function sb_get_speaker_topics($speaker_id) {
    $topics = [];

    // Get form fields configuration
    $form_fields = sb_normalize_form_fields(get_option('sb_form_fields', []));

    foreach ($form_fields as $field) {
        if (empty($field['enabled']) || empty($field['key'])) {
            continue;
        }

        $label = isset($field['label']) ? strtolower($field['label']) : '';

        // Look for topic-related fields
        if (strpos($label, 'topic') !== false ||
            strpos($label, 'subject') !== false ||
            strpos($label, 'specialt') !== false ||
            strpos($label, 'expertise') !== false) {

            $value = get_post_meta($speaker_id, $field['key'], true);

            if (!empty($value)) {
                if (is_array($value)) {
                    $topics = array_merge($topics, array_filter($value));
                } elseif (is_string($value)) {
                    // Split by common delimiters
                    $split_topics = preg_split('/[,;\n\r]+/', $value);
                    $split_topics = array_map('trim', $split_topics);
                    $topics = array_merge($topics, array_filter($split_topics));
                }
            }
        }
    }

    return array_unique(array_filter($topics));
}

/**
 * Helper function to get speaker profile image
 *
 * @param int $speaker_id Speaker post ID
 * @return string|false Image URL or false if not found
 */
function sb_get_speaker_profile_image($speaker_id) {
    // Get form fields configuration
    $form_fields = sb_normalize_form_fields(get_option('sb_form_fields', []));

    foreach ($form_fields as $field) {
        if (empty($field['enabled']) || empty($field['key'])) {
            continue;
        }

        $label = isset($field['label']) ? strtolower($field['label']) : '';
        $type = isset($field['type']) ? $field['type'] : '';

        // Look for image/photo fields
        if ($type === 'image' && (
            strpos($label, 'photo') !== false ||
            strpos($label, 'image') !== false ||
            strpos($label, 'picture') !== false ||
            strpos($label, 'headshot') !== false ||
            strpos($label, 'profile') !== false
        )) {
            $value = get_post_meta($speaker_id, $field['key'], true);

            if (!empty($value)) {
                // Check if it's an attachment ID
                if (is_numeric($value)) {
                    $image_url = wp_get_attachment_image_url($value, 'thumbnail');
                    if ($image_url) {
                        return $image_url;
                    }
                } elseif (filter_var($value, FILTER_VALIDATE_URL)) {
                    return $value;
                }
            }
        }
    }

    // Fallback to WordPress featured image
    if (has_post_thumbnail($speaker_id)) {
        return get_the_post_thumbnail_url($speaker_id, 'thumbnail');
    }

    // Fallback to default image if set
    $default_image = get_option('sb_default_image');
    if ($default_image) {
        return $default_image;
    }

    return false;
}

/**
 * Helper function to get speaker headline
 *
 * @param int $speaker_id Speaker post ID
 * @return string|false Headline or false if not found
 */
function sb_get_speaker_headline($speaker_id) {
    // Get form fields configuration
    $form_fields = sb_normalize_form_fields(get_option('sb_form_fields', []));

    foreach ($form_fields as $field) {
        if (empty($field['enabled']) || empty($field['key'])) {
            continue;
        }

        $label = isset($field['label']) ? strtolower($field['label']) : '';

        // Look for headline-related fields
        if (strpos($label, 'headline') !== false ||
            strpos($label, 'title') !== false ||
            strpos($label, 'tagline') !== false ||
            strpos($label, 'subtitle') !== false) {

            $value = get_post_meta($speaker_id, $field['key'], true);

            if (!empty($value) && is_string($value)) {
                return trim($value);
            }
        }
    }

    return false;
}

/**
 * Helper function to get speaker organization
 *
 * @param int $speaker_id Speaker post ID
 * @return string|false Organization or false if not found
 */
function sb_get_speaker_organization($speaker_id) {
    // Get form fields configuration
    $form_fields = sb_normalize_form_fields(get_option('sb_form_fields', []));

    foreach ($form_fields as $field) {
        if (empty($field['enabled']) || empty($field['key'])) {
            continue;
        }

        $label = isset($field['label']) ? strtolower($field['label']) : '';

        // Look for organization-related fields
        if (strpos($label, 'organization') !== false ||
            strpos($label, 'company') !== false ||
            strpos($label, 'employer') !== false ||
            strpos($label, 'business') !== false ||
            strpos($label, 'club') !== false ||
            strpos($label, 'rotary') !== false) {

            $value = get_post_meta($speaker_id, $field['key'], true);

            if (!empty($value) && is_string($value)) {
                return trim($value);
            }
        }
    }

    return false;
}

/** END OF FILE: includes/sb-map.php */
