<?php
/**
 * File: includes/sb-geocoding.php
 * Purpose: Part of the Speakers Bureau plugin.
 */

if (!defined('ABSPATH')) exit;

function sb_geocode_address($address, $country = '') {
    // Build query parameters
    $query_params = [
        'q' => $address,
        'format' => 'json',
        'limit' => 1
    ];

    // Add country code if provided (helps with accuracy)
    if (!empty($country)) {
        $query_params['countrycodes'] = strtoupper($country);
    }

    $api_url = "https://nominatim.openstreetmap.org/search?" . http_build_query($query_params);
    $response = wp_remote_get($api_url, ['headers'=>['User-Agent'=>'Speakers Bureau Plugin']]);
    if (is_wp_error($response)) return false;
    $data = json_decode(wp_remote_retrieve_body($response), true);
    if (!empty($data[0])) {
        return ['lat'=>$data[0]['lat'], 'lng'=>$data[0]['lon']];
    }
    return false;
}

/**
 * Main geocoding function for a speaker
 * Separated so it can be called from multiple hooks
 */
function sb_geocode_speaker($post_id) {
    // Skip if autosave
    if (defined('DOING_AUTOSAVE') && DOING_AUTOSAVE) return;

    // Verify it's a speaker post
    if (get_post_type($post_id) !== 'speaker') return;

    // Get address components - check multiple possible field names
    // Street address (optional, may not exist)
    $address = get_post_meta($post_id, 'address', true);
    if (empty($address)) $address = get_post_meta($post_id, 'sb_address', true);

    // City - check multiple possible field names
    $city = get_post_meta($post_id, 'city', true);
    if (empty($city)) $city = get_post_meta($post_id, 'sb_city', true);

    // State - check multiple possible field names
    $state = get_post_meta($post_id, 'state', true);
    if (empty($state)) $state = get_post_meta($post_id, 'sb_state', true);

    // Zip - check multiple possible field names
    $zip = get_post_meta($post_id, 'zip', true);
    if (empty($zip)) $zip = get_post_meta($post_id, 'zip_code', true);
    if (empty($zip)) $zip = get_post_meta($post_id, 'sb_zip', true);
    if (empty($zip)) $zip = get_post_meta($post_id, 'postal_code', true);

    // Try to get country from various possible field names
    $country = get_post_meta($post_id, 'country', true);
    if (empty($country)) $country = get_post_meta($post_id, 'sb_country', true);

    // If no country specified, use default from settings
    if (empty($country)) {
        $settings = get_option('sb_settings', []);
        $country = isset($settings['geo_default_country']) ? $settings['geo_default_country'] : 'US';
    }

    // Build address string (do NOT include country in address - only use as filter parameter)
    $parts = array_filter([$address, $city, $state, $zip]);
    if (!$parts) return;
    $full_address = implode(', ', $parts);

    // Geocode with country parameter for better accuracy
    $coords = sb_geocode_address($full_address, $country);
    if ($coords) {
        update_post_meta($post_id, 'geo_lat', $coords['lat']);
        update_post_meta($post_id, 'geo_lng', $coords['lng']);
        // Also update old format for backward compatibility
        update_post_meta($post_id, '_sb_lat', $coords['lat']);
        update_post_meta($post_id, '_sb_lng', $coords['lng']);
    }
}

// Geocode on save - use priority 50 to run AFTER meta boxes save their data (which run at priority 10)
add_action('save_post_speaker', 'sb_geocode_speaker', 50);

// Also geocode when address fields are updated directly
add_action('updated_post_meta', function($meta_id, $post_id, $meta_key, $meta_value) {
    // Check if this is an address-related field
    $address_fields = ['city', 'sb_city', 'state', 'sb_state', 'zip', 'zip_code', 'sb_zip', 'postal_code', 'address', 'sb_address', 'country', 'sb_country'];

    if (in_array($meta_key, $address_fields)) {
        // Make sure it's a speaker post
        if (get_post_type($post_id) === 'speaker') {
            // Prevent infinite loop by checking if we're not already updating geo coordinates
            if (!in_array($meta_key, ['geo_lat', 'geo_lng', '_sb_lat', '_sb_lng'])) {
                sb_geocode_speaker($post_id);
            }
        }
    }
}, 10, 4);

/** END OF FILE: includes/sb-geocoding.php */
