<?php
/**
 * File: includes/sb-columns.php
 * Purpose: Part of the Speakers Bureau plugin.
 */

if (!defined('ABSPATH')) exit;

function sb_add_speaker_columns($columns) {
    $columns['linked_user'] = __('Linked User','sb');
    $columns['speaker_views'] = __('Profile Views','sb');
    $columns['last_viewed'] = __('Last Viewed','sb');
    return $columns;
}
add_filter('manage_speaker_posts_columns', 'sb_add_speaker_columns');

function sb_render_speaker_columns($column, $post_id) {
    switch ($column) {
        case 'linked_user':
            $user_id = get_post_meta($post_id, '_speaker_user_id', true);
            if ($user_id) {
                $user = get_user_by('id', $user_id);
                if ($user) {
                    echo esc_html($user->display_name) . ' (' . esc_html($user->user_email) . ')';
                } else {
                    echo '<em>'.esc_html__('User not found','sb').'</em>';
                }
            } else {
                echo '<em>'.esc_html__('Not linked','sb').'</em>';
            }
            break;

        case 'speaker_views':
            $views = get_post_meta($post_id, '_speaker_views', true);
            $views = $views ? intval($views) : 0;
            echo '<div class="sb-views-editor">';
            echo '<span class="sb-views-display" onclick="sbEditViews(' . $post_id . ', this)">';
            echo '<strong>' . number_format($views) . '</strong>';
            if ($views > 0) {
                echo '<br><small style="color: #646970;">views</small>';
            }
            echo '<br><small style="color: #2271b1; cursor: pointer;">✏️ Edit</small>';
            echo '</span>';
            echo '<span class="sb-views-editor-form" style="display: none;">';
            echo '<input type="number" min="0" value="' . $views . '" style="width: 80px;">';
            echo '<br><button onclick="sbSaveViews(' . $post_id . ', this)" class="button button-small">Save</button>';
            echo ' <button onclick="sbCancelEditViews(this)" class="button button-small">Cancel</button>';
            echo '</span>';
            echo '</div>';
            break;

        case 'last_viewed':
            $last_viewed = get_post_meta($post_id, '_speaker_last_viewed', true);
            if ($last_viewed) {
                $date = new DateTime($last_viewed);
                $now = new DateTime();
                $diff = $now->diff($date);

                if ($diff->days == 0) {
                    echo '<span style="color: #00a32a;">Today</span>';
                } elseif ($diff->days == 1) {
                    echo '<span style="color: #dba617;">Yesterday</span>';
                } elseif ($diff->days <= 7) {
                    echo '<span style="color: #dba617;">' . $diff->days . ' days ago</span>';
                } else {
                    echo '<span style="color: #646970;">' . $date->format('M j, Y') . '</span>';
                }
                echo '<br><small style="color: #646970;">' . $date->format('g:i A') . '</small>';
            } else {
                echo '<em style="color: #646970;">Never</em>';
            }
            break;
    }
}
add_action('manage_speaker_posts_custom_column', 'sb_render_speaker_columns', 10, 2);

// Make view count and last viewed columns sortable
function sb_speaker_sortable_columns($columns) {
    $columns['speaker_views'] = 'speaker_views';
    $columns['last_viewed'] = 'last_viewed';
    return $columns;
}
add_filter('manage_edit-speaker_sortable_columns', 'sb_speaker_sortable_columns');

// Handle sorting by meta values
function sb_speaker_column_sorting($query) {
    if (!is_admin() || !$query->is_main_query()) return;

    $orderby = $query->get('orderby');

    if ($orderby === 'speaker_views') {
        $query->set('meta_key', '_speaker_views');
        $query->set('orderby', 'meta_value_num');
    } elseif ($orderby === 'last_viewed') {
        $query->set('meta_key', '_speaker_last_viewed');
        $query->set('orderby', 'meta_value');
    }
}
add_action('pre_get_posts', 'sb_speaker_column_sorting');

/**
 * Add JavaScript and CSS for inline view count editing
 */
function sb_views_editor_scripts() {
    $screen = get_current_screen();
    if ($screen && $screen->post_type === 'speaker' && $screen->base === 'edit') {
        ?>
        <style>
        .sb-views-editor {
            position: relative;
        }
        .sb-views-display:hover {
            background: #f0f0f1;
            border-radius: 3px;
            padding: 2px 4px;
            margin: -2px -4px;
        }
        .sb-views-editor-form input {
            margin-bottom: 4px;
        }
        </style>
        <script>
        function sbEditViews(postId, element) {
            const container = element.parentNode;
            const display = container.querySelector('.sb-views-display');
            const form = container.querySelector('.sb-views-editor-form');

            display.style.display = 'none';
            form.style.display = 'block';
            form.querySelector('input').focus();
        }

        function sbCancelEditViews(element) {
            const container = element.closest('.sb-views-editor');
            const display = container.querySelector('.sb-views-display');
            const form = container.querySelector('.sb-views-editor-form');

            display.style.display = 'block';
            form.style.display = 'none';
        }

        function sbSaveViews(postId, element) {
            const container = element.closest('.sb-views-editor');
            const input = container.querySelector('input');
            const newViews = parseInt(input.value) || 0;

            // Show loading state
            element.disabled = true;
            element.textContent = 'Saving...';

            // Make AJAX request
            fetch(ajaxurl, {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/x-www-form-urlencoded',
                },
                body: new URLSearchParams({
                    action: 'sb_update_speaker_views',
                    post_id: postId,
                    views: newViews,
                    nonce: '<?php echo wp_create_nonce("sb_update_views"); ?>'
                })
            })
            .then(response => response.json())
            .then(data => {
                if (data.success) {
                    // Update display with new value
                    const display = container.querySelector('.sb-views-display strong');
                    display.textContent = new Intl.NumberFormat().format(newViews);

                    // Update small text
                    const small = container.querySelector('.sb-views-display small');
                    if (newViews > 0 && !small.textContent.includes('views')) {
                        small.textContent = 'views';
                    } else if (newViews === 0) {
                        small.textContent = '';
                    }

                    // Show success feedback
                    const form = container.querySelector('.sb-views-editor-form');
                    form.style.display = 'none';
                    container.querySelector('.sb-views-display').style.display = 'block';

                    // Brief success highlight
                    container.style.background = '#d4edda';
                    setTimeout(() => container.style.background = '', 2000);
                } else {
                    alert('Error saving views: ' + (data.data || 'Unknown error'));
                }
            })
            .catch(error => {
                alert('Error saving views: ' + error.message);
            })
            .finally(() => {
                element.disabled = false;
                element.textContent = 'Save';
            });
        }
        </script>
        <?php
    }
}
add_action('admin_head', 'sb_views_editor_scripts');

/**
 * Add bulk action for setting random view counts (for testing popularity sorting)
 */
function sb_add_bulk_views_action($bulk_actions) {
    $bulk_actions['sb_random_views'] = 'Set Random View Counts (Testing)';
    return $bulk_actions;
}
add_filter('bulk_actions-edit-speaker', 'sb_add_bulk_views_action');

/**
 * Handle bulk action for setting random view counts
 */
function sb_handle_bulk_views_action($redirect_to, $doaction, $post_ids) {
    if ($doaction !== 'sb_random_views') {
        return $redirect_to;
    }

    if (!current_user_can('edit_posts')) {
        return $redirect_to;
    }

    $count = 0;
    foreach ($post_ids as $post_id) {
        $post = get_post($post_id);
        if ($post && $post->post_type === 'speaker') {
            $random_views = rand(1, 100); // Random views between 1 and 100
            update_post_meta($post_id, '_speaker_views', $random_views);
            $count++;
        }
    }

    $redirect_to = add_query_arg('sb_random_views', $count, $redirect_to);
    return $redirect_to;
}
add_filter('handle_bulk_actions-edit-speaker', 'sb_handle_bulk_views_action', 10, 3);

/**
 * Show admin notice after bulk setting random views
 */
function sb_bulk_views_admin_notice() {
    if (!empty($_REQUEST['sb_random_views'])) {
        $count = intval($_REQUEST['sb_random_views']);
        printf(
            '<div class="notice notice-success is-dismissible"><p>Set random view counts for %d speakers. You can now test the "Most Popular" sorting.</p></div>',
            $count
        );
    }
}
add_action('admin_notices', 'sb_bulk_views_admin_notice');

/** END OF FILE: includes/sb-columns.php */
