<?php
/**
 * File: includes/sb-blocks.php
 * Purpose: WordPress Gutenberg blocks for Speakers Bureau
 */

if (!defined('ABSPATH')) exit;

/**
 * Register the Speaker Topics Index block
 */
function sb_register_blocks() {
    // Only register if Gutenberg is available
    if (!function_exists('register_block_type')) {
        return;
    }

    register_block_type('speakers-bureau/topics-index', [
        'render_callback' => 'sb_render_topics_index_block',
        'attributes' => [
            'columns' => [
                'type' => 'number',
                'default' => 4
            ],
            'showCounts' => [
                'type' => 'boolean',
                'default' => true
            ],
            'sortOrder' => [
                'type' => 'string',
                'default' => 'alphabetical'
            ],
            'layout' => [
                'type' => 'string',
                'default' => 'grid'
            ],
            'buttonSize' => [
                'type' => 'string',
                'default' => 'medium'
            ],
            'buttonSpacing' => [
                'type' => 'number',
                'default' => 8
            ],
            'buttonColor' => [
                'type' => 'string',
                'default' => '#0073aa'
            ],
            'buttonHoverColor' => [
                'type' => 'string',
                'default' => '#005177'
            ],
            'buttonTextColor' => [
                'type' => 'string',
                'default' => '#ffffff'
            ],
            'buttonTextHoverColor' => [
                'type' => 'string',
                'default' => '#ffffff'
            ],
            'topicLimit' => [
                'type' => 'number',
                'default' => 0
            ],
            'gridColumnSpacing' => [
                'type' => 'number',
                'default' => 20
            ],
            'gridRowSpacing' => [
                'type' => 'number',
                'default' => 10
            ],
            'gridFontColor' => [
                'type' => 'string',
                'default' => '#0073aa'
            ],
            'gridFontSize' => [
                'type' => 'number',
                'default' => 16
            ],
            'topicMaxLength' => [
                'type' => 'number',
                'default' => 15
            ]
        ]
    ]);
}
add_action('init', 'sb_register_blocks');

/**
 * Get topics data with caching
 */
function sb_get_topics_data_cached($force_refresh = false) {
    $cache_key = 'sb_topics_data_v2';
    $cache_duration = 12 * HOUR_IN_SECONDS; // 12 hours

    // Check if we need to force refresh
    if (!$force_refresh) {
        $cached = get_transient($cache_key);
        if ($cached !== false) {
            return $cached;
        }
    }

    // Build optimized query - get speakers with topics using a single query
    global $wpdb;
    $query = $wpdb->prepare("
        SELECT p.ID, pm.meta_value as topics
        FROM {$wpdb->posts} p
        INNER JOIN {$wpdb->postmeta} pm ON p.ID = pm.post_id
        WHERE p.post_type = %s
        AND p.post_status = %s
        AND pm.meta_key = %s
        AND pm.meta_value != ''
        ORDER BY p.post_date DESC
    ", 'speaker', 'publish', 'topics');

    $results = $wpdb->get_results($query);

    if (empty($results)) {
        // Cache empty result for shorter time
        set_transient($cache_key, [], 1 * HOUR_IN_SECONDS);
        return [];
    }

    // Process topics data
    $topics_data = [];
    foreach ($results as $row) {
        $topics_raw = $row->topics;
        if (empty($topics_raw)) continue;

        // Handle both string and array topics
        if (is_serialized($topics_raw)) {
            $topics_list = maybe_unserialize($topics_raw);
            if (!is_array($topics_list)) {
                $topics_list = [$topics_list];
            }
        } else {
            // Split on common delimiters and clean up
            $topics_list = preg_split('/[,;]\s*/', trim($topics_raw));
        }

        foreach ($topics_list as $topic) {
            $topic = trim($topic);
            if (empty($topic) || strlen($topic) < 2) continue;

            // Clean up topic text
            $topic = ucwords(strtolower($topic));
            $topic = preg_replace('/\s+/', ' ', $topic); // normalize spaces

            if (!isset($topics_data[$topic])) {
                $topics_data[$topic] = 0;
            }
            $topics_data[$topic]++;
        }
    }

    // Cache the processed data
    set_transient($cache_key, $topics_data, $cache_duration);
    return $topics_data;
}

/**
 * Get speakers page URL with caching
 */
function sb_get_speakers_page_url_cached() {
    static $speakers_page_url = null;

    if ($speakers_page_url !== null) {
        return $speakers_page_url;
    }

    $cache_key = 'sb_speakers_page_url';
    $cached_url = get_transient($cache_key);

    if ($cached_url !== false) {
        $speakers_page_url = $cached_url;
        return $speakers_page_url;
    }

    // Method 1: Search for pages containing [speaker_list] shortcode
    global $wpdb;
    $shortcode_page = $wpdb->get_row(
        "SELECT ID FROM {$wpdb->posts}
         WHERE post_type = 'page'
         AND post_status = 'publish'
         AND post_content LIKE '%[speaker_list%'
         LIMIT 1"
    );

    if ($shortcode_page) {
        $speakers_page_url = get_permalink($shortcode_page->ID);
    }

    // Method 2: Check common page slugs
    if (!$speakers_page_url) {
        $common_slugs = ['speakers', 'speaker-directory', 'directory'];
        foreach ($common_slugs as $slug) {
            $page = get_page_by_path($slug);
            if ($page) {
                $speakers_page_url = get_permalink($page->ID);
                break;
            }
        }
    }

    // Method 3: Fallback to post type archive
    if (!$speakers_page_url) {
        $speakers_page_url = get_post_type_archive_link('speaker');
    }

    // Method 4: Final fallback
    if (!$speakers_page_url) {
        $speakers_page_url = home_url('/speakers/');
    }

    // Cache for 24 hours
    set_transient($cache_key, $speakers_page_url, 24 * HOUR_IN_SECONDS);
    return $speakers_page_url;
}

/**
 * Render the Topics Index block
 */
function sb_render_topics_index_block($attributes) {
    $columns = isset($attributes['columns']) ? max(1, min(6, (int)$attributes['columns'])) : 4;
    $show_counts = isset($attributes['showCounts']) ? (bool)$attributes['showCounts'] : true;
    $sort_order = isset($attributes['sortOrder']) ? sanitize_text_field($attributes['sortOrder']) : 'alphabetical';
    $layout = isset($attributes['layout']) ? sanitize_text_field($attributes['layout']) : 'grid';
    $button_size = isset($attributes['buttonSize']) ? sanitize_text_field($attributes['buttonSize']) : 'medium';
    $button_spacing = isset($attributes['buttonSpacing']) ? max(0, (int)$attributes['buttonSpacing']) : 8;
    $button_color = isset($attributes['buttonColor']) ? sanitize_hex_color($attributes['buttonColor']) : '#0073aa';
    $button_hover_color = isset($attributes['buttonHoverColor']) ? sanitize_hex_color($attributes['buttonHoverColor']) : '#005177';
    $button_text_color = isset($attributes['buttonTextColor']) ? sanitize_hex_color($attributes['buttonTextColor']) : '#ffffff';
    $button_text_hover_color = isset($attributes['buttonTextHoverColor']) ? sanitize_hex_color($attributes['buttonTextHoverColor']) : '#ffffff';
    $topic_limit = isset($attributes['topicLimit']) ? max(0, (int)$attributes['topicLimit']) : 0;
    $grid_column_spacing = isset($attributes['gridColumnSpacing']) ? max(0, (int)$attributes['gridColumnSpacing']) : 20;
    $grid_row_spacing = isset($attributes['gridRowSpacing']) ? max(0, (int)$attributes['gridRowSpacing']) : 10;
    $grid_font_color = isset($attributes['gridFontColor']) ? sanitize_hex_color($attributes['gridFontColor']) : '#0073aa';
    $grid_font_size = isset($attributes['gridFontSize']) ? max(8, min(32, (int)$attributes['gridFontSize'])) : 16;
    $topic_max_length = isset($attributes['topicMaxLength']) ? max(5, min(100, (int)$attributes['topicMaxLength'])) : 15;

    // Get cached topics data
    $topics_data = sb_get_topics_data_cached();

    if (empty($topics_data)) {
        return '<p class="sb-no-topics">' . esc_html__('No speaker topics found.', 'speakers-bureau') . '</p>';
    }

    // Sort topics
    if ($sort_order === 'popular') {
        arsort($topics_data); // Sort by count, descending
    } else {
        ksort($topics_data); // Sort alphabetically
    }

    // Apply topic limit if specified
    if ($topic_limit > 0) {
        $topics_data = array_slice($topics_data, 0, $topic_limit, true);
    }

    // Get cached speakers page URL
    $speakers_page_url = sb_get_speakers_page_url_cached();

    // Helper function to truncate topic names
    $truncate_topic = function($topic, $max_length) {
        if (strlen($topic) <= $max_length) {
            return ['display' => $topic, 'title' => ''];
        }
        return [
            'display' => substr($topic, 0, $max_length) . '...',
            'title' => $topic
        ];
    };

    // Generate HTML
    ob_start();

    if ($layout === 'buttons'): ?>
        <div class="sb-topics-index sb-topics-buttons" data-button-size="<?php echo esc_attr($button_size); ?>">
            <div class="sb-topics-button-container">
                <?php foreach ($topics_data as $topic => $count): ?>
                    <?php
                    $search_url = add_query_arg('q', urlencode($topic), $speakers_page_url);
                    $topic_display = $truncate_topic($topic, $topic_max_length);
                    ?>
                    <a href="<?php echo esc_url($search_url); ?>" class="sb-topic-button"<?php echo $topic_display['title'] ? ' title="' . esc_attr($topic_display['title']) . '"' : ''; ?>>
                        <span class="sb-topic-name"><?php echo esc_html($topic_display['display']); ?></span>
                        <?php if ($show_counts): ?>
                            <span class="sb-topic-count"><?php echo intval($count); ?></span>
                        <?php endif; ?>
                    </a>
                <?php endforeach; ?>
            </div>
        </div>
    <?php else: ?>
        <div class="sb-topics-index sb-topics-cols-<?php echo esc_attr($columns); ?>">
            <div class="sb-topics-grid">
                <?php foreach ($topics_data as $topic => $count): ?>
                    <?php
                    $search_url = add_query_arg('q', urlencode($topic), $speakers_page_url);
                    $topic_display = $truncate_topic($topic, $topic_max_length);
                    ?>
                    <div class="sb-topic-item">
                        <a href="<?php echo esc_url($search_url); ?>" class="sb-topic-link"<?php echo $topic_display['title'] ? ' title="' . esc_attr($topic_display['title']) . '"' : ''; ?>>
                            <span class="sb-topic-name"><?php echo esc_html($topic_display['display']); ?></span>
                            <?php if ($show_counts): ?>
                                <span class="sb-topic-count">(<?php echo intval($count); ?>)</span>
                            <?php endif; ?>
                        </a>
                    </div>
                <?php endforeach; ?>
            </div>
        </div>
    <?php endif; ?>

    <style>
    .sb-topics-index {
        margin: 20px 0;
    }

    /* Grid Layout Styles */
    .sb-topics-grid {
        display: grid;
        gap: <?php echo esc_attr($grid_row_spacing); ?>px <?php echo esc_attr($grid_column_spacing); ?>px;
        margin: 20px 0;
    }

    .sb-topics-cols-1 .sb-topics-grid { grid-template-columns: 1fr; }
    .sb-topics-cols-2 .sb-topics-grid { grid-template-columns: repeat(2, 1fr); }
    .sb-topics-cols-3 .sb-topics-grid { grid-template-columns: repeat(3, 1fr); }
    .sb-topics-cols-4 .sb-topics-grid { grid-template-columns: repeat(4, 1fr); }
    .sb-topics-cols-5 .sb-topics-grid { grid-template-columns: repeat(5, 1fr); }
    .sb-topics-cols-6 .sb-topics-grid { grid-template-columns: repeat(6, 1fr); }

    .sb-topic-item {
        line-height: 1.4;
    }

    .sb-topics-index .sb-topic-link {
        color: <?php echo esc_attr($grid_font_color); ?> !important;
        text-decoration: none;
        display: block;
        padding: 4px 0;
        border-bottom: 1px solid transparent;
        transition: all 0.2s ease;
        font-size: <?php echo esc_attr($grid_font_size); ?>px;
        line-height: 1.4;
    }

    .sb-topics-index .sb-topic-link:hover {
        color: <?php echo esc_attr($grid_font_color); ?> !important;
        border-bottom-color: <?php echo esc_attr($grid_font_color); ?>;
        opacity: 0.8;
        text-decoration: none !important;
    }

    .sb-topics-index .sb-topic-link:visited {
        color: <?php echo esc_attr($grid_font_color); ?> !important;
    }

    /* Button Layout Styles */
    .sb-topics-button-container {
        display: flex;
        flex-wrap: wrap;
        gap: <?php echo esc_attr($button_spacing); ?>px;
        margin: 20px 0;
    }

    .sb-topic-button {
        display: inline-flex;
        align-items: center;
        gap: 4px;
        text-decoration: none;
        border: 1px solid <?php echo esc_attr($button_color); ?>;
        background: <?php echo esc_attr($button_color); ?>;
        color: <?php echo esc_attr($button_text_color); ?>;
        border-radius: 20px;
        transition: all 0.2s ease;
        font-weight: 500;
        white-space: nowrap;
    }

    .sb-topic-button:hover {
        background: <?php echo esc_attr($button_hover_color); ?>;
        border-color: <?php echo esc_attr($button_hover_color); ?>;
        color: <?php echo esc_attr($button_text_hover_color); ?>;
        text-decoration: none;
        transform: translateY(-1px);
        box-shadow: 0 2px 4px rgba(0,0,0,0.1);
    }

    .sb-topic-button:focus {
        outline: 2px solid <?php echo esc_attr($button_color); ?>;
        outline-offset: 2px;
    }

    /* Button Sizes */
    .sb-topics-buttons[data-button-size="small"] .sb-topic-button {
        padding: 6px 12px;
        font-size: 0.85em;
        border-radius: 16px;
    }

    .sb-topics-buttons[data-button-size="medium"] .sb-topic-button {
        padding: 8px 16px;
        font-size: 0.9em;
        border-radius: 20px;
    }

    .sb-topics-buttons[data-button-size="large"] .sb-topic-button {
        padding: 12px 20px;
        font-size: 1em;
        border-radius: 24px;
    }

    .sb-topic-name {
        font-weight: inherit;
    }

    .sb-topics-buttons .sb-topic-count {
        background: rgba(255,255,255,0.2);
        border-radius: 10px;
        padding: 2px 6px;
        font-size: 0.8em;
        font-weight: 600;
        color: inherit;
    }

    .sb-topic-link .sb-topic-count {
        color: #666;
        font-size: 0.9em;
        margin-left: 4px;
    }

    .sb-no-topics {
        text-align: center;
        color: #666;
        font-style: italic;
        padding: 20px;
        background: #f8f8f8;
        border: 1px solid #ddd;
        border-radius: 4px;
    }

    /* Responsive adjustments */
    @media (max-width: 768px) {
        .sb-topics-cols-4 .sb-topics-grid,
        .sb-topics-cols-5 .sb-topics-grid,
        .sb-topics-cols-6 .sb-topics-grid {
            grid-template-columns: repeat(2, 1fr);
        }

        .sb-topics-button-container {
            gap: <?php echo max(4, $button_spacing - 2); ?>px;
        }
    }

    @media (max-width: 480px) {
        .sb-topics-index .sb-topics-grid {
            grid-template-columns: 1fr !important;
        }

        .sb-topics-button-container {
            gap: <?php echo max(2, $button_spacing - 4); ?>px;
        }

        .sb-topics-buttons[data-button-size="large"] .sb-topic-button {
            padding: 10px 16px;
            font-size: 0.95em;
        }
    }
    </style>

    <?php
    return ob_get_clean();
}

/**
 * Add block category for our blocks
 */
function sb_block_categories($categories, $post) {
    return array_merge(
        $categories,
        [
            [
                'slug'  => 'speakers-bureau',
                'title' => __('Speakers Bureau', 'speakers-bureau'),
                'icon'  => 'microphone'
            ]
        ]
    );
}
add_filter('block_categories_all', 'sb_block_categories', 10, 2);

/**
 * Clear topics cache when speakers are updated
 */
function sb_clear_topics_cache() {
    delete_transient('sb_topics_data_v2');
}

// Hook into speaker post saves and meta updates
add_action('save_post_speaker', 'sb_clear_topics_cache');
add_action('updated_post_meta', function($meta_id, $object_id, $meta_key) {
    if ($meta_key === 'topics' && get_post_type($object_id) === 'speaker') {
        sb_clear_topics_cache();
    }
}, 10, 3);

// Clear cache when speakers are deleted
add_action('before_delete_post', function($post_id) {
    if (get_post_type($post_id) === 'speaker') {
        sb_clear_topics_cache();
    }
});

// Clear speakers page URL cache when pages are updated
add_action('save_post_page', function() {
    delete_transient('sb_speakers_page_url');
});

/**
 * Admin action to refresh topics cache
 */
function sb_handle_refresh_topics_cache() {
    if (!current_user_can('manage_options')) {
        wp_die(__('Access denied.', 'speakers-bureau'));
    }

    check_admin_referer('sb_refresh_topics_cache');

    // Force refresh the cache
    $topics_data = sb_get_topics_data_cached(true);
    $count = count($topics_data);

    wp_safe_redirect(add_query_arg([
        'page' => 'sb-settings',
        'tab' => 'general',
        'sb_notice' => rawurlencode(sprintf(__('Topics cache refreshed successfully. Found %d unique topics.', 'speakers-bureau'), $count))
    ], admin_url('admin.php')));
    exit;
}
add_action('admin_post_sb_refresh_topics_cache', 'sb_handle_refresh_topics_cache');

/**
 * Enqueue block editor assets
 */
function sb_enqueue_block_editor_assets() {
    wp_enqueue_script(
        'sb-blocks-js',
        SB_PLUGIN_URL . 'assets/sb-blocks.js',
        ['wp-blocks', 'wp-element', 'wp-editor', 'wp-components', 'wp-i18n'],
        SB_PLUGIN_VER
    );
}
add_action('enqueue_block_editor_assets', 'sb_enqueue_block_editor_assets');

/** END OF FILE: includes/sb-blocks.php */