<?php
/**
 * File: includes/sb-auth-bootstrap.php
 * Purpose: Create "speaker" role, keep speakers out of wp-admin, ensure front-end pages,
 *          and provide shortcodes to render the plugin templates.
 * Notes:
 *  - Role "speaker" has minimal caps (front-end editing only; no wp-admin or media browsing).
 *  - Pages auto-created (idempotent): /account-login/, /my-profile/, /register-speaker/.
 *  - Shortcodes:
 *      [sb_login]             -> front-end login w/ forgot password link.
 *      [sb_speaker_edit]      -> renders templates/speaker-edit.php (fragment).
 *      [sb_speaker_register]  -> renders templates/speaker-register.php (fragment).
 *  - Additive only. Does not remove any existing logic.
 */

if (!defined('ABSPATH')) exit;

/** ---------- Utilities ---------- */
if (!function_exists('sb_ensure_page_with_shortcode_24x')) {
function sb_ensure_page_with_shortcode_24x($slug, $title, $shortcode) {
    $existing = get_page_by_path($slug);
    if ($existing && $existing->post_status !== 'trash') {
        $content = (string)$existing->post_content;
        $needs_update = false;

        // Clean up old duplicate shortcodes based on page type
        if ($slug === 'register-speaker') {
            // Remove old speaker_register shortcode if present
            if (strpos($content, '[speaker_register]') !== false) {
                $content = str_replace('[speaker_register]', '', $content);
                $needs_update = true;
            }
        } elseif ($slug === 'my-profile') {
            // Remove old speaker_edit shortcode if present
            if (strpos($content, '[speaker_edit]') !== false) {
                $content = str_replace('[speaker_edit]', '', $content);
                $needs_update = true;
            }
        }

        // If page doesn't contain the new shortcode, add it
        if (strpos($content, $shortcode) === false) {
            $content = rtrim($content . "\n\n" . $shortcode);
            $needs_update = true;
        }

        // Clean up extra whitespace
        $content = preg_replace('/\n{3,}/', "\n\n", trim($content));

        if ($needs_update) {
            wp_update_post([
                'ID'           => $existing->ID,
                'post_content' => $content,
            ]);
        }
        return (int)$existing->ID;
    }
    // Create fresh page
    return (int) wp_insert_post([
        'post_title'   => $title,
        'post_name'    => $slug,
        'post_status'  => 'publish',
        'post_type'    => 'page',
        'post_content' => $shortcode,
    ]);
}
}

/** ---------- Role + Pages Bootstrap ---------- */
if (!function_exists('sb_auth_bootstrap_init_24x')) {
function sb_auth_bootstrap_init_24x() {
    // Create/normalize SPEAKER role with minimal caps
    $role = get_role('speaker');
    if (!$role) {
        add_role('speaker', 'Speaker', [
            'read'         => true,
            // Keep uploads front-end-only; we use wp_handle_upload in templates.
            'upload_files' => false,
        ]);
    } else {
        $role->add_cap('read');
        $role->remove_cap('edit_posts');
        $role->remove_cap('edit_pages');
        $role->remove_cap('list_users');
        $role->remove_cap('manage_options');
        $role->remove_cap('upload_files');
    }

    // Ensure key pages exist (idempotent) and have the right shortcodes
    sb_ensure_page_with_shortcode_24x('account-login',     'Account Login',     '[sb_login]');
    sb_ensure_page_with_shortcode_24x('my-profile',        'My Profile',        '[sb_speaker_edit]');
    sb_ensure_page_with_shortcode_24x('register-speaker',  'Register Speaker',  '[sb_speaker_register]');

    // Prevent search queries on my-profile page to avoid WordPress core warnings
    add_action('pre_get_posts', 'sb_disable_search_on_my_profile');
    add_action('parse_request', 'sb_clean_my_profile_search_params', 1);
}
add_action('init', 'sb_auth_bootstrap_init_24x', 9); // early so pages exist for rendering

/**
 * Prevent WordPress search processing on my-profile page
 */
function sb_clean_my_profile_search_params($wp) {
    // Check if request is for my-profile page
    if (isset($wp->request) && (
        $wp->request === 'my-profile' ||
        strpos($wp->request, 'my-profile/') === 0 ||
        (isset($wp->query_vars['pagename']) && $wp->query_vars['pagename'] === 'my-profile')
    )) {
        // Remove search parameter from WordPress query vars to prevent core search processing
        // This prevents the deprecated str_starts_with() warning in block templates
        if (isset($wp->query_vars['s'])) {
            unset($wp->query_vars['s']);
        }
    }

    // Also prevent the warning on speaker single pages
    if (isset($wp->query_vars['post_type']) && $wp->query_vars['post_type'] === 'speaker') {
        if (isset($wp->query_vars['s']) && empty($wp->query_vars['s'])) {
            unset($wp->query_vars['s']);
        }
    }
}

/**
 * Disable search functionality on my-profile page to prevent WordPress core warnings
 */
function sb_disable_search_on_my_profile($query) {
    if (!is_admin() && $query->is_main_query()) {
        // Check if we're on the my-profile page
        $queried_object = get_queried_object();
        $is_my_profile = is_page('my-profile') ||
                        ($queried_object &&
                         isset($queried_object->post_name) &&
                         $queried_object->post_name === 'my-profile');

        if ($is_my_profile) {
            // Disable WordPress search processing to prevent deprecated warnings
            // but keep $_GET['s'] for admin speaker search functionality
            $query->set('s', '');
            $query->is_search = false;
        }
    }
}
}

/** ---------- Role helpers ---------- */
if (!function_exists('sb_current_user_is_speaker')) {
function sb_current_user_is_speaker() {
    if (!is_user_logged_in()) return false;
    $u = wp_get_current_user();
    return in_array('speaker', (array) $u->roles, true);
}
}

/** ---------- Keep speakers out of wp-admin (but NEVER admins) ---------- */
if (!function_exists('sb_restrict_admin_for_speakers')) {
function sb_restrict_admin_for_speakers() {
    if (!is_user_logged_in()) return;

    // If user can manage options, do not restrict (fixes admins who also have 'speaker' role)
    if (current_user_can('manage_options')) return;

    // Only restrict users who (also) have the 'speaker' role
    if (!sb_current_user_is_speaker()) return;

    // Allow ajax/cron
    if (defined('DOING_AJAX') && DOING_AJAX) return;
    if (defined('DOING_CRON') && DOING_CRON) return;

    // If they try wp-admin, redirect to a friendly front-end page
    if (is_admin()) {
        $target = home_url('/');
        if ($p = get_page_by_path('my-profile')) {
            $target = get_permalink($p);
        } elseif ($l = get_page_by_path('account-login')) {
            $target = get_permalink($l);
        }
        wp_safe_redirect($target);
        exit;
    }
}
add_action('admin_init', 'sb_restrict_admin_for_speakers');
}

/** ---------- Hide admin bar for speakers on front-end (but NEVER admins) ---------- */
add_filter('show_admin_bar', function($show) {
    // If user is admin, leave as-is
    if (current_user_can('manage_options')) return $show;
    // Otherwise hide for speakers
    return sb_current_user_is_speaker() ? false : $show;
});

/** ---------- Shortcodes ---------- */

// Login form (front-end)
if (!function_exists('sb_render_login_form_24x')) {
function sb_render_login_form_24x() {
    $action = isset($_GET['action']) ? sanitize_text_field($_GET['action']) : 'login';
    $notice = '';
    $error = '';

    // Handle password reset form submission
    if ($action === 'resetpass' && $_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['user_login'])) {
        $user_login = sanitize_text_field($_POST['user_login']);
        $user_data = false;

        if (strpos($user_login, '@')) {
            $user_data = get_user_by('email', trim($user_login));
        } else {
            $user_data = get_user_by('login', trim($user_login));
        }

        if (!$user_data) {
            $error = __('Invalid username or email address.', 'speakers-bureau');
        } else {
            $user_login = $user_data->user_login;
            $user_email = $user_data->user_email;
            $key = get_password_reset_key($user_data);

            if (is_wp_error($key)) {
                $error = $key->get_error_message();
            } else {
                $message = __('Someone has requested a password reset for the following account:') . "\r\n\r\n";
                $message .= network_home_url('/') . "\r\n\r\n";
                $message .= sprintf(__('Username: %s'), $user_login) . "\r\n\r\n";
                $message .= __('If this was a mistake, just ignore this email and nothing will happen.') . "\r\n\r\n";
                $message .= __('To reset your password, visit the following address:') . "\r\n\r\n";
                $message .= network_site_url("wp-login.php?action=rp&key=$key&login=" . rawurlencode($user_login), 'login') . "\r\n";

                $title = sprintf(__('[%s] Password Reset'), wp_specialchars_decode(get_option('blogname'), ENT_QUOTES));

                if (wp_mail($user_email, wp_specialchars_decode($title), $message)) {
                    $notice = __('Check your email for the confirmation link, then visit the link to reset your password.', 'speakers-bureau');
                } else {
                    $error = __('The email could not be sent. Possible reason: your host may have disabled the mail() function.', 'speakers-bureau');
                }
            }
        }
    }

    ob_start();
    ?>
    <div style="max-width: 400px; margin: 40px auto; padding: 30px; background: #fff; border-radius: 8px; box-shadow: 0 2px 10px rgba(0,0,0,0.1);">

        <?php if (is_user_logged_in()): ?>
            <?php $u = wp_get_current_user(); ?>
            <div style="text-align: center;">
                <div style="font-size: 48px; color: #00a32a; margin-bottom: 20px;">👋</div>
                <h2 style="color: #0073aa; margin-bottom: 10px;"><?php esc_html_e('Welcome back!', 'speakers-bureau'); ?></h2>
                <p style="margin-bottom: 20px; color: #666;">
                    <?php printf(__('You are logged in as %s', 'speakers-bureau'), '<strong>' . esc_html($u->display_name ?: $u->user_login) . '</strong>'); ?>
                </p>

                <div style="margin: 25px 0;">
                    <?php if ($p = get_page_by_path('my-profile')): ?>
                        <a href="<?php echo esc_url(get_permalink($p)); ?>" class="button button-primary" style="display: inline-block; padding: 12px 24px; text-decoration: none; margin-bottom: 10px;">
                            <?php esc_html_e('Go to My Profile', 'speakers-bureau'); ?>
                        </a><br>
                    <?php endif; ?>

                    <a href="<?php echo esc_url(wp_logout_url(home_url('/'))); ?>" class="button button-secondary" style="display: inline-block; padding: 8px 16px; text-decoration: none;">
                        <?php esc_html_e('Log Out', 'speakers-bureau'); ?>
                    </a>
                </div>
            </div>

        <?php elseif ($action === 'resetpass'): ?>
            <div style="text-align: center; margin-bottom: 30px;">
                <h2 style="color: #0073aa; margin-bottom: 10px;"><?php esc_html_e('Reset Password', 'speakers-bureau'); ?></h2>
                <p style="color: #666;"><?php esc_html_e('Enter your username or email address below.', 'speakers-bureau'); ?></p>
            </div>

            <?php if ($notice): ?>
                <div style="padding: 12px; background: #d4edda; border: 1px solid #c3e6cb; border-radius: 4px; margin-bottom: 20px; color: #155724;">
                    <?php echo esc_html($notice); ?>
                </div>
            <?php endif; ?>

            <?php if ($error): ?>
                <div style="padding: 12px; background: #f8d7da; border: 1px solid #f5c6cb; border-radius: 4px; margin-bottom: 20px; color: #721c24;">
                    <?php echo esc_html($error); ?>
                </div>
            <?php endif; ?>

            <form method="post" action="">
                <div style="margin-bottom: 20px;">
                    <label for="user_login" style="display: block; margin-bottom: 8px; font-weight: 600;">
                        <?php esc_html_e('Username or Email', 'speakers-bureau'); ?>
                    </label>
                    <input type="text" name="user_login" id="user_login"
                           style="width: 100%; padding: 12px; border: 1px solid #ddd; border-radius: 4px; font-size: 16px;"
                           placeholder="<?php esc_attr_e('Enter username or email', 'speakers-bureau'); ?>"
                           required>
                </div>

                <div style="text-align: center; margin-bottom: 20px;">
                    <button type="submit" class="button button-primary" style="width: 100%; padding: 12px; font-size: 16px;">
                        <?php esc_html_e('Send Reset Link', 'speakers-bureau'); ?>
                    </button>
                </div>
            </form>

            <div style="text-align: center;">
                <a href="?" style="color: #0073aa; text-decoration: none;">
                    <?php esc_html_e('← Back to Login', 'speakers-bureau'); ?>
                </a>
            </div>

        <?php else: ?>
            <div style="text-align: center; margin-bottom: 30px;">
                <h2 style="color: #0073aa; margin-bottom: 10px;"><?php esc_html_e('Speaker Login', 'speakers-bureau'); ?></h2>
                <p style="color: #666;"><?php esc_html_e('Access your speaker profile and settings.', 'speakers-bureau'); ?></p>
            </div>

            <?php
            // Custom styled login form
            $redirect_to = isset($_GET['redirect_to']) ? esc_url_raw($_GET['redirect_to']) : home_url('/my-profile/');
            $args = [
                'echo' => true,
                'redirect' => $redirect_to,
                'form_id' => 'sb_loginform',
                'label_username' => __('Username or Email', 'speakers-bureau'),
                'label_password' => __('Password', 'speakers-bureau'),
                'label_remember' => __('Remember Me', 'speakers-bureau'),
                'label_log_in' => __('Log In', 'speakers-bureau'),
                'id_username' => 'user_login',
                'id_password' => 'user_pass',
                'id_remember' => 'rememberme',
                'id_submit' => 'wp-submit',
                'remember' => true,
                'value_username' => '',
                'value_remember' => false
            ];
            wp_login_form($args);
            ?>

            <style>
            #sb_loginform {
                text-align: left;
            }
            #sb_loginform p {
                margin-bottom: 15px;
            }
            #sb_loginform label {
                display: block;
                margin-bottom: 6px;
                font-weight: 600;
                color: #333;
            }
            #sb_loginform input[type="text"],
            #sb_loginform input[type="password"] {
                width: 100%;
                padding: 12px;
                border: 1px solid #ddd;
                border-radius: 4px;
                font-size: 16px;
                box-sizing: border-box;
            }
            #sb_loginform input[type="checkbox"] {
                margin-right: 8px;
            }
            #sb_loginform input[type="submit"] {
                width: 100%;
                padding: 12px;
                background: #0073aa;
                color: white;
                border: none;
                border-radius: 4px;
                font-size: 16px;
                cursor: pointer;
                margin-top: 10px;
            }
            #sb_loginform input[type="submit"]:hover {
                background: #005a87;
            }
            .sb-login-links {
                text-align: center;
                margin-top: 20px;
                padding-top: 20px;
                border-top: 1px solid #eee;
            }
            .sb-login-links a {
                color: #0073aa;
                text-decoration: none;
                margin: 0 10px;
            }
            .sb-login-links a:hover {
                text-decoration: underline;
            }
            </style>

            <div class="sb-login-links">
                <a href="?action=resetpass"><?php esc_html_e('Forgot or Reset Password', 'speakers-bureau'); ?></a>
                <?php if ($register_page = get_page_by_path('register-speaker')): ?>
                    |
                    <a href="<?php echo esc_url(get_permalink($register_page)); ?>"><?php esc_html_e('Register as Speaker', 'speakers-bureau'); ?></a>
                <?php endif; ?>
            </div>

        <?php endif; ?>
    </div>
    <?php
    return ob_get_clean();
}
add_shortcode('sb_login', 'sb_render_login_form_24x');
}

// Render My Profile (speaker-edit template fragment)
if (!function_exists('sb_shortcode_speaker_edit_24x')) {
function sb_shortcode_speaker_edit_24x() {
    // Check if user is logged in
    if (!is_user_logged_in()) {
        // Get current page URL for redirect after login
        $current_url = (is_ssl() ? 'https://' : 'http://') . $_SERVER['HTTP_HOST'] . $_SERVER['REQUEST_URI'];

        // Get login page
        $login_page = get_page_by_path('account-login');
        if ($login_page) {
            $login_url = get_permalink($login_page);
            // Add redirect parameter
            $redirect_url = add_query_arg('redirect_to', urlencode($current_url), $login_url);
        } else {
            // Fallback to wp-login with redirect
            $redirect_url = wp_login_url($current_url);
        }

        // Redirect to login page
        if (!wp_doing_ajax() && !defined('DOING_CRON')) {
            wp_safe_redirect($redirect_url);
            exit;
        }

        // If we can't redirect (ajax/cron), show login message
        return '<div style="text-align: center; padding: 40px; background: #f8f9fa; border-radius: 6px; border-left: 4px solid #0073aa;">
            <h3 style="color: #0073aa; margin-bottom: 15px;">' . esc_html__('Login Required', 'speakers-bureau') . '</h3>
            <p style="margin-bottom: 20px;">' . esc_html__('You need to be logged in to access your profile.', 'speakers-bureau') . '</p>
            <a href="' . esc_url($redirect_url) . '" class="button button-primary">' . esc_html__('Go to Login', 'speakers-bureau') . '</a>
        </div>';
    }

    // Enqueue required assets
    if (!did_action('wp_enqueue_media')) wp_enqueue_media();
    wp_enqueue_style('sb-frontend');
    wp_enqueue_script('sb-frontend-js');

    $tpl = dirname(__DIR__) . '/templates/speaker-edit.php';
    if (!file_exists($tpl)) {
        return '<div class="notice notice-error"><p>'.esc_html__('Speaker edit template missing.', 'speakers-bureau').'</p></div>';
    }
    ob_start();
    include $tpl; // template echoes markup; its access control is inside
    return ob_get_clean();
}
add_shortcode('sb_speaker_edit', 'sb_shortcode_speaker_edit_24x');
// Backward compatibility for old shortcode name
add_shortcode('speaker_edit', 'sb_shortcode_speaker_edit_24x');
}

// Render Register Speaker (speaker-register template fragment)
if (!function_exists('sb_shortcode_speaker_register_24x')) {
function sb_shortcode_speaker_register_24x() {
    // Enqueue required assets
    if (!did_action('wp_enqueue_media')) wp_enqueue_media();
    wp_enqueue_style('sb-frontend');
    wp_enqueue_script('sb-frontend-js');

    $tpl = dirname(__DIR__) . '/templates/speaker-register.php';
    if (!file_exists($tpl)) {
        return '<div class="notice notice-error"><p>'.esc_html__('Speaker register template missing.', 'speakers-bureau').'</p></div>';
    }
    ob_start();
    include $tpl; // template echoes markup; handles publish/draft + emails
    return ob_get_clean();
}
add_shortcode('sb_speaker_register', 'sb_shortcode_speaker_register_24x');
// Backward compatibility for old shortcode name
add_shortcode('speaker_register', 'sb_shortcode_speaker_register_24x');
}

/** END OF FILE: includes/sb-auth-bootstrap.php */
