/**
 * File: assets/sb-map.js
 * Purpose: Initialize and render speaker location maps using LeafletJS
 */

(function($) {
    'use strict';

    /**
     * Initialize all speaker maps on the page
     */
    function initializeSpeakerMaps() {
        $('.sb-speaker-map').each(function() {
            var mapContainer = $(this);
            var mapId = mapContainer.attr('id');
            var configData = mapContainer.data('map-config');

            if (!configData || !configData.markers || configData.markers.length === 0) {
                console.warn('No map configuration or markers found for map:', mapId);
                return;
            }

            // Initialize the map
            var map = L.map(mapId).setView(
                [configData.center.lat, configData.center.lng],
                configData.zoom
            );

            // Add OpenStreetMap tiles
            L.tileLayer('https://{s}.tile.openstreetmap.org/{z}/{x}/{y}.png', {
                attribution: '&copy; <a href="https://www.openstreetmap.org/copyright">OpenStreetMap</a> contributors',
                maxZoom: 19,
            }).addTo(map);

            // Create marker cluster group if there are many markers
            var markers = [];

            // Add markers for each speaker
            configData.markers.forEach(function(markerData) {
                var marker = L.marker([markerData.lat, markerData.lng], {
                    // Custom icon (optional - using default for now)
                }).addTo(map);

                // Add popup with speaker information
                if (markerData.popup) {
                    marker.bindPopup(markerData.popup, {
                        maxWidth: 380,
                        minWidth: 280,
                        className: 'sb-map-leaflet-popup'
                    });
                }

                markers.push(marker);
            });

            // If there are multiple markers, fit the map to show all of them
            if (markers.length > 1) {
                var group = new L.featureGroup(markers);
                map.fitBounds(group.getBounds().pad(0.1));
            }

            // Store map instance for potential later access
            mapContainer.data('map-instance', map);
        });
    }

    /**
     * Initialize maps when DOM is ready
     */
    $(document).ready(function() {
        // Wait a bit to ensure Leaflet is fully loaded
        setTimeout(initializeSpeakerMaps, 100);
    });

})(jQuery);
