/**
 * File: assets/sb-blocks.js
 * Purpose: Gutenberg block definitions for Speakers Bureau
 */

(function() {
    const { registerBlockType } = wp.blocks;
    const { InspectorControls } = wp.blockEditor || wp.editor;
    const { PanelBody, RangeControl, ToggleControl, SelectControl } = wp.components;
    const { __ } = wp.i18n;
    const { Fragment, createElement } = wp.element;

    registerBlockType('speakers-bureau/topics-index', {
        title: __('Speaker Topics Index', 'speakers-bureau'),
        description: __('Display an index of all speaker topics in a searchable grid layout.', 'speakers-bureau'),
        icon: 'list-view',
        category: 'speakers-bureau',
        keywords: [
            __('speakers', 'speakers-bureau'),
            __('topics', 'speakers-bureau'),
            __('index', 'speakers-bureau')
        ],
        attributes: {
            columns: {
                type: 'number',
                default: 4
            },
            showCounts: {
                type: 'boolean',
                default: true
            },
            sortOrder: {
                type: 'string',
                default: 'alphabetical'
            },
            layout: {
                type: 'string',
                default: 'grid'
            },
            buttonSize: {
                type: 'string',
                default: 'medium'
            },
            buttonSpacing: {
                type: 'number',
                default: 8
            },
            buttonColor: {
                type: 'string',
                default: '#0073aa'
            },
            buttonHoverColor: {
                type: 'string',
                default: '#005177'
            },
            buttonTextColor: {
                type: 'string',
                default: '#ffffff'
            },
            buttonTextHoverColor: {
                type: 'string',
                default: '#ffffff'
            },
            topicLimit: {
                type: 'number',
                default: 0
            },
            gridColumnSpacing: {
                type: 'number',
                default: 20
            },
            gridRowSpacing: {
                type: 'number',
                default: 10
            },
            gridFontColor: {
                type: 'string',
                default: '#0073aa'
            },
            gridFontSize: {
                type: 'number',
                default: 16
            },
            topicMaxLength: {
                type: 'number',
                default: 15
            }
        },
        edit: function(props) {
            const { attributes, setAttributes } = props;
            const { columns, showCounts, sortOrder, layout, buttonSize, buttonSpacing, buttonColor, buttonHoverColor, buttonTextColor, buttonTextHoverColor, topicLimit, gridColumnSpacing, gridRowSpacing, gridFontColor, gridFontSize, topicMaxLength } = attributes;

            return createElement(Fragment, {},
                // Inspector Controls (Sidebar)
                createElement(InspectorControls, {},
                    createElement(PanelBody, {
                        title: __('Layout Settings', 'speakers-bureau'),
                        initialOpen: true
                    },
                        createElement(SelectControl, {
                            label: __('Layout Style', 'speakers-bureau'),
                            value: layout,
                            options: [
                                { label: __('Grid Layout', 'speakers-bureau'), value: 'grid' },
                                { label: __('Button Layout', 'speakers-bureau'), value: 'buttons' }
                            ],
                            onChange: function(value) {
                                setAttributes({ layout: value });
                            }
                        }),
                        layout === 'grid' && createElement(RangeControl, {
                            label: __('Number of Columns', 'speakers-bureau'),
                            value: columns,
                            onChange: function(value) {
                                setAttributes({ columns: value });
                            },
                            min: 1,
                            max: 6
                        }),
                        createElement(ToggleControl, {
                            label: __('Show Speaker Counts', 'speakers-bureau'),
                            checked: showCounts,
                            onChange: function(value) {
                                setAttributes({ showCounts: value });
                            }
                        }),
                        createElement(SelectControl, {
                            label: __('Sort Order', 'speakers-bureau'),
                            value: sortOrder,
                            options: [
                                { label: __('Alphabetical', 'speakers-bureau'), value: 'alphabetical' },
                                { label: __('By Popularity', 'speakers-bureau'), value: 'popular' }
                            ],
                            onChange: function(value) {
                                setAttributes({ sortOrder: value });
                            }
                        }),
                        createElement(RangeControl, {
                            label: __('Topic Limit (0 = No Limit)', 'speakers-bureau'),
                            value: topicLimit,
                            onChange: function(value) {
                                setAttributes({ topicLimit: value });
                            },
                            min: 0,
                            max: 100
                        }),
                        createElement(RangeControl, {
                            label: __('Topic Max Length (chars)', 'speakers-bureau'),
                            value: topicMaxLength,
                            onChange: function(value) {
                                setAttributes({ topicMaxLength: value });
                            },
                            min: 5,
                            max: 100,
                            help: __('Topics longer than this will be truncated with "..." and show full text on hover', 'speakers-bureau')
                        })
                    ),
                    layout === 'grid' && createElement(PanelBody, {
                        title: __('Grid Styling', 'speakers-bureau'),
                        initialOpen: false
                    },
                        createElement(RangeControl, {
                            label: __('Column Spacing (px)', 'speakers-bureau'),
                            value: gridColumnSpacing,
                            onChange: function(value) {
                                setAttributes({ gridColumnSpacing: value });
                            },
                            min: 0,
                            max: 50
                        }),
                        createElement(RangeControl, {
                            label: __('Row Spacing (px)', 'speakers-bureau'),
                            value: gridRowSpacing,
                            onChange: function(value) {
                                setAttributes({ gridRowSpacing: value });
                            },
                            min: 0,
                            max: 50
                        }),
                        createElement(RangeControl, {
                            label: __('Font Size (px)', 'speakers-bureau'),
                            value: gridFontSize,
                            onChange: function(value) {
                                setAttributes({ gridFontSize: value });
                            },
                            min: 8,
                            max: 32
                        }),
                        createElement('div', { style: { marginBottom: '16px' } },
                            createElement('label', { style: { fontSize: '11px', fontWeight: '500', textTransform: 'uppercase', marginBottom: '8px', display: 'block' } },
                                __('Font Color', 'speakers-bureau')
                            ),
                            createElement('input', {
                                type: 'color',
                                value: gridFontColor,
                                onChange: function(e) {
                                    setAttributes({ gridFontColor: e.target.value });
                                },
                                style: { width: '100%', height: '32px', border: 'none', borderRadius: '4px' }
                            })
                        )
                    ),
                    layout === 'buttons' && createElement(PanelBody, {
                        title: __('Button Styling', 'speakers-bureau'),
                        initialOpen: true
                    },
                        createElement(SelectControl, {
                            label: __('Button Size', 'speakers-bureau'),
                            value: buttonSize,
                            options: [
                                { label: __('Small', 'speakers-bureau'), value: 'small' },
                                { label: __('Medium', 'speakers-bureau'), value: 'medium' },
                                { label: __('Large', 'speakers-bureau'), value: 'large' }
                            ],
                            onChange: function(value) {
                                setAttributes({ buttonSize: value });
                            }
                        }),
                        createElement(RangeControl, {
                            label: __('Button Spacing (px)', 'speakers-bureau'),
                            value: buttonSpacing,
                            onChange: function(value) {
                                setAttributes({ buttonSpacing: value });
                            },
                            min: 0,
                            max: 20
                        }),
                        createElement('div', { style: { marginBottom: '16px' } },
                            createElement('label', { style: { fontSize: '11px', fontWeight: '500', textTransform: 'uppercase', marginBottom: '8px', display: 'block' } },
                                __('Button Color', 'speakers-bureau')
                            ),
                            createElement('input', {
                                type: 'color',
                                value: buttonColor,
                                onChange: function(e) {
                                    setAttributes({ buttonColor: e.target.value });
                                },
                                style: { width: '100%', height: '32px', border: 'none', borderRadius: '4px' }
                            })
                        ),
                        createElement('div', { style: { marginBottom: '16px' } },
                            createElement('label', { style: { fontSize: '11px', fontWeight: '500', textTransform: 'uppercase', marginBottom: '8px', display: 'block' } },
                                __('Button Hover Color', 'speakers-bureau')
                            ),
                            createElement('input', {
                                type: 'color',
                                value: buttonHoverColor,
                                onChange: function(e) {
                                    setAttributes({ buttonHoverColor: e.target.value });
                                },
                                style: { width: '100%', height: '32px', border: 'none', borderRadius: '4px' }
                            })
                        ),
                        createElement('div', { style: { marginBottom: '16px' } },
                            createElement('label', { style: { fontSize: '11px', fontWeight: '500', textTransform: 'uppercase', marginBottom: '8px', display: 'block' } },
                                __('Button Text Color', 'speakers-bureau')
                            ),
                            createElement('input', {
                                type: 'color',
                                value: buttonTextColor,
                                onChange: function(e) {
                                    setAttributes({ buttonTextColor: e.target.value });
                                },
                                style: { width: '100%', height: '32px', border: 'none', borderRadius: '4px' }
                            })
                        ),
                        createElement('div', { style: { marginBottom: '16px' } },
                            createElement('label', { style: { fontSize: '11px', fontWeight: '500', textTransform: 'uppercase', marginBottom: '8px', display: 'block' } },
                                __('Button Text Hover Color', 'speakers-bureau')
                            ),
                            createElement('input', {
                                type: 'color',
                                value: buttonTextHoverColor,
                                onChange: function(e) {
                                    setAttributes({ buttonTextHoverColor: e.target.value });
                                },
                                style: { width: '100%', height: '32px', border: 'none', borderRadius: '4px' }
                            })
                        )
                    )
                ),

                // Block Preview
                createElement('div', {
                    className: 'sb-topics-index-preview',
                    style: {
                        padding: '20px',
                        border: '2px dashed #ccc',
                        borderRadius: '4px',
                        textAlign: layout === 'buttons' ? 'left' : 'center',
                        backgroundColor: '#f8f9fa'
                    }
                },
                    createElement('div', {
                        style: { fontSize: '18px', marginBottom: '10px', textAlign: 'center' }
                    }, layout === 'buttons' ? '🔘' : '📋'),
                    createElement('h4', {
                        style: { margin: '10px 0', color: '#333', textAlign: 'center' }
                    }, __('Speaker Topics Index', 'speakers-bureau')),
                    createElement('p', {
                        style: { margin: '5px 0', color: '#666', fontSize: '14px', textAlign: 'center' }
                    },
                        __('Layout: ', 'speakers-bureau') + (layout === 'buttons' ? __('Buttons', 'speakers-bureau') : __('Grid', 'speakers-bureau')) + ' | ' +
                        (layout === 'grid' ? (__('Columns: ', 'speakers-bureau') + columns + ' | ') : (__('Size: ', 'speakers-bureau') + buttonSize + ' | ')) +
                        __('Show Counts: ', 'speakers-bureau') + (showCounts ? __('Yes', 'speakers-bureau') : __('No', 'speakers-bureau'))
                    ),

                    // Preview content based on layout
                    layout === 'buttons' ?
                        createElement('div', {
                            style: {
                                display: 'flex',
                                flexWrap: 'wrap',
                                gap: buttonSpacing + 'px',
                                marginTop: '15px',
                                justifyContent: 'center'
                            }
                        },
                            ['Leadership', 'Technology', 'Healthcare', 'Education', 'Business'].map(function(topic, index) {
                                const buttonSizes = {
                                    small: { padding: '4px 8px', fontSize: '11px', borderRadius: '12px' },
                                    medium: { padding: '6px 12px', fontSize: '12px', borderRadius: '16px' },
                                    large: { padding: '8px 16px', fontSize: '13px', borderRadius: '20px' }
                                };
                                return createElement('div', {
                                    key: index,
                                    style: Object.assign({}, buttonSizes[buttonSize], {
                                        backgroundColor: buttonColor,
                                        color: buttonTextColor,
                                        border: '1px solid ' + buttonColor,
                                        borderRadius: buttonSizes[buttonSize].borderRadius,
                                        display: 'inline-flex',
                                        alignItems: 'center',
                                        gap: '4px',
                                        cursor: 'pointer'
                                    })
                                },
                                    createElement('span', {}, topic),
                                    showCounts && createElement('span', {
                                        style: {
                                            background: 'rgba(255,255,255,0.2)',
                                            borderRadius: '8px',
                                            padding: '1px 4px',
                                            fontSize: '10px'
                                        }
                                    }, '3')
                                );
                            })
                        ) :
                        createElement('div', {
                            style: {
                                display: 'grid',
                                gridTemplateColumns: 'repeat(' + Math.min(columns, 3) + ', 1fr)',
                                gap: '10px',
                                marginTop: '15px',
                                fontSize: '12px'
                            }
                        },
                            ['Leadership', 'Technology', 'Healthcare', 'Education', 'Business', 'Marketing'].slice(0, Math.min(columns * 2, 6)).map(function(topic, index) {
                                return createElement('div', {
                                    key: index,
                                    style: {
                                        padding: '4px 8px',
                                        backgroundColor: '#e3f2fd',
                                        border: '1px solid #90caf9',
                                        borderRadius: '3px',
                                        color: '#1976d2'
                                    }
                                }, topic + (showCounts ? ' (3)' : ''));
                            })
                        ),

                    createElement('p', {
                        style: { marginTop: '15px', fontSize: '12px', color: '#888', textAlign: 'center' }
                    }, __('This is a preview. The actual topics will be generated from your speaker database.', 'speakers-bureau'))
                )
            );
        },
        save: function() {
            // Server-side rendering, so we return null
            return null;
        }
    });
})();

/** END OF FILE: assets/sb-blocks.js */