/**
 * File: assets/sb-admin.js
 * Purpose: Admin UI for Speakers Bureau Form Builder
 *
 * - Media uploader for image fields
 * - Add/Delete rows
 * - Up/Down arrows for reordering (no jQuery UI dependency)
 * - Support for Select fields using Placeholder/Options textarea -> JSON options
 */

jQuery(document).ready(function($){

  /**
   * Media uploader for image fields
   */
  $(document).on('click', '.sb-upload-btn', function(e){
    e.preventDefault();
    var button = $(this);
    var customUploader = wp.media({
      title: 'Select Image',
      button: { text: 'Use this image' },
      multiple: false
    }).on('select', function(){
      var attachment = customUploader.state().get('selection').first().toJSON();
      button.prev('input').val(attachment.url);
    }).open();
  });

  /**
   * Utility: reindex field names after reorder/delete
   */
  function reindexRows() {
    $('#sb-form-builder tbody tr').each(function(i){
      var $tr = $(this);
      $tr.find('input, select, textarea').each(function(){
        var name = $(this).attr('name');
        if (name) {
          var newName = name.replace(/sb_form_fields\[\d+\]/, 'sb_form_fields['+i+']');
          $(this).attr('name', newName);
        }
      });
    });
  }

  /**
   * Utility: parse options textarea into JSON array
   */
  function parseOptions(raw) {
    if (!raw) return [];
    var parts = raw.split(/\r?\n|,/).map(function(s){ return s.trim(); }).filter(Boolean);
    var seen = {};
    var out = [];
    parts.forEach(function(p){
      if (!seen[p]) { seen[p] = true; out.push(p); }
    });
    return out;
  }

  /**
   * Sync textarea -> hidden JSON for Select fields
   */
  function syncOptions($tr) {
    var raw = $tr.find('.sb-placeholder-options').val() || '';
    var opts = parseOptions(raw);
    $tr.find('.sb-options-json').val(JSON.stringify(opts));
  }

  /**
   * Add Field button
   */
  $('#sb-add-field').on('click', function(e){
    e.preventDefault();
    var index = $('#sb-form-builder tbody tr').length;

    var row = `
      <tr>
        <td style="text-align:center; white-space:nowrap;">
          <button type="button" class="button-link sb-move-up" title="Move Up">⬆️</button><br>
          <button type="button" class="button-link sb-move-down" title="Move Down">⬇️</button>
        </td>
        <td><input type="text" name="sb_form_fields[${index}][label]" value="" /></td>
        <td><input type="text" name="sb_form_fields[${index}][key]" value="" /></td>
        <td>
          <select name="sb_form_fields[${index}][type]" class="sb-type">
            <option value="text">Text</option>
            <option value="email">Email</option>
            <option value="url">URL</option>
            <option value="tel">Phone</option>
            <option value="textarea">Textarea</option>
            <option value="image">Image</option>
            <option value="section_header">Section Header</option>
            <option value="checkbox">Checkbox</option>
            <option value="number">Number</option>
            <option value="select">Select</option>
          </select>
        </td>
        <td>
          <textarea name="sb_form_fields[${index}][placeholder]" class="sb-placeholder-options" rows="2"></textarea>
          <input type="hidden" name="sb_form_fields[${index}][options]" class="sb-options-json" value="[]" />
          <p class="description sb-placeholder-help">Placeholder text (for Select: enter options, comma or newline separated)</p>
        </td>
        <td>
          <label><input type="checkbox" name="sb_form_fields[${index}][enabled]" value="1"> Enabled</label><br>
          <label><input type="checkbox" name="sb_form_fields[${index}][card]" value="1"> Show on Directory Card</label><br>
          <label><input type="checkbox" name="sb_form_fields[${index}][show_section]" value="1"> Show Header on Profile Page</label><br>
          <label><input type="checkbox" name="sb_form_fields[${index}][featured]" value="1"> Use as Featured Image</label>
        </td>
        <td style="text-align:center;">
          <button type="button" class="button-link-delete sb-delete-row" data-key="">
            🗑️
          </button>
        </td>
      </tr>
    `;

    $('#sb-form-builder tbody').append(row);
    reindexRows();
  });

  /**
   * Delete Field
   */
  $(document).on('click', '.sb-delete-row', function(e){
    e.preventDefault();
    var $btn = $(this);
    var $tr  = $btn.closest('tr');
    var key  = $btn.data('key') || $tr.find('input[name*="[key]"]').val();

    if (window.confirm('Delete this field definition?')) {
      if (window.confirm('Also delete ALL existing data stored for this field across speaker profiles?\n\nThis cannot be undone.')) {
        if (key) {
          $('#sb-delete-meta-requests').append('<input type="hidden" name="sb_delete_meta_keys[]" value="'+ $('<div/>').text(key).html() +'">');
        }
      }
      $tr.remove();
      reindexRows();
    }
  });

  /**
   * Move row up
   */
  $(document).on('click', '.sb-move-up', function(){
    var $tr = $(this).closest('tr');
    $tr.prev().before($tr);
    reindexRows();
  });

  /**
   * Move row down
   */
  $(document).on('click', '.sb-move-down', function(){
    var $tr = $(this).closest('tr');
    $tr.next().after($tr);
    reindexRows();
  });

  /**
   * Sync select options on textarea change
   */
  $(document).on('input', '.sb-placeholder-options', function(){
    syncOptions($(this).closest('tr'));
  });

  /**
   * Change help text if type = select
   */
  $(document).on('change', '.sb-type', function(){
    var $tr = $(this).closest('tr');
    var type = $(this).val();
    if (type === 'select') {
      $tr.find('.sb-placeholder-help').text('Enter dropdown options (comma or newline separated).');
    } else {
      $tr.find('.sb-placeholder-help').text('Placeholder text (for Select: enter options, comma or newline separated).');
    }
  });

  /**
   * Before submit: sync all option textareas to JSON
   */
  $('#sb-form-builder').closest('form').on('submit', function(){
    $('#sb-form-builder tbody tr').each(function(){
      syncOptions($(this));
    });
    reindexRows();
  });

  /**
   * Geocoding progress with AJAX batch processing
   */
  $('#sb-rebuild-coordinates-btn').on('click', function(e) {
    e.preventDefault();

    var $btn = $(this);
    var $progress = $('#sb-geocode-progress');
    var $progressBar = $('#sb-geocode-progress-bar');
    var $progressText = $('#sb-geocode-progress-text');
    var $status = $('#sb-geocode-status');

    // Disable button and show progress
    $btn.prop('disabled', true).text('Processing...');
    $progress.show();
    $progressBar.css('width', '0%');
    $progressText.text('0%');
    $status.text('Getting speaker count...');

    var totalProcessed = 0;
    var totalUpdated = 0;
    var totalSkipped = 0;
    var totalErrors = 0;

    // Check if sbGeocodeData is defined
    if (typeof sbGeocodeData === 'undefined') {
      $status.text('Error: Configuration not loaded. Please refresh the page.');
      $btn.prop('disabled', false).text('Rebuild Coordinates');
      return;
    }

    // Step 1: Get count of speakers
    $.ajax({
      url: sbGeocodeData.ajaxurl,
      type: 'POST',
      data: {
        action: 'sb_get_geocode_count',
        nonce: sbGeocodeData.nonce
      },
      success: function(response) {
        if (!response.success) {
          $status.text('Error: ' + (response.data.message || 'Unknown error'));
          $btn.prop('disabled', false).text('Rebuild Coordinates');
          return;
        }

        var total = response.data.total;
        var speakerIds = response.data.speaker_ids;

        if (total === 0) {
          $status.text('No speakers found to geocode.');
          $btn.prop('disabled', false).text('Rebuild Coordinates');
          $progress.hide();
          return;
        }

        $status.text('Processing ' + total + ' speakers...');

        // Process in batches of 5 speakers at a time (to avoid timeout and respect API limits)
        var batchSize = 5;
        var batches = [];
        for (var i = 0; i < speakerIds.length; i += batchSize) {
          batches.push(speakerIds.slice(i, i + batchSize));
        }

        // Process batches sequentially
        processBatch(0);

        function processBatch(batchIndex) {
          if (batchIndex >= batches.length) {
            // All done!
            $progressBar.css('width', '100%');
            $progressText.text('100%');
            $status.html('<strong>Complete!</strong> Updated: ' + totalUpdated + ', Skipped: ' + totalSkipped + ', Errors: ' + totalErrors);
            $btn.prop('disabled', false).text('Rebuild Coordinates');
            setTimeout(function() {
              $progress.fadeOut();
            }, 3000);
            return;
          }

          var batch = batches[batchIndex];

          $.ajax({
            url: sbGeocodeData.ajaxurl,
            type: 'POST',
            data: {
              action: 'sb_geocode_batch',
              nonce: sbGeocodeData.nonce,
              speaker_ids: batch
            },
            success: function(response) {
              if (response.success) {
                totalProcessed += batch.length;
                totalUpdated += response.data.updated;
                totalSkipped += response.data.skipped;
                totalErrors += response.data.errors;

                var percent = Math.round((totalProcessed / total) * 100);
                $progressBar.css('width', percent + '%');
                $progressText.text(percent + '%');
                $status.text('Processed ' + totalProcessed + ' of ' + total + ' speakers... (Updated: ' + totalUpdated + ', Errors: ' + totalErrors + ')');

                // Process next batch
                processBatch(batchIndex + 1);
              } else {
                $status.text('Error in batch ' + (batchIndex + 1) + ': ' + (response.data.message || 'Unknown error'));
                $btn.prop('disabled', false).text('Rebuild Coordinates');
              }
            },
            error: function() {
              $status.text('Network error processing batch ' + (batchIndex + 1));
              $btn.prop('disabled', false).text('Rebuild Coordinates');
            }
          });
        }
      },
      error: function() {
        $status.text('Network error getting speaker count');
        $btn.prop('disabled', false).text('Rebuild Coordinates');
      }
    });
  });

});

/** END OF FILE: assets/sb-admin.js */
