<?php
/**
 * File: templates/speaker-register.php
 * Purpose: Front-end "Register Speaker" form with upload-only image fields, user creation, publish/draft logic.
 * Notes:
 *  - Renders all enabled Form Builder fields (skips Section 8 / Private).
 *  - Creates (or reuses) a WP user with role 'speaker' from the email field.
 *  - Post status: publish for admin submissions; draft for public submissions.
 *  - Image fields: Upload/Replace (local file only) + Remove. No numeric/URL box.
 *  - Sends emails to the new speaker and the admin.
 */

if (!defined('ABSPATH')) exit;

// Ensure role/shortcode bootstrap is available
$__sb_auth = dirname(__DIR__).'/includes/sb-auth-bootstrap.php';
if (file_exists($__sb_auth)) require_once $__sb_auth;

/** ====== LOAD LAYOUT & FIELDS ====== */
$defaults_layout = [
    1 => ['title'=>'Profile','heading'=>'h4','display'=>1,'hidden'=>0,'class'=>'sb-section-1'],
    2 => ['title'=>'Overview','heading'=>'h2','display'=>1,'hidden'=>0,'class'=>'sb-section-2'],
    3 => ['title'=>'Topics','heading'=>'h3','display'=>1,'hidden'=>0,'class'=>'sb-section-3'],
    4 => ['title'=>'Background','heading'=>'h3','display'=>1,'hidden'=>0,'class'=>'sb-section-4'],
    5 => ['title'=>'','heading'=>'h3','display'=>0,'hidden'=>1,'class'=>'sb-section-5'],
    6 => ['title'=>'','heading'=>'h3','display'=>0,'hidden'=>1,'class'=>'sb-section-6'],
    7 => ['title'=>'More Information','heading'=>'h2','display'=>1,'hidden'=>0,'class'=>'sb-section-7'],
    8 => ['title'=>'Private Fields','heading'=>'h3','display'=>0,'hidden'=>1,'class'=>'sb-section-8'],
];
$layout = get_option('sb_profile_layout', $defaults_layout);
foreach (range(1,8) as $i) {
    $layout[$i] = wp_parse_args(isset($layout[$i]) ? (array)$layout[$i] : [], $defaults_layout[$i]);
}

$fields = function_exists('sb_normalize_form_fields') ? sb_normalize_form_fields() : [];
if (!is_array($fields)) $fields = [];

// Group enabled fields by section (skip Section 8 on register form)
$fields_by_section = [];
foreach ($fields as $f) {
    if (empty($f['enabled'])) continue;
    $sec = isset($f['section']) ? (int) $f['section'] : 2;
    if ($sec < 1 || $sec > 8) $sec = 2;
    if ($sec === 8) continue;
    $fields_by_section[$sec][] = $f;
}

// Find profile email key (first enabled email field; fallback 'email')
$email_key = 'email';
foreach ($fields as $f) {
    if (!empty($f['enabled']) && ($f['type'] ?? '') === 'email' && !empty($f['key'])) {
        $email_key = $f['key'];
        break;
    }
}

/** CAPTCHA helper functions */
if (!function_exists('sb_generate_captcha')) {
function sb_generate_captcha() {
    $num1 = rand(1, 20);
    $num2 = rand(1, 20);
    $operations = ['+', '-'];
    $operation = $operations[array_rand($operations)];

    switch ($operation) {
        case '+':
            $answer = $num1 + $num2;
            $question = "$num1 + $num2";
            break;
        case '-':
            // Ensure positive result
            if ($num1 < $num2) {
                $temp = $num1;
                $num1 = $num2;
                $num2 = $temp;
            }
            $answer = $num1 - $num2;
            $question = "$num1 - $num2";
            break;
    }

    return [
        'question' => $question,
        'answer' => $answer,
        'hash' => wp_hash($answer . wp_salt())
    ];
}}

if (!function_exists('sb_verify_captcha')) {
function sb_verify_captcha($user_answer, $hash) {
    if (empty($user_answer) || empty($hash)) {
        return false;
    }

    $user_answer = intval($user_answer);
    $expected_hash = wp_hash($user_answer . wp_salt());

    return hash_equals($hash, $expected_hash);
}}

/** Upload helper (returns attachment ID) */
if (!function_exists('sb_handle_frontend_upload')) {
function sb_handle_frontend_upload($field_name, $attach_to_post) {
    if (!isset($_FILES[$field_name]) || empty($_FILES[$field_name]['name'])) {
        return new WP_Error('no_file', __('No file uploaded', 'speakers-bureau'));
    }
    require_once ABSPATH.'wp-admin/includes/file.php';
    require_once ABSPATH.'wp-admin/includes/image.php';
    require_once ABSPATH.'wp-admin/includes/media.php';

    $overrides = array('test_form' => false);
    $file = wp_handle_upload($_FILES[$field_name], $overrides);
    if (isset($file['error'])) {
        return new WP_Error('upload_error', $file['error']);
    }
    $filetype = wp_check_filetype(basename($file['file']), null);
    $attachment = array(
        'guid'           => $file['url'],
        'post_mime_type' => $filetype['type'],
        'post_title'     => sanitize_file_name(basename($file['file'])),
        'post_content'   => '',
        'post_status'    => 'inherit',
    );
    $attach_id = wp_insert_attachment($attachment, $file['file'], $attach_to_post);
    if (is_wp_error($attach_id)) return $attach_id;

    $attach_data = wp_generate_attachment_metadata($attach_id, $file['file']);
    wp_update_attachment_metadata($attach_id, $attach_data);
    return $attach_id;
}}
/** END helper */

/** ====== GENERATE CAPTCHA ====== */
$captcha = sb_generate_captcha();

/** ====== SAVE HANDLER ====== */
$notice = '';
$error  = '';
$is_logged_in = is_user_logged_in();
$is_admin = current_user_can('manage_options');

// Initialize verification system
$verification = class_exists('SB_Registration_Verification') ? new SB_Registration_Verification() : null;

if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['sb_register_nonce']) && wp_verify_nonce($_POST['sb_register_nonce'], 'sb_register')) {
    $speaker_name = isset($_POST['sb_post_title']) ? sanitize_text_field(wp_unslash($_POST['sb_post_title'])) : '';
    $email_val    = isset($_POST['sb_'.$email_key]) ? sanitize_email(wp_unslash($_POST['sb_'.$email_key])) : '';

    // CAPTCHA validation (only for logged-in users doing full registration)
    $captcha_hash = isset($_POST['sb_captcha_hash']) ? sanitize_text_field(wp_unslash($_POST['sb_captcha_hash'])) : '';
    $captcha_answer = isset($_POST['sb_captcha_answer']) ? sanitize_text_field(wp_unslash($_POST['sb_captcha_answer'])) : '';
    $captcha_valid = !($is_logged_in || $is_admin) || sb_verify_captcha($captcha_answer, $captcha_hash);

    if ($speaker_name === '') {
        $error = __('Please enter a speaker name.', 'speakers-bureau');
    } elseif (!$email_val) {
        $error = __('Please enter a valid email address.', 'speakers-bureau');
    } elseif (!$captcha_valid) {
        $error = __('Please solve the math problem correctly to prove you are human.', 'speakers-bureau');
    } else {
        // Handle different registration flows based on user status
        if (!$is_logged_in && !$is_admin) {
            // NON-LOGGED-IN USERS: Two-step verification process
            if (!$verification) {
                $error = __('Registration system not available.', 'speakers-bureau');
            } elseif ($verification->has_pending_registration($email_val)) {
                $error = __('A verification email has already been sent to this address. Please check your email and follow the verification link.', 'speakers-bureau');
            } else {
                // For simplified two-step registration, only collect name and email
                $form_data = [
                    'speaker_name' => $speaker_name,
                    'email' => $email_val
                ];

                // Create pending registration and send verification email
                $uuid = $verification->create_pending_registration($email_val, $speaker_name, $form_data);
                if ($uuid && $verification->send_verification_email($uuid, $email_val, $speaker_name)) {
                    $notice = __('Thank you! A verification email has been sent to your email address. Please check your email and click the verification link to complete your registration.', 'speakers-bureau');
                } else {
                    $error = __('There was an error sending the verification email. Please try again.', 'speakers-bureau');
                }
            }
        } else {
            // LOGGED-IN USERS AND ADMINS: Original immediate registration process
            $user = get_user_by('email', $email_val);
            if (!$user) {
                $prefix   = sanitize_user(current(explode('@', $email_val)));
                if ($prefix === '') $prefix = 'speaker';
                $username = $prefix;
                $i = 1;
                while (username_exists($username)) { $username = $prefix . $i++; }
                $password = wp_generate_password(20, true, true);
                $user_id  = wp_create_user($username, $password, $email_val);
                if (is_wp_error($user_id)) {
                    $error = $user_id->get_error_message();
                    $user_id = 0;
                } else {
                    $user   = get_user_by('id', $user_id);
                    $user->set_role('speaker');
                }
            } else {
                $user_id = $user->ID;
                if (!in_array('speaker', (array)$user->roles, true)) $user->add_role('speaker');
            }

            if (!$error && $user_id) {
                // Check if auto-publish is enabled
                $display_settings = get_option('sb_display_settings', []);
                $auto_publish = !empty($display_settings['auto_publish_profiles']);

                // Helper function to check if required fields are completed
                $required_fields_completed = function() use ($fields) {
                    foreach ($fields as $f) {
                        if (!empty($f['required']) && !empty($f['enabled'])) {
                            $key = $f['key'] ?? '';
                            if ($key && empty($_POST['sb_'.$key])) {
                                return false; // Required field is empty
                            }
                        }
                    }
                    return true; // All required fields are filled
                };

                // Determine post status
                if (current_user_can('manage_options')) {
                    $status = 'publish'; // Admins always publish
                } elseif ($auto_publish && $required_fields_completed()) {
                    $status = 'publish'; // Auto-publish if enabled and required fields are complete
                } else {
                    $status = 'draft'; // Default to draft
                }

                $new_post_id = wp_insert_post([
                    'post_type'   => 'speaker',
                    'post_title'  => $speaker_name,
                    'post_status' => $status,
                    'post_author' => $user_id,
                ], true);

                if (is_wp_error($new_post_id)) {
                    $error = $new_post_id->get_error_message();
                } else {
                    foreach ($fields as $f) {
                        if (empty($f['enabled'])) continue;
                        $sec = isset($f['section']) ? (int) $f['section'] : 2;
                        if ($sec === 8) continue;
                        $type = $f['type'] ?? 'text';
                        $key  = $f['key']  ?? '';
                        if ($key === '') continue;

                        if ($type === 'image') {
                            $file_field   = 'sb_'.$key.'_file';
                            $remove_field = 'sb_'.$key.'_remove';
                            $wants_remove = !empty($_POST[$remove_field]);
                            if ($wants_remove) { delete_post_meta($new_post_id, $key); continue; }

                            if (isset($_FILES[$file_field]) && !empty($_FILES[$file_field]['name'])) {
                                $res = sb_handle_frontend_upload($file_field, $new_post_id);
                                if (!is_wp_error($res)) {
                                    update_post_meta($new_post_id, $key, (int)$res);
                                }
                            }
                            continue;
                        }

                        $name = 'sb_'.$key;
                        if ($type === 'checkbox' || (!empty($f['multiple']))) {
                            $val = isset($_POST[$name]) ? (array) $_POST[$name] : [];
                            $val = array_values(array_filter(array_map('sanitize_text_field', wp_unslash($val))));
                            update_post_meta($new_post_id, $key, $val);
                        } else {
                            $raw = isset($_POST[$name]) ? wp_unslash($_POST[$name]) : '';
                            if (is_array($raw)) $raw = implode(', ', $raw);
                            $raw = trim((string)$raw);
                            switch ($type) {
                                case 'email':
                                    $val = sanitize_email($raw);
                                    break;
                                case 'url':
                                    $val = esc_url_raw($raw);
                                    break;
                                case 'textarea':
                                    // Preserve line breaks for textareas
                                    $val = sanitize_textarea_field($raw);
                                    break;
                                default:
                                    $val = sanitize_text_field($raw);
                                    break;
                            }
                            if ($val !== '') update_post_meta($new_post_id, $key, $val);
                        }

                        // Save privacy preferences for email and phone fields
                        if (in_array($type, ['email', 'phone', 'tel'])) {
                            $privacy_name = 'sb_'.$key.'_private';
                            if (!empty($_POST[$privacy_name])) {
                                update_post_meta($new_post_id, $key.'_private', '1');
                            }
                        }
                    }

                    // Send emails using template system
                    if (function_exists('sb_send_speaker_confirmation_email') && function_exists('sb_send_admin_notification_email')) {
                        sb_send_speaker_confirmation_email($new_post_id, $email_val, $speaker_name);
                        sb_send_admin_notification_email($new_post_id, $speaker_name);
                    } else {
                        // Fallback to original email system
                        $admin_email = get_option('admin_email');
                        $login_page  = get_page_by_path('account-login');
                        $login_url   = $login_page ? get_permalink($login_page) : wp_login_url();
                        $profile_url = home_url('/my-profile/?speaker_id=' . $new_post_id);

                        $subject_u = sprintf(__('Your %s profile','speakers-bureau'), get_bloginfo('name'));
                        $body_u    = "Hi,\n\nA speaker profile was created for you on ".get_bloginfo('name').".\n"
                                   . "Login: {$login_url}\n"
                                   . "Edit your profile: {$profile_url}\n\n"
                                   . "If you need a password, use: ".wp_lostpassword_url()."\n";
                        wp_mail($email_val, $subject_u, $body_u);

                        $subject_a = __('New speaker submitted','speakers-bureau');
                        $body_a    = "Title: {$speaker_name}\nStatus: {$status}\nAuthor user ID: {$user_id}\nProfile: {$profile_url}\n";
                        $admin_email = get_option('admin_email');
                        wp_mail($admin_email, $subject_a, $body_a);
                    }

                    $notice = ($status === 'publish')
                        ? __('Speaker published and account created.','speakers-bureau')
                        : __('Speaker saved as draft and account created.','speakers-bureau');
                }
            }
        }
    }
}

/** ====== Documentation helper ====== */
if (!function_exists('sb_render_speaker_guide_content')) {
function sb_render_speaker_guide_content() {
    $guide_file = plugin_dir_path(dirname(__FILE__)) . 'SPEAKER-WALKTHROUGH.md';

    if (!file_exists($guide_file)) {
        return '<p>' . esc_html__('Speaker guide not available.', 'speakers-bureau') . '</p>';
    }

    $content = file_get_contents($guide_file);
    if ($content === false) {
        return '<p>' . esc_html__('Could not load speaker guide.', 'speakers-bureau') . '</p>';
    }

    // Convert markdown to HTML (simplified version for frontend)
    return sb_simple_markdown_to_html($content);
}}

if (!function_exists('sb_simple_markdown_to_html')) {
function sb_simple_markdown_to_html($markdown) {
    $html = $markdown;

    // Code blocks
    $html = preg_replace('/```([a-z]*)\n(.*?)\n```/s', '<pre style="background: #f4f4f4; padding: 12px; border-radius: 4px; overflow-x: auto;"><code>$2</code></pre>', $html);

    // Inline code
    $html = preg_replace('/`([^`]+)`/', '<code style="background: #f4f4f4; padding: 2px 4px; border-radius: 3px; font-family: monospace;">$1</code>', $html);

    // Headers
    $html = preg_replace('/^### (.+)$/m', '<h3 style="color: #0073aa; margin-top: 20px; margin-bottom: 10px;">$1</h3>', $html);
    $html = preg_replace('/^## (.+)$/m', '<h2 style="color: #0073aa; margin-top: 24px; margin-bottom: 12px; border-bottom: 2px solid #0073aa; padding-bottom: 6px;">$1</h2>', $html);
    $html = preg_replace('/^# (.+)$/m', '<h1 style="color: #0073aa; margin-top: 24px; margin-bottom: 16px; border-bottom: 3px solid #0073aa; padding-bottom: 8px;">$1</h1>', $html);

    // Bold and italic
    $html = preg_replace('/\*\*(.+?)\*\*/', '<strong>$1</strong>', $html);
    $html = preg_replace('/\*(.+?)\*/', '<em>$1</em>', $html);

    // Links
    $html = preg_replace('/\[(.+?)\]\((.+?)\)/', '<a href="$2" target="_blank" style="color: #0073aa;">$1</a>', $html);

    // Lists
    $html = preg_replace('/^- (.+)$/m', '<li style="margin: 4px 0;">$1</li>', $html);
    $html = preg_replace('/^(\d+)\. (.+)$/m', '<li style="margin: 4px 0;">$2</li>', $html);

    // Wrap consecutive <li> elements in <ul>
    $html = preg_replace('/(<li>.*<\/li>)/s', '<ul style="margin: 12px 0; padding-left: 24px;">$1</ul>', $html);
    $html = preg_replace('/<\/ul>\s*<ul[^>]*>/', '', $html);

    // Checkboxes and special formatting
    $html = preg_replace('/- ☑️/', '<li style="list-style: none; margin: 4px 0;">✅', $html);
    $html = preg_replace('/- ✅/', '<li style="list-style: none; margin: 4px 0;">✅', $html);

    // Paragraphs
    $html = preg_replace('/\n\n/', '</p><p style="margin: 12px 0; line-height: 1.6;">', $html);
    $html = '<p style="margin: 12px 0; line-height: 1.6;">' . $html . '</p>';

    // Clean up empty paragraphs and fix header/list formatting
    $html = preg_replace('/<p[^>]*><\/p>/', '', $html);
    $html = preg_replace('/<p[^>]*>(<h[1-6][^>]*>)/i', '$1', $html);
    $html = preg_replace('/(<\/h[1-6]>)<\/p>/i', '$1', $html);
    $html = preg_replace('/<p[^>]*>(<ul[^>]*>)/i', '$1', $html);
    $html = preg_replace('/(<\/ul>)<\/p>/i', '$1', $html);
    $html = preg_replace('/<p[^>]*>(<pre[^>]*>)/i', '$1', $html);
    $html = preg_replace('/(<\/pre>)<\/p>/i', '$1', $html);

    return $html;
}}

/** ====== Render helpers ======*/
if (!function_exists('sb_fb_parse_options')) {
function sb_fb_parse_options($raw) {
    $out = [];
    $raw = trim((string) $raw);
    if ($raw === '') return $out;

    // Support both comma-separated and line-separated options
    $items = [];
    if (strpos($raw, "\n") !== false) {
        // Line-separated format
        $items = preg_split('/\r\n|\r|\n/', $raw);
    } else {
        // Comma-separated format
        $items = explode(',', $raw);
    }

    foreach ($items as $item) {
        $item = trim($item);
        if ($item === '') continue;

        // Support "Label|Value" format
        if (strpos($item, '|') !== false) {
            list($l, $v) = array_map('trim', explode('|', $item, 2));
            if ($v === '') $v = $l;
        } else {
            $l = $item;
            $v = $item;
        }
        $out[] = ['label' => $l, 'value' => $v];
    }
    return $out;
}}

if (!function_exists('sb_render_register_control')) {
function sb_render_register_control($field) {
    $type   = $field['type'] ?? 'text';
    $key    = $field['key']  ?? '';
    $label  = $field['label'] ?? '';
    if ($key === '') return '';

    $name = 'sb_'.$key;

    ob_start();
    echo '<div class="sb-input sb-input-'.esc_attr($type).'">';
    if (empty($field['hide_label']) && $label !== '') {
        echo '<label for="sb_'.esc_attr($key).'" class="sb-label">'.esc_html($label).'</label>';
    }

    // Get placeholder text from field configuration
    $placeholder_text = !empty($field['placeholder']) ? esc_attr($field['placeholder']) : '';
    $placeholder_attr = $placeholder_text ? ' placeholder="' . $placeholder_text . '"' : '';

    switch ($type) {
        case 'textarea':
            echo '<textarea id="sb_'.esc_attr($key).'" name="'.esc_attr($name).'" rows="5" style="width:100%;"' . $placeholder_attr . '></textarea>';
            break;

        case 'email':
            echo '<input type="email" id="sb_'.esc_attr($key).'" name="'.esc_attr($name).'" value="" style="width:100%;"' . $placeholder_attr . '>';
            // Add privacy checkbox for email fields
            echo '<div class="sb-privacy-option" style="margin-top:8px;">';
            echo '<label for="sb_'.esc_attr($key).'_private" style="display:flex;align-items:center;font-size:13px;color:#666;">';
            echo '<input type="checkbox" id="sb_'.esc_attr($key).'_private" name="sb_'.esc_attr($key).'_private" value="1" style="margin-right:6px;"> ';
            echo esc_html__("Keep this email address private (won't be shown on public profile)", 'speakers-bureau');
            echo '</label></div>';
            break;

        case 'url':
            echo '<input type="url" id="sb_'.esc_attr($key).'" name="'.esc_attr($name).'" value="" style="width:100%;"' . $placeholder_attr . '>';
            break;

        case 'tel':
            echo '<input type="tel" id="sb_'.esc_attr($key).'" name="'.esc_attr($name).'" value="" style="width:100%;"' . $placeholder_attr . '>';
            // Add privacy checkbox for tel fields
            echo '<div class="sb-privacy-option" style="margin-top:8px;">';
            echo '<label for="sb_'.esc_attr($key).'_private" style="display:flex;align-items:center;font-size:13px;color:#666;">';
            echo '<input type="checkbox" id="sb_'.esc_attr($key).'_private" name="sb_'.esc_attr($key).'_private" value="1" style="margin-right:6px;"> ';
            echo esc_html__("Keep this phone number private (won't be shown on public profile)", 'speakers-bureau');
            echo '</label></div>';
            break;

        case 'phone':
            echo '<input type="tel" id="sb_'.esc_attr($key).'" name="'.esc_attr($name).'" value="" style="width:100%;"' . $placeholder_attr . '>';
            // Add privacy checkbox for phone fields
            echo '<div class="sb-privacy-option" style="margin-top:8px;">';
            echo '<label for="sb_'.esc_attr($key).'_private" style="display:flex;align-items:center;font-size:13px;color:#666;">';
            echo '<input type="checkbox" id="sb_'.esc_attr($key).'_private" name="sb_'.esc_attr($key).'_private" value="1" style="margin-right:6px;"> ';
            echo esc_html__("Keep this phone number private (won't be shown on public profile)", 'speakers-bureau');
            echo '</label></div>';
            break;

        case 'select':
            $options = sb_fb_parse_options($field['placeholder'] ?? '');
            echo '<select id="sb_'.esc_attr($key).'" name="'.esc_attr($name).'" style="width:100%;">';
            echo '<option value="">'.esc_html__('— Select —','speakers-bureau').'</option>';
            foreach ($options as $o) {
                echo '<option value="'.esc_attr($o['value']).'">'.esc_html($o['label']).'</option>';
            }
            echo '</select>';
            break;

        case 'checkbox':
            $options = sb_fb_parse_options($field['placeholder'] ?? '');
            foreach ($options as $idx => $o) {
                $id = 'sb_'.esc_attr($key).'_'.$idx;
                echo '<label for="'.$id.'" style="display:block;margin:2px 0;">';
                echo '<input type="checkbox" id="'.$id.'" name="'.esc_attr($name).'[]" value="'.esc_attr($o['value']).'"> ';
                echo esc_html($o['label']).'</label>';
            }
            break;

        case 'image':
            $file_field   = 'sb_'.esc_attr($key).'_file';
            $remove_field = 'sb_'.esc_attr($key).'_remove';
            echo '<div class="sb-image-field" id="sb_'.esc_attr($key).'_wrap">';
            echo '  <div class="sb-image-controls" style="display:flex;gap:8px;align-items:center;">';
            echo '    <input type="file" accept="image/*" name="'.$file_field.'">';

            $js = "var f=document.querySelector('input[name=\"".$remove_field."\"]');"
                . "if(f){f.value=1;}"
                . "var w=this.closest('.sb-image-field');"
                . "if(w){var p=w.querySelector('.sb-image-preview'); if(p){p.innerHTML=''; p.style.display='none';}}";
            echo '    <button type="button" class="button button-secondary" onclick="'.esc_attr($js).'">'.esc_html__('Remove','speakers-bureau').'</button>';
            echo '    <input type="hidden" name="'.$remove_field.'" value="">';
            echo '  </div>';
            echo '  <div class="sb-image-preview" style="margin-top:8px; display:none;"></div>';
            echo '</div>';
            break;

        default:
            echo '<input type="text" id="sb_'.esc_attr($key).'" name="'.esc_attr($name).'" value="" style="width:100%;"' . $placeholder_attr . '>';
            break;
    }

    echo '</div>';
    return ob_get_clean();
}}
/** END helpers */

/** ====== Page Markup (fragment) ====== */
?>
<div class="sb-wrap sb-register">

    <?php if (!$is_logged_in): ?>
        <div class="sb-registration-info" style="margin-bottom: 24px; padding: 20px; background: #f8f9fa; border-radius: 6px; border-left: 4px solid #0073aa;">
            <h3 style="margin-top: 0; color: #0073aa;">
                <?php esc_html_e('Two-Step Registration Process', 'speakers-bureau'); ?>
            </h3>
            <p style="margin-bottom: 12px;">
                <?php esc_html_e('To ensure security and verify your email address, our registration process has two steps:', 'speakers-bureau'); ?>
            </p>
            <ol style="margin: 12px 0; padding-left: 24px;">
                <li><?php esc_html_e('Fill out this form with your details', 'speakers-bureau'); ?></li>
                <li><?php esc_html_e('Check your email for a verification link', 'speakers-bureau'); ?></li>
                <li><?php esc_html_e('Click the link to set your password and complete registration', 'speakers-bureau'); ?></li>
            </ol>
            <p style="margin-bottom: 0; font-size: 14px; color: #666;">
                <?php esc_html_e('Note: Image uploads will be available after you complete the verification process.', 'speakers-bureau'); ?>
            </p>
        </div>
    <?php endif; ?>

    <?php if ($notice): ?>
        <div class="notice notice-success is-dismissible" style="margin:16px 0;"><p><?php echo esc_html($notice); ?></p></div>
    <?php elseif ($error): ?>
        <div class="notice notice-error is-dismissible" style="margin:16px 0;"><p><?php echo esc_html($error); ?></p></div>
    <?php endif; ?>

    <form method="post" action="" enctype="multipart/form-data">
        <?php wp_nonce_field('sb_register', 'sb_register_nonce'); ?>

        <!-- Required: Speaker Name -->
        <div class="sb-input sb-input-text">
            <label for="sb_post_title" class="sb-label"><?php esc_html_e('Speaker Name','speakers-bureau'); ?></label>
            <input type="text" id="sb_post_title" name="sb_post_title" value="" style="width:100%;" required>
        </div>

        <?php if ($is_logged_in || $is_admin): ?>
            <!-- Full form for logged-in users -->
            <div class="sb-register-grid">
                <?php foreach ([1,2,3,4,5,6,7] as $sec):
                    $class   = !empty($layout[$sec]['class']) ? $layout[$sec]['class'] : 'sb-section-'.$sec;
                    $title   = isset($layout[$sec]['title']) ? trim($layout[$sec]['title']) : '';
                    $heading = isset($layout[$sec]['heading']) ? strtolower($layout[$sec]['heading']) : 'h3';
                    if (!in_array($heading, ['h1','h2','h3','h4','h5','h6'], true)) $heading = 'h3';
                    $sec_fields  = $fields_by_section[$sec] ?? [];
                    if (!$sec_fields) continue; ?>
                <section class="<?php echo esc_attr($class); ?>">
                    <?php if ($title !== ''): ?>
                        <<?php echo esc_attr($heading); ?> class="sb-section-title"><?php echo wp_kses_post($title); ?></<?php echo esc_attr($heading); ?>>
                    <?php endif; ?>

                    <?php
                    foreach ($sec_fields as $f) {
                        if (($f['type'] ?? '') === 'edit_link') continue; // not applicable on register form
                        echo sb_render_register_control($f);
                    }
                    ?>
                </section>
                <?php endforeach; ?>
            </div>
        <?php else: ?>
            <!-- Simplified form for non-logged users: just email field -->
            <div class="sb-simplified-register">
                <div class="sb-input sb-input-email">
                    <label for="sb_<?php echo esc_attr($email_key); ?>" class="sb-label"><?php esc_html_e('Email Address', 'speakers-bureau'); ?></label>
                    <input type="email" id="sb_<?php echo esc_attr($email_key); ?>" name="sb_<?php echo esc_attr($email_key); ?>" value="" style="width:100%;" required placeholder="<?php esc_attr_e('your.email@example.com', 'speakers-bureau'); ?>">
                    <p class="description" style="margin-top: 5px; color: #666; font-size: 13px;">
                        <?php esc_html_e('This will be your login email address. You\'ll receive a verification email to continue.', 'speakers-bureau'); ?>
                    </p>
                </div>
            </div>
        <?php endif; ?>

        <?php if (($is_logged_in || $is_admin) && !$notice): ?>
            <!-- CAPTCHA Section for full registration (logged-in users completing full forms) -->
            <div class="sb-captcha-section" style="background: #f0f8ff; border: 1px solid #0073aa; border-radius: 6px; padding: 20px; margin: 20px 0;">
                <h4 style="margin-top: 0; color: #0073aa; display: flex; align-items: center;">
                    <span style="font-size: 20px; margin-right: 8px;">🤖</span>
                    <?php esc_html_e('Security Check', 'speakers-bureau'); ?>
                </h4>
                <p style="margin: 10px 0; color: #333;">
                    <?php esc_html_e('Please solve this simple math problem to verify you are human:', 'speakers-bureau'); ?>
                </p>
                <div style="display: flex; align-items: center; gap: 10px; flex-wrap: wrap;">
                    <label for="sb_captcha_answer" style="font-size: 16px; font-weight: 600; color: #0073aa;">
                        <?php echo esc_html($captcha['question']); ?> =
                    </label>
                    <input type="number"
                           id="sb_captcha_answer"
                           name="sb_captcha_answer"
                           required
                           style="width: 80px; padding: 8px; border: 2px solid #0073aa; border-radius: 4px; font-size: 16px; text-align: center;"
                           placeholder="?">
                    <input type="hidden" name="sb_captcha_hash" value="<?php echo esc_attr($captcha['hash']); ?>">
                </div>
                <p class="description" style="margin: 10px 0 0 0; font-size: 13px; color: #666;">
                    <?php esc_html_e('This helps us prevent automated spam registrations.', 'speakers-bureau'); ?>
                </p>
            </div>
        <?php endif; ?>

        <p>
            <button type="submit" class="button button-primary">
                <?php
                if ($is_logged_in || $is_admin) {
                    esc_html_e('Submit Registration','speakers-bureau');
                } else {
                    esc_html_e('Send Verification Email','speakers-bureau');
                }
                ?>
            </button>
        </p>
    </form>

    <div class="sb-help-section" style="margin-top: 32px; padding: 16px; background: #f8f9fa; border-radius: 4px; border-left: 4px solid #0073aa;">
        <h3 style="margin-top: 0; color: #0073aa;"><?php esc_html_e('Need Help?', 'speakers-bureau'); ?></h3>
        <p style="margin-bottom: 12px;"><?php esc_html_e('For detailed instructions on registration and profile management:', 'speakers-bureau'); ?></p>
        <p>
            <button type="button" id="sb-show-guide" class="button button-secondary">
                <?php esc_html_e('Show Registration Guide', 'speakers-bureau'); ?>
            </button>
        </p>

        <div id="sb-guide-content" style="display: none; margin-top: 16px; padding: 20px; background: #fff; border: 1px solid #ddd; border-radius: 4px; max-height: 400px; overflow-y: auto;">
            <?php echo sb_render_speaker_guide_content(); ?>
        </div>
    </div>

    <script>
    document.addEventListener('DOMContentLoaded', function() {
        var showButton = document.getElementById('sb-show-guide');
        var guideContent = document.getElementById('sb-guide-content');

        if (showButton && guideContent) {
            showButton.addEventListener('click', function() {
                if (guideContent.style.display === 'none') {
                    guideContent.style.display = 'block';
                    showButton.textContent = '<?php echo esc_js(__('Hide Registration Guide', 'speakers-bureau')); ?>';
                } else {
                    guideContent.style.display = 'none';
                    showButton.textContent = '<?php echo esc_js(__('Show Registration Guide', 'speakers-bureau')); ?>';
                }
            });
        }

        // CAPTCHA enhancement
        var captchaInput = document.getElementById('sb_captcha_answer');
        if (captchaInput) {
            // Focus on CAPTCHA when form is near submission
            captchaInput.addEventListener('input', function() {
                if (this.value.length > 0) {
                    this.style.borderColor = '#00a32a';
                    this.style.backgroundColor = '#f0fff0';
                } else {
                    this.style.borderColor = '#0073aa';
                    this.style.backgroundColor = '#fff';
                }
            });

            // Validate on blur
            captchaInput.addEventListener('blur', function() {
                if (this.value.length === 0) {
                    this.style.borderColor = '#dc3232';
                    this.style.backgroundColor = '#fff8f8';
                }
            });
        }
    });
    </script>

    <style>
    .sb-captcha-section {
        animation: slideIn 0.3s ease-out;
    }

    @keyframes slideIn {
        from {
            opacity: 0;
            transform: translateY(-10px);
        }
        to {
            opacity: 1;
            transform: translateY(0);
        }
    }

    .sb-captcha-section input[type="number"]:focus {
        outline: none;
        border-color: #005a87;
        box-shadow: 0 0 0 2px rgba(0, 115, 170, 0.2);
        transform: scale(1.05);
        transition: all 0.2s ease;
    }

    .sb-captcha-section input[type="number"] {
        transition: all 0.2s ease;
    }

    @media (max-width: 600px) {
        .sb-captcha-section > div {
            flex-direction: column;
            align-items: flex-start;
            gap: 8px;
        }

        .sb-captcha-section input[type="number"] {
            width: 100px;
        }
    }
    </style>
</div>
<?php
/** END OF FILE: templates/speaker-register.php */
