<?php
/**
 * Plugin Name: Speakers Bureau
 * Plugin URI:  http://www.garryheath.com/plugins
 * Description: Speakers directory (CPT, shortcodes, settings, form builder, templates).
 * Version:     3.3.0
 * Author:      Garry Heath
 * License:     GPLv2 or later
 *
 * File: speakers-bureau.php
 * Purpose: Main bootstrap that loads all plugin modules and wires the activation-only migrations.
 *
 * UPDATE (2025-09-24) - Version 3.3.0:
 * - Added Speaker Topics Index Gutenberg block for organized topic browsing
 * - Created [speaker_topics_index] shortcode with customizable display options
 * - Enhanced topic extraction and processing from speaker profiles
 * - Added responsive grid layout with search integration
 * - Improved block editor integration with custom category
 *
 * UPDATE (2025-09-23) - Version 3.2.0:
 * - Fixed pagination functionality in speaker directory
 * - Enhanced phone number consolidation in import process
 * - Added auto-publish option for completed speaker profiles
 * - Fixed deprecated PHP warnings for WordPress 6.x compatibility
 * - Added Yoast SEO compatibility and null title prevention
 * - Improved search parameter handling on profile pages
 * - Enhanced auto-import functionality with progress tracking
 *
 * UPDATE (2025-09-22) - Version 3.1.0:
 * - Added privacy controls for email and phone fields
 * - Enhanced HTML email system with rich text editing
 * - Added profile update email notifications with speaker listing
 * - New Display Settings tab for complete directory customization
 * - Added CAPTCHA anti-spam protection for registration
 * - Improved import functionality with Rotary 5320 integration
 * - Enhanced user experience with better styling and responsiveness
 *
 * Previous UPDATE (2025-09-20):
 * - Restored/includes all module files (admin, helpers, shortcodes, settings, CPT, form builder, etc).
 * - Migrations are required but run ONLY on activation (no admin_init/global hooks).
 * - Kept behavior compatible with previous versions (no dropped features).
 */

if (!defined('ABSPATH')) exit;

/** Constants */
define('SB_PLUGIN_FILE', __FILE__);
define('SB_PLUGIN_DIR', plugin_dir_path(__FILE__));
define('SB_PLUGIN_URL', plugin_dir_url(__FILE__));
if (!defined('SB_PLUGIN_VER')) {
    define('SB_PLUGIN_VER', '3.3.0');
}

/**
 * Defensive loader: include a file if it exists (prevents fatals on installs that may
 * not have all optional modules).
 */
function sb_require($relative) {
    $path = SB_PLUGIN_DIR . ltrim($relative, '/');
    if (file_exists($path)) {
        require_once $path;
        return true;
    }
    return false;
}

/**
 * Load core modules — order matters a little:
 * 1) CPT first (registers the speaker post type),
 * 2) assets (enqueue CSS/JS),
 * 3) form builder class (so settings tab can find SB_Form_Builder),
 * 4) settings (registers admin menu),
 * 5) shortcodes last.
 */
sb_require('includes/sb-cpt.php');           // registers the `speaker` CPT
sb_require('includes/sb-auth-bootstrap.php'); // authentication, roles, and login shortcodes
sb_require('includes/sb-assets.php');        // enqueue frontend/admin CSS/JS
sb_require('includes/sb-meta-boxes.php');    // meta boxes and form handling
sb_require('includes/sb-form-builder.php');  // defines class SB_Form_Builder
sb_require('includes/sb-settings.php');      // admin settings page (adds "Speakers Bureau" menu)
sb_require('includes/sb-shortcodes.php');    // registers [speaker_list], [speaker_register], [speaker_edit]
sb_require('includes/sb-templates.php');     // template overrides
sb_require('includes/sb-columns.php');       // admin columns customization
sb_require('includes/sb-search.php');        // search helpers
sb_require('includes/sb-geocoding.php');     // geocoding functionality
sb_require('includes/sb-radius-search.php'); // radius/distance search
sb_require('includes/sb-address-fields.php'); // address field helpers
sb_require('includes/sb-registration-verification.php'); // two-step registration verification
sb_require('includes/sb-rotary-import.php'); // Rotary 5320 speaker import functionality
sb_require('includes/sb-blocks.php');        // Gutenberg blocks for speaker content

/**
 * Activation-only migrations.
 * We load the file unconditionally, but DO NOT hook migrations globally.
 */
sb_require('includes/sb-migrations.php');
if (function_exists('register_activation_hook') && class_exists('SB_Migrations')) {
    register_activation_hook(__FILE__, ['SB_Migrations', 'run']);
}

/**
 * Optional defensive shims (only define if missing) to avoid fatals
 * should a site have older templates referencing these.
 * They *do not* replace your “real” implementations in helpers.
 */
if (!function_exists('sb_normalize_form_fields')) {
    function sb_normalize_form_fields($fields = null) {
        if ($fields === null) {
            $fields = get_option('sb_form_fields', []);
        }
        if (is_string($fields)) {
            $decoded = json_decode($fields, true);
            return is_array($decoded) ? $decoded : [];
        }
        return is_array($fields) ? $fields : [];
    }
}
if (!function_exists('sb_get_searchable_fields')) {
    function sb_get_searchable_fields() {
        $fields = get_option('sb_form_fields', []);
        $fields = sb_normalize_form_fields($fields);
        $keys = [];
        foreach ($fields as $field) {
            if (!empty($field['search']) && !empty($field['key'])) {
                $keys[] = $field['key'];
            }
        }
        return $keys;
    }
}

/**
 * Exclude attachment posts from WordPress search results globally
 * This prevents image files from appearing in search results
 */
function sb_exclude_attachments_from_search($query) {
    if (!is_admin() && $query->is_search() && $query->is_main_query()) {
        // Get the current post types being searched
        $post_types = $query->get('post_type');

        // If no post types specified, WordPress searches all public post types
        if (empty($post_types)) {
            // Get all public post types except attachments
            $public_post_types = get_post_types(['public' => true]);
            unset($public_post_types['attachment']); // Remove attachments
            $query->set('post_type', array_values($public_post_types));
        } else {
            // If post types are specified, ensure attachment is not included
            if (is_array($post_types)) {
                $post_types = array_diff($post_types, ['attachment']);
                $query->set('post_type', $post_types);
            } elseif ($post_types === 'attachment') {
                // If searching only attachments, return no results
                $query->set('post_type', 'nonexistent_post_type');
            }
        }
    }
}
add_action('pre_get_posts', 'sb_exclude_attachments_from_search');

/**
 * Suppress EXIF warnings for images with malformed EXIF data
 * This prevents "Incorrect APP1 Exif Identifier Code" warnings during image uploads
 */
function sb_suppress_exif_warnings() {
    // Only suppress during image processing in admin
    if (is_admin() && (
        (isset($_POST['action']) && $_POST['action'] === 'upload-attachment') ||
        (isset($_GET['action']) && $_GET['action'] === 'upload') ||
        doing_action('wp_handle_upload') ||
        doing_action('media_handle_upload')
    )) {
        // Temporarily disable exif_read_data warnings
        set_error_handler(function($errno, $errstr, $errfile, $errline) {
            // Only suppress EXIF-related warnings from WordPress core image.php
            if ($errno == E_WARNING &&
                strpos($errstr, 'exif_read_data') !== false &&
                strpos($errstr, 'Incorrect APP1 Exif Identifier Code') !== false &&
                strpos($errfile, 'wp-admin/includes/image.php') !== false) {
                return true; // Suppress this warning
            }
            return false; // Let other warnings through
        }, E_WARNING);
    }
}
add_action('init', 'sb_suppress_exif_warnings');

/**
 * Handle custom sorting for speaker lists
 */
function sb_handle_custom_speaker_sorting($query) {
    // Only affect main queries for speaker post type
    if (!$query->is_main_query() || $query->get('post_type') !== 'speaker' || is_admin()) {
        return;
    }

    $orderby = $query->get('orderby');

    // Handle custom default sorting
    if ($orderby === 'sb_default_sort') {
        // Default sort: Date added (post_date) + Last updated (post_modified) + Views
        // Create a complex ordering that prioritizes recently added/updated posts with views

        // Use SQL to create a composite score: recent posts get higher scores
        global $wpdb;
        $query->set('orderby', 'sb_score');
        $query->set('meta_key', '_speaker_views');

        add_filter('posts_orderby', function($orderby_sql) use ($wpdb) {
            // Create a scoring system that combines:
            // 1. Days since creation (newer = higher score)
            // 2. Days since last update (more recent = higher score)
            // 3. View count (more views = higher score)
            $orderby_sql = "
                (
                    DATEDIFF(NOW(), {$wpdb->posts}.post_date) * -0.1 +
                    DATEDIFF(NOW(), {$wpdb->posts}.post_modified) * -0.2 +
                    COALESCE(mt1.meta_value, 0) * 0.1
                ) DESC, {$wpdb->posts}.post_date DESC
            ";
            return $orderby_sql;
        }, 10, 1);

        // Add the meta join for view counts
        add_filter('posts_join', function($join) use ($wpdb) {
            $join .= " LEFT JOIN {$wpdb->postmeta} mt1 ON ({$wpdb->posts}.ID = mt1.post_id AND mt1.meta_key = '_speaker_views') ";
            return $join;
        }, 10, 1);
    }

    // Handle custom popular sorting
    if ($orderby === 'sb_popular_sort') {
        global $wpdb;

        // Override the orderby to handle view count sorting with NULL values
        add_filter('posts_orderby', function($orderby_sql) use ($wpdb) {
            // Sort by view count, treating NULL/missing values as 0, then by date
            $orderby_sql = "COALESCE(mt_views.meta_value, 0) + 0 DESC, {$wpdb->posts}.post_date DESC";
            return $orderby_sql;
        }, 10, 1);

        // Add the meta join for view counts (LEFT JOIN to include posts without views)
        add_filter('posts_join', function($join) use ($wpdb) {
            // Add LEFT JOIN for view counts without affecting existing joins
            if (strpos($join, 'mt_views') === false) {
                $join .= " LEFT JOIN {$wpdb->postmeta} mt_views ON ({$wpdb->posts}.ID = mt_views.post_id AND mt_views.meta_key = '_speaker_views') ";
            }
            return $join;
        }, 10, 1);
    }
}
add_action('pre_get_posts', 'sb_handle_custom_speaker_sorting');

/**
 * Migrate old coordinate field names to new format
 * Run once to convert _sb_lat/_sb_lng to geo_lat/geo_lng
 */
function sb_migrate_coordinate_field_names() {
    if (get_option('sb_coordinates_migrated')) {
        return; // Already migrated
    }

    global $wpdb;

    // Migrate _sb_lat to geo_lat
    $wpdb->query($wpdb->prepare("
        UPDATE {$wpdb->postmeta}
        SET meta_key = 'geo_lat'
        WHERE meta_key = '_sb_lat'
        AND post_id IN (SELECT ID FROM {$wpdb->posts} WHERE post_type = 'speaker')
        AND NOT EXISTS (
            SELECT 1 FROM {$wpdb->postmeta} pm2
            WHERE pm2.post_id = {$wpdb->postmeta}.post_id
            AND pm2.meta_key = 'geo_lat'
        )
    "));

    // Migrate _sb_lng to geo_lng
    $wpdb->query($wpdb->prepare("
        UPDATE {$wpdb->postmeta}
        SET meta_key = 'geo_lng'
        WHERE meta_key = '_sb_lng'
        AND post_id IN (SELECT ID FROM {$wpdb->posts} WHERE post_type = 'speaker')
        AND NOT EXISTS (
            SELECT 1 FROM {$wpdb->postmeta} pm2
            WHERE pm2.post_id = {$wpdb->postmeta}.post_id
            AND pm2.meta_key = 'geo_lng'
        )
    "));

    // Mark migration as complete
    update_option('sb_coordinates_migrated', true);
}
add_action('init', 'sb_migrate_coordinate_field_names');

/**
 * Local development environment specific fixes for PHP warnings
 */
function sb_local_environment_fixes() {
    // Check if we're on a local development environment
    $is_local = (
        isset($_SERVER['HTTP_HOST']) && (
            strpos($_SERVER['HTTP_HOST'], '.local') !== false ||
            strpos($_SERVER['HTTP_HOST'], 'localhost') !== false ||
            strpos($_SERVER['HTTP_HOST'], '127.0.0.1') !== false ||
            strpos($_SERVER['HTTP_HOST'], '.test') !== false ||
            strpos($_SERVER['HTTP_HOST'], '.dev') !== false
        )
    );

    if ($is_local) {
        // More aggressive error suppression for local development - apply globally
        set_error_handler(function($errno, $errstr, $errfile, $errline) {
            // Suppress str_starts_with warnings from WordPress core
            if ($errno === E_DEPRECATED &&
                strpos($errstr, 'str_starts_with()') !== false &&
                strpos($errstr, 'Passing null to parameter') !== false &&
                strpos($errfile, 'wp-includes') !== false) {
                return true; // Suppress this specific warning
            }

            // Also suppress other common local development warnings
            if ($errno === E_DEPRECATED &&
                (strpos($errstr, 'block-template.php') !== false ||
                 strpos($errstr, 'WP_Theme_JSON') !== false)) {
                return true; // Suppress block template related warnings
            }

            return false; // Let other errors through
        }, E_DEPRECATED | E_WARNING);
    }
}
add_action('init', 'sb_local_environment_fixes', 1);

/**
 * Ensure permalink structure is properly flushed for speakers
 */
function sb_ensure_speaker_permalinks() {
    // Check if we need to flush rewrite rules
    if (get_option('sb_flush_rewrite_rules')) {
        flush_rewrite_rules();
        delete_option('sb_flush_rewrite_rules');
    }

    // Also check if speaker post type is registered and flush if needed
    if (!post_type_exists('speaker')) {
        // If speaker post type doesn't exist, we might need to re-register it
        update_option('sb_flush_rewrite_rules', true);
    }
}
add_action('init', 'sb_ensure_speaker_permalinks', 999);

/**
 * Flag for permalink flushing when plugin is activated
 */
function sb_activation_flush_rewrites() {
    update_option('sb_flush_rewrite_rules', true);
}
register_activation_hook(__FILE__, 'sb_activation_flush_rewrites');

/**
 * Add Open Graph and Twitter Card meta tags for speaker profiles
 */
function sb_add_speaker_social_meta() {
    if (!is_singular('speaker')) {
        return;
    }

    global $post;
    if (!$post || !isset($post->ID)) {
        return;
    }

    $speaker_id = $post->ID;
    $speaker_title = get_the_title($speaker_id);
    $speaker_url = get_permalink($speaker_id);

    // Ensure we have valid title and URL
    if (empty($speaker_title) || empty($speaker_url)) {
        return;
    }

    // Get all form fields to extract bio and topics
    $fields = function_exists('sb_get_fields_config') ? sb_get_fields_config() : [];

    // Ensure fields is an array
    if (!is_array($fields)) {
        $fields = [];
    }

    $bio_text = '';
    $topics_text = '';
    $profile_image = '';

    // Extract bio, topics, and profile image from speaker fields
    foreach ($fields as $field) {
        // Ensure field is an array and has required properties
        if (!is_array($field) || empty($field['enabled']) || empty($field['key'])) {
            continue;
        }

        $key = $field['key'] ?? '';
        $type = $field['type'] ?? 'text';
        $label = isset($field['label']) ? strtolower((string)$field['label']) : '';
        $value = get_post_meta($speaker_id, $key, true);

        if (empty($key) || empty($value)) continue;

        // Look for bio/description fields
        if ($type === 'textarea' && !empty($label) && (
            strpos($label, 'bio') !== false ||
            strpos($label, 'description') !== false ||
            strpos($label, 'about') !== false ||
            strpos($label, 'summary') !== false
        )) {
            $bio_text = (string)$value;
        }

        // Look for topics/specialties fields
        if (!empty($label) && (
            strpos($label, 'topic') !== false ||
            strpos($label, 'specialt') !== false ||
            strpos($label, 'expertise') !== false ||
            strpos($label, 'subject') !== false
        )) {
            if (is_array($value)) {
                $topics_text = implode(', ', array_filter($value));
            } else {
                $topics_text = (string)$value;
            }
        }

        // Look for profile image
        if ($type === 'image' && !empty($label) && (
            strpos($label, 'photo') !== false ||
            strpos($label, 'image') !== false ||
            strpos($label, 'picture') !== false ||
            strpos($label, 'headshot') !== false
        )) {
            if (is_numeric($value)) {
                $profile_image = wp_get_attachment_image_url($value, 'large');
            } elseif (filter_var($value, FILTER_VALIDATE_URL)) {
                $profile_image = esc_url($value);
            }
        }
    }

    // Fallback to WordPress featured image if no profile image found
    if (empty($profile_image) && has_post_thumbnail($speaker_id)) {
        $profile_image = get_the_post_thumbnail_url($speaker_id, 'large');
    }

    // Create description from bio and topics
    $description_parts = [];
    if (!empty($bio_text)) {
        // Strip HTML and limit to 160 characters for social media
        $bio_clean = wp_strip_all_tags((string)$bio_text);
        $bio_clean = preg_replace('/\s+/', ' ', $bio_clean); // Normalize whitespace
        $bio_clean = trim($bio_clean);
        if (!empty($bio_clean)) {
            $description_parts[] = wp_trim_words($bio_clean, 25, '...');
        }
    }
    if (!empty($topics_text)) {
        $topics_clean = wp_strip_all_tags((string)$topics_text);
        $topics_clean = trim($topics_clean);
        if (!empty($topics_clean)) {
            $description_parts[] = 'Topics: ' . $topics_clean;
        }
    }

    $description = implode(' | ', $description_parts);
    if (empty($description)) {
        $description = 'Speaker profile for ' . get_the_title($speaker_id);
    }

    // Ensure description doesn't exceed 160 characters (optimal for social media)
    $description = (string)$description;
    if (strlen($description) > 160) {
        $description = substr($description, 0, 157) . '...';
    }

    // Get site name
    $site_name = get_bloginfo('name');

    // Output Open Graph meta tags
    echo '<meta property="og:type" content="profile" />' . "\n";
    echo '<meta property="og:title" content="' . esc_attr($speaker_title) . '" />' . "\n";
    echo '<meta property="og:description" content="' . esc_attr($description) . '" />' . "\n";
    echo '<meta property="og:url" content="' . esc_url($speaker_url) . '" />' . "\n";
    echo '<meta property="og:site_name" content="' . esc_attr($site_name) . '" />' . "\n";

    if (!empty($profile_image)) {
        echo '<meta property="og:image" content="' . esc_url($profile_image) . '" />' . "\n";
        echo '<meta property="og:image:width" content="1200" />' . "\n";
        echo '<meta property="og:image:height" content="630" />' . "\n";
        echo '<meta property="og:image:alt" content="Profile photo of ' . esc_attr($speaker_title) . '" />' . "\n";
    }

    // Output Twitter Card meta tags
    echo '<meta name="twitter:card" content="summary_large_image" />' . "\n";
    echo '<meta name="twitter:title" content="' . esc_attr($speaker_title) . '" />' . "\n";
    echo '<meta name="twitter:description" content="' . esc_attr($description) . '" />' . "\n";

    if (!empty($profile_image)) {
        echo '<meta name="twitter:image" content="' . esc_url($profile_image) . '" />' . "\n";
        echo '<meta name="twitter:image:alt" content="Profile photo of ' . esc_attr($speaker_title) . '" />' . "\n";
    }

    // Additional meta tags for better SEO
    echo '<meta name="description" content="' . esc_attr($description) . '" />' . "\n";

    // Schema.org structured data for person
    echo '<script type="application/ld+json">' . "\n";
    echo '{' . "\n";
    echo '  "@context": "https://schema.org",' . "\n";
    echo '  "@type": "Person",' . "\n";
    echo '  "name": "' . esc_js($speaker_title) . '",' . "\n";
    echo '  "url": "' . esc_url($speaker_url) . '",' . "\n";
    if (!empty($description)) {
        echo '  "description": "' . esc_js($description) . '",' . "\n";
    }
    if (!empty($profile_image)) {
        echo '  "image": "' . esc_url($profile_image) . '",' . "\n";
    }
    if (!empty($topics_text)) {
        echo '  "knowsAbout": "' . esc_js($topics_text) . '",' . "\n";
    }
    echo '  "jobTitle": "Speaker"' . "\n";
    echo '}' . "\n";
    echo '</script>' . "\n";
}
add_action('wp_head', 'sb_add_speaker_social_meta');

/**
 * AJAX handler for updating speaker view counts (admin only)
 */
function sb_update_speaker_views_ajax() {
    // Security checks
    if (!current_user_can('edit_posts')) {
        wp_die('Insufficient permissions');
    }

    if (!wp_verify_nonce($_POST['nonce'], 'sb_update_views')) {
        wp_die('Security check failed');
    }

    $post_id = intval($_POST['post_id']);
    $views = intval($_POST['views']);

    // Validate post exists and is a speaker
    $post = get_post($post_id);
    if (!$post || $post->post_type !== 'speaker') {
        wp_send_json_error('Invalid speaker post');
        return;
    }

    // Update the view count
    update_post_meta($post_id, '_speaker_views', $views);

    // Log the admin change
    $user = wp_get_current_user();
    error_log("Speakers Bureau: Admin {$user->user_login} updated speaker {$post->post_title} (ID: {$post_id}) view count to {$views}");

    wp_send_json_success([
        'post_id' => $post_id,
        'views' => $views,
        'formatted_views' => number_format($views)
    ]);
}
add_action('wp_ajax_sb_update_speaker_views', 'sb_update_speaker_views_ajax');

/** END OF FILE: speakers-bureau.php */
