<?php
/**
 * File: templates/speaker-verify.php
 * Purpose: Email verification and password setup page for speaker registration
 */

if (!defined('ABSPATH')) exit;

// Get verification system instance
$verification = new SB_Registration_Verification();

$token = isset($_GET['token']) ? sanitize_text_field($_GET['token']) : '';
$notice = '';
$error = '';
$step = 'verify'; // 'verify' or 'complete'

// Handle form submission
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    if (isset($_POST['sb_verify_nonce']) && wp_verify_nonce($_POST['sb_verify_nonce'], 'sb_verify_registration')) {
        $password = isset($_POST['user_password']) ? $_POST['user_password'] : '';
        $password_confirm = isset($_POST['user_password_confirm']) ? $_POST['user_password_confirm'] : '';

        if (strlen($password) < 8) {
            $error = __('Password must be at least 8 characters long.', 'speakers-bureau');
        } elseif ($password !== $password_confirm) {
            $error = __('Passwords do not match.', 'speakers-bureau');
        } else {
            $result = $verification->complete_registration($token, $password);
            if (is_wp_error($result)) {
                $error = $result->get_error_message();
            } else {
                // Registration completed successfully and user is now logged in
                // Redirect to profile completion page
                $redirect_url = isset($result['redirect_url']) ? $result['redirect_url'] : home_url('/my-profile/');
                wp_redirect($redirect_url . '?welcome=1');
                exit;
            }
        }
    }
}

// Verify token
$pending_data = false;
if ($token && $step === 'verify') {
    $pending_data = $verification->get_pending_registration($token);
    if (!$pending_data) {
        $error = __('Invalid or expired verification link. Please request a new registration.', 'speakers-bureau');
    } else {
        $step = 'password_setup';
    }
}

get_header();
?>

<div class="sb-wrap sb-verify-page" style="max-width: 600px; margin: 40px auto; padding: 20px;">
    <div class="sb-verify-container" style="background: #fff; padding: 40px; border-radius: 8px; box-shadow: 0 2px 10px rgba(0,0,0,0.1);">

        <?php if ($step === 'verify'): ?>
            <h1 style="text-align: center; color: #0073aa; margin-bottom: 30px;">
                <?php esc_html_e('Email Verification', 'speakers-bureau'); ?>
            </h1>

            <?php if ($error): ?>
                <div class="notice notice-error" style="margin: 20px 0; padding: 12px; border-left: 4px solid #dc3232;">
                    <p><?php echo esc_html($error); ?></p>
                </div>
                <div style="text-align: center; margin-top: 30px;">
                    <a href="<?php echo esc_url(home_url('/speaker-register/')); ?>" class="button button-primary">
                        <?php esc_html_e('Start New Registration', 'speakers-bureau'); ?>
                    </a>
                </div>
            <?php else: ?>
                <div style="text-align: center;">
                    <div style="font-size: 48px; margin-bottom: 20px;">⏳</div>
                    <p style="font-size: 18px; color: #666;">
                        <?php esc_html_e('Verifying your email address...', 'speakers-bureau'); ?>
                    </p>
                    <p>
                        <?php esc_html_e('If this page does not redirect automatically, please check your verification link.', 'speakers-bureau'); ?>
                    </p>
                </div>
            <?php endif; ?>

        <?php elseif ($step === 'password_setup'): ?>
            <h1 style="text-align: center; color: #0073aa; margin-bottom: 30px;">
                <?php esc_html_e('Complete Your Registration', 'speakers-bureau'); ?>
            </h1>

            <?php if ($error): ?>
                <div class="notice notice-error" style="margin: 20px 0; padding: 12px; border-left: 4px solid #dc3232;">
                    <p><?php echo esc_html($error); ?></p>
                </div>
            <?php endif; ?>

            <div style="margin-bottom: 30px; padding: 20px; background: #f8f9fa; border-radius: 4px; border-left: 4px solid #00a32a;">
                <h3 style="margin-top: 0; color: #00a32a;">
                    <?php esc_html_e('Email Verified Successfully!', 'speakers-bureau'); ?>
                </h3>
                <p style="margin-bottom: 0;">
                    <?php printf(__('Welcome, %s! Please set a password to complete your registration.', 'speakers-bureau'),
                        '<strong>' . esc_html($pending_data['name']) . '</strong>'); ?>
                </p>
                <p style="margin-bottom: 0; font-size: 14px; color: #666;">
                    <?php echo esc_html($pending_data['email']); ?>
                </p>
            </div>

            <form method="post" action="">
                <?php wp_nonce_field('sb_verify_registration', 'sb_verify_nonce'); ?>

                <div style="margin-bottom: 20px;">
                    <label for="user_password" style="display: block; margin-bottom: 8px; font-weight: 600;">
                        <?php esc_html_e('Password', 'speakers-bureau'); ?>
                    </label>
                    <input type="password" id="user_password" name="user_password"
                           style="width: 100%; padding: 12px; border: 1px solid #ddd; border-radius: 4px; font-size: 16px;"
                           required minlength="8"
                           placeholder="<?php esc_attr_e('Enter a secure password (min 8 characters)', 'speakers-bureau'); ?>">
                    <p style="margin: 8px 0 0; font-size: 14px; color: #666;">
                        <?php esc_html_e('Minimum 8 characters required', 'speakers-bureau'); ?>
                    </p>
                </div>

                <div style="margin-bottom: 30px;">
                    <label for="user_password_confirm" style="display: block; margin-bottom: 8px; font-weight: 600;">
                        <?php esc_html_e('Confirm Password', 'speakers-bureau'); ?>
                    </label>
                    <input type="password" id="user_password_confirm" name="user_password_confirm"
                           style="width: 100%; padding: 12px; border: 1px solid #ddd; border-radius: 4px; font-size: 16px;"
                           required minlength="8"
                           placeholder="<?php esc_attr_e('Confirm your password', 'speakers-bureau'); ?>">
                </div>

                <div style="text-align: center;">
                    <button type="submit" class="button button-primary" style="padding: 12px 30px; font-size: 16px;">
                        <?php esc_html_e('Complete Registration', 'speakers-bureau'); ?>
                    </button>
                </div>
            </form>

            <div style="margin-top: 30px; padding: 15px; background: #fff3cd; border: 1px solid #ffeaa7; border-radius: 4px;">
                <h4 style="margin-top: 0; color: #856404;">
                    <?php esc_html_e('What happens next?', 'speakers-bureau'); ?>
                </h4>
                <ul style="margin-bottom: 0; padding-left: 20px;">
                    <li><?php esc_html_e('Your account will be created and you\'ll be automatically logged in', 'speakers-bureau'); ?></li>
                    <li><?php esc_html_e('You\'ll be redirected to complete your speaker profile', 'speakers-bureau'); ?></li>
                    <li><?php esc_html_e('Your profile will be created as a draft for admin review', 'speakers-bureau'); ?></li>
                    <li><?php esc_html_e('You can edit your profile anytime after completion', 'speakers-bureau'); ?></li>
                </ul>
            </div>

        <?php elseif ($step === 'completed'): ?>
            <h1 style="text-align: center; color: #00a32a; margin-bottom: 30px;">
                <?php esc_html_e('Registration Complete!', 'speakers-bureau'); ?>
            </h1>

            <div style="text-align: center; margin-bottom: 30px;">
                <div style="font-size: 64px; color: #00a32a; margin-bottom: 20px;">✅</div>

                <?php if ($notice): ?>
                    <div class="notice notice-success" style="margin: 20px 0; padding: 12px; border-left: 4px solid #00a32a;">
                        <p><?php echo esc_html($notice); ?></p>
                    </div>
                <?php endif; ?>
            </div>

            <div style="text-align: center; margin-top: 30px;">
                <?php
                $login_page = get_page_by_path('account-login');
                $login_url = $login_page ? get_permalink($login_page) : wp_login_url();
                ?>
                <a href="<?php echo esc_url($login_url); ?>" class="button button-primary" style="margin-right: 15px;">
                    <?php esc_html_e('Log In', 'speakers-bureau'); ?>
                </a>
                <a href="<?php echo esc_url(home_url()); ?>" class="button button-secondary">
                    <?php esc_html_e('Back to Home', 'speakers-bureau'); ?>
                </a>
            </div>

        <?php endif; ?>

    </div>
</div>

<script>
document.addEventListener('DOMContentLoaded', function() {
    // Auto-verify token on page load if step is verify
    <?php if ($step === 'verify' && $token && !$error): ?>
        // Automatically proceed to password setup if token is valid
        window.location.reload();
    <?php endif; ?>

    // Password confirmation validation
    var password = document.getElementById('user_password');
    var confirm = document.getElementById('user_password_confirm');

    if (password && confirm) {
        function validatePasswords() {
            if (confirm.value && password.value !== confirm.value) {
                confirm.setCustomValidity('<?php echo esc_js(__('Passwords do not match', 'speakers-bureau')); ?>');
            } else {
                confirm.setCustomValidity('');
            }
        }

        password.addEventListener('input', validatePasswords);
        confirm.addEventListener('input', validatePasswords);
    }
});
</script>

<?php get_footer(); ?>
<?php
/** END OF FILE: templates/speaker-verify.php */