<?php
/**
 * Shortcode Template: [speaker_list]
 * Location: templates/speaker-list.php
 *
 * Features
 * - Standard keyword search across "searchable" fields from Form Builder (+ title)
 * - Optional geo search: ZIP, radius, country (controlled in Settings → Directory Options)
 * - Optional sort by "Nearest" (when geo search enabled and a ZIP is provided)
 * - Card grid with featured image fallback to "profile_image" meta
 * - Manual pagination when sorting by nearest
 */

if (!defined('ABSPATH')) exit;

global $wpdb;

/* ---------- Settings (Directory Options) ---------- */
$gen              = get_option('sb_settings', []);

// If settings are empty, apply defaults to restore functionality
if (empty($gen)) {
    $gen = [
        'enable_geo'           => 1,
        'geo_default_country'  => 'US',
        'geo_default_radius'   => 50,
        'enable_nearest_sort'  => 1,
    ];
}

$geo_enabled      = !empty($gen['enable_geo']);
$def_country      = !empty($gen['geo_default_country']) ? strtoupper($gen['geo_default_country']) : 'US';
$def_radius       = !empty($gen['geo_default_radius']) ? (int)$gen['geo_default_radius'] : 50;
$nearest_enabled  = !empty($gen['enable_nearest_sort']);



/* ---------- Helpers ---------- */
if (!function_exists('sb_list_image_from_meta_fallback')) {
    function sb_list_image_from_meta_fallback($post_id) {
        // Prefer featured image
        if (has_post_thumbnail($post_id)) {
            return get_the_post_thumbnail_url($post_id, 'medium_large');
        }

        // Fallback to "profile_image" key if present
        $raw = get_post_meta($post_id, 'profile_image', true);
        if ($raw) {
            // Expect "id|url" but gracefully handle plain url or id
            $id  = null; $url = null;
            if (is_string($raw) && strpos($raw, '|') !== false) {
                list($id, $url) = array_pad(explode('|', $raw, 2), 2, null);
            } elseif (is_numeric($raw)) {
                $id = (int)$raw;
            } elseif (filter_var($raw, FILTER_VALIDATE_URL)) {
                $url = $raw;
            }
            if ($id && ($img = wp_get_attachment_image_url((int)$id, 'medium_large'))) {
                return $img;
            }
            if ($url) return esc_url($url);
        }
        return ''; // no image
    }
}

if (!function_exists('sb_extract_words')) {
    function sb_extract_words($text, $limit = 28) {
        $text = wp_strip_all_tags((string)$text);
        $parts = preg_split('/\s+/', $text);
        if (count($parts) > $limit) {
            $parts = array_slice($parts, 0, $limit);
            return implode(' ', $parts) . '…';
        }
        return $text;
    }
}

if (!function_exists('sb_render_card_field')) {
    function sb_render_card_field($post_id, $field_key, $max_words) {
        $value = get_post_meta($post_id, $field_key, true);
        if (!$value) return '';

        if (is_array($value)) {
            $value = implode(', ', array_filter($value));
        }

        return sb_extract_words($value, $max_words);
    }
}

if (!function_exists('sb_get_searchable_meta_keys')) {
    function sb_get_searchable_meta_keys() {
        // Pull from Form Builder config; default to keys commonly used if empty
        $fields = function_exists('sb_normalize_form_fields')
            ? sb_normalize_form_fields()
            : get_option('sb_form_fields', []);

        $keys = [];
        if (is_array($fields)) {
            foreach ($fields as $f) {
                // "enabled" should be on; "searchable" checked or absent defaults to on
                $enabled    = !empty($f['enabled']);
                $key        = isset($f['key']) ? sanitize_key($f['key']) : '';
                $searchable = array_key_exists('searchable', $f) ? !empty($f['searchable']) : true;
                if ($enabled && $key && $searchable) {
                    $keys[] = $key;
                }
            }
        }
        // Always include post_title via 's' param, independent from meta keys
        return array_values(array_unique($keys));
    }
}

if (!function_exists('sb_geo_box_from_zip')) {
    function sb_geo_box_from_zip($zip, $country = 'US', $radius_miles = 50) {
        $zip = trim((string)$zip);
        $country = strtoupper(trim((string)$country));
        $radius = max(1, (int)$radius_miles);

        if ($zip === '') return [null, null];

        // Get (lat, lng) for the center — use cached option
        $cache = get_option('sb_geo_cache', []);
        if (!is_array($cache)) $cache = [];
        $key = $country . '|' . $zip;
        $lat = null; $lng = null;

        if (isset($cache[$key]['lat'], $cache[$key]['lng'])) {
            $lat = (float)$cache[$key]['lat'];
            $lng = (float)$cache[$key]['lng'];
        } else {
            // try to resolve with Nominatim (polite UA + small timeout)
            $url = add_query_arg([
                'format'       => 'json',
                'postalcode'   => $zip,
                'countrycodes' => strtolower($country),
                'limit'        => 1
            ], 'https://nominatim.openstreetmap.org/search');

            $resp = wp_remote_get($url, [
                'timeout' => 8,
                'headers' => ['User-Agent' => 'Speakers-Bureau-Plugin/2.0 (+wordpress)']
            ]);
            if (!is_wp_error($resp) && wp_remote_retrieve_response_code($resp) === 200) {
                $body = json_decode(wp_remote_retrieve_body($resp), true);
                if (is_array($body) && !empty($body[0]['lat']) && !empty($body[0]['lon'])) {
                    $lat = (float)$body[0]['lat'];
                    $lng = (float)$body[0]['lon'];
                    $cache[$key] = ['lat'=>$lat,'lng'=>$lng,'t'=>time()];
                    if (count($cache) > 5000) $cache = array_slice($cache, -4000, null, true);
                    update_option('sb_geo_cache', $cache, false);
                }
            }
        }
        if ($lat === null || $lng === null) return [null, null];

        // Bounding box in degrees
        $earth = 3959; // miles
        $dLat  = rad2deg($radius / $earth);
        $dLng  = rad2deg($radius / ($earth * cos(deg2rad($lat))));

        return [
            'lat_min' => $lat - $dLat,
            'lat_max' => $lat + $dLat,
            'lng_min' => $lng - $dLng,
            'lng_max' => $lng + $dLng,
            'latC'    => $lat,
            'lngC'    => $lng,
            'radius'  => $radius
        ];
    }
}

/* ---------- Inputs ---------- */
$q        = isset($_GET['q']) ? wp_unslash($_GET['q']) : '';
$q        = is_string($q) ? trim($q) : '';
// Get page number from URL - check both query param and pretty permalinks
$paged = 1;
if (isset($_GET['paged'])) {
    $paged = max(1, (int) $_GET['paged']);
} else {
    // Check for pretty permalink pagination (/page/N/)
    global $wp_query;
    if (isset($wp_query->query_vars['paged']) && $wp_query->query_vars['paged'] > 0) {
        $paged = max(1, (int) $wp_query->query_vars['paged']);
    }
}
$sort     = isset($_GET['sort']) ? sanitize_key($_GET['sort']) : '';
$country  = isset($_GET['country']) ? strtoupper(sanitize_text_field(wp_unslash($_GET['country']))) : $def_country;
$zip      = isset($_GET['zip']) ? sanitize_text_field(wp_unslash($_GET['zip'])) : '';
$radius   = isset($_GET['radius']) ? (int) $_GET['radius'] : $def_radius;

// Handle zip code cookie for "nearest" sorting
if ($sort === 'nearest') {
    // If zip is provided, save it to cookie
    if (!empty($zip)) {
        setcookie('sb_nearest_zip', $zip, time() + (86400 * 30), '/'); // 30 day cookie
    }
    // If no zip provided but sort is nearest, try to get from cookie
    elseif (empty($zip) && isset($_COOKIE['sb_nearest_zip'])) {
        $zip = sanitize_text_field($_COOKIE['sb_nearest_zip']);
    }
}

$searchable_keys = sb_get_searchable_meta_keys();


/* ---------- Get Display Settings ---------- */
$display_settings = get_option('sb_display_settings', [
    'columns' => 3,
    'per_page' => 12,
    'card_fields' => ['bio', 'topics'],
    'max_words' => 25,
    'card_style' => 'default',
    'image_style' => 'square',
    'show_excerpt' => true,
    'excerpt_length' => 20,
    'primary_color' => '#0073aa',
    'card_border_radius' => '8',
    'card_shadow' => 'light',
    'show_date_created' => false,
    'show_date_updated' => false,
    'date_format' => 'm/d/Y g:ia',
    'auto_publish_profiles' => false
]);

// Ensure all defaults are set (for newly added settings)
$defaults = [
    'columns' => 3,
    'per_page' => 12,
    'card_fields' => ['bio', 'topics'],
    'max_words' => 25,
    'card_style' => 'default',
    'image_style' => 'square',
    'show_excerpt' => true,
    'excerpt_length' => 20,
    'primary_color' => '#0073aa',
    'card_border_radius' => '8',
    'card_shadow' => 'light',
    'show_date_created' => false,
    'show_date_updated' => false,
    'date_format' => 'm/d/Y g:ia',
    'auto_publish_profiles' => false
];
$display_settings = array_merge($defaults, $display_settings);

/* ---------- Build Query Args ---------- */
$per_page = $display_settings['per_page'];

$args = [
    'post_type'      => 'speaker',
    'post_status'    => 'publish',
    'paged'          => $paged,
    'posts_per_page' => $per_page,
];

$meta_query = ['relation' => 'AND'];

// Handle different sorting options
switch ($sort) {
    case 'date':
    case 'newest':
        // Newest first - sort by publication date
        $args['orderby'] = 'date';
        $args['order'] = 'DESC';
        break;

    case 'popular':
        // Sort by view count (most popular first)
        // We'll handle sorting manually after the query to ensure all speakers are included
        $args['orderby'] = 'date';
        $args['order'] = 'DESC';
        // Set a flag so we know to sort manually later
        $sort_by_views_manually = true;
        break;

    case '':
    case 'default':
    default:
        // Default sort: newest posts first (simple and reliable)
        $args['orderby'] = 'date';
        $args['order'] = 'DESC';
        break;
}

/* Keyword across meta keys (OR) */
if ($q !== '') {
    $or = ['relation' => 'OR'];
    foreach ($searchable_keys as $key) {
        $or[] = [
            'key'     => $key,
            'value'   => $q,
            'compare' => 'LIKE'
        ];
    }

    // Add the meta query OR conditions if we have any
    if (count($or) > 1) {
        $meta_query[] = $or;
    }

    // Use a more controlled approach for title search
    add_filter('posts_where', function($where, $query) use ($q) {
        global $wpdb;

        // Only apply to our speaker queries
        if ($query->get('post_type') === 'speaker' && !empty($q)) {
            $search_term = '%' . $wpdb->esc_like($q) . '%';
            // Add title search as an additional OR condition within the existing WHERE clause
            $where = str_replace(
                'AND ((',
                'AND ((' . $wpdb->prepare("({$wpdb->posts}.post_title LIKE %s) OR ", $search_term),
                $where
            );
            // If no meta query, add title search directly
            if (strpos($where, 'AND ((') === false) {
                $where .= $wpdb->prepare(" AND ({$wpdb->posts}.post_title LIKE %s)", $search_term);
            }
        }

        return $where;
    }, 10, 2);
}

/* Geo filter (if enabled and user supplied a ZIP) */
$geoBox = null;
if ($geo_enabled && $zip !== '') {
    $geoBox = sb_geo_box_from_zip($zip, $country ?: $def_country, $radius ?: $def_radius);
    if (is_array($geoBox) && isset($geoBox['lat_min'])) {
        // Reduce result set using meta_query bounding box
        $meta_query[] = [
            'key'   => 'geo_lat',
            'value' => [$geoBox['lat_min'], $geoBox['lat_max']],
            'type'  => 'DECIMAL(10,6)',
            'compare' => 'BETWEEN'
        ];
        $meta_query[] = [
            'key'   => 'geo_lng',
            'value' => [$geoBox['lng_min'], $geoBox['lng_max']],
            'type'  => 'DECIMAL(10,6)',
            'compare' => 'BETWEEN'
        ];
    }
}

if (count($meta_query) > 1) {
    $args['meta_query'] = $meta_query;
}

/* Sorting */
$use_nearest = ($sort === 'nearest' && $nearest_enabled && $geoBox &&
               !empty($geoBox['lat']) && !empty($geoBox['lng']) &&
               is_numeric($geoBox['lat']) && is_numeric($geoBox['lng']));

// Handle case where nearest sort is requested but no zip code is available
$nearest_no_zip = ($sort === 'nearest' && $nearest_enabled && empty($zip));
if ($nearest_no_zip) {
    // Fall back to default sorting when no zip code is provided
    $sort = '';
}
if ($use_nearest) {
    // Optimize nearest sort by limiting to reasonable distance and count
    $max_distance_miles = 500; // Reasonable max distance
    $max_nearest_results = 500; // Prevent excessive memory usage

    // Calculate rough coordinate bounds (approximate)
    $lat_delta = $max_distance_miles / 69; // 1 degree lat ≈ 69 miles
    $lng_delta = $max_distance_miles / (69 * cos(deg2rad(floatval($geoBox['lat']))));

    // Add geographic constraints to improve query performance
    $user_lat = floatval($geoBox['lat']);
    $user_lng = floatval($geoBox['lng']);

    $geo_meta_query = [
        'key' => 'geo_lat',
        'value' => [$user_lat - $lat_delta, $user_lat + $lat_delta],
        'type' => 'DECIMAL(10,7)',
        'compare' => 'BETWEEN'
    ];

    $geo_lng_query = [
        'key' => 'geo_lng',
        'value' => [$user_lng - $lng_delta, $user_lng + $lng_delta],
        'type' => 'DECIMAL(10,7)',
        'compare' => 'BETWEEN'
    ];

    // Add to existing meta_query or create new one
    if (!isset($args['meta_query'])) {
        $args['meta_query'] = ['relation' => 'AND'];
    }

    $args['meta_query'][] = $geo_meta_query;
    $args['meta_query'][] = $geo_lng_query;

    // Limit results for performance
    $args['posts_per_page'] = $max_nearest_results;
    unset($args['paged']);
}

/* ---------- Run Query ---------- */
$query = new WP_Query($args);

// Manual sorting for popular view counts
if (isset($sort_by_views_manually) && $sort_by_views_manually) {
    // Sort all posts by view count manually
    $posts_with_views = [];
    foreach ($query->posts as $post) {
        $views = get_post_meta($post->ID, '_speaker_views', true);
        $views = $views ? intval($views) : 0;
        $posts_with_views[] = [
            'post' => $post,
            'views' => $views,
            'date' => $post->post_date
        ];
    }

    // Sort by views (desc), then by date (desc)
    usort($posts_with_views, function($a, $b) {
        if ($a['views'] === $b['views']) {
            return strcmp($b['date'], $a['date']); // Newer first
        }
        return $b['views'] - $a['views']; // Higher views first
    });

    // Extract the sorted posts
    $query->posts = array_map(function($item) {
        return $item['post'];
    }, $posts_with_views);
}

// Filter out any non-speaker posts that might have snuck through
$query->posts = array_filter($query->posts, function($post) {
    return $post->post_type === 'speaker';
});



/* ---------- Manual nearest sort & pagination ---------- */
$max_pages = $query->max_num_pages;
$page_posts = $query->posts;

if ($use_nearest && is_array($page_posts)) {
    // Compute distances
    foreach ($page_posts as &$p) {
        $pid = is_object($p) ? $p->ID : (int)$p;
        $lat = get_post_meta($pid, 'geo_lat', true);
        $lng = get_post_meta($pid, 'geo_lng', true);
        if ($lat !== '' && $lng !== '' && is_numeric($lat) && is_numeric($lng)) {
            $dlat = deg2rad((float)$lat - (float)$geoBox['latC']);
            $dlng = deg2rad((float)$lng - (float)$geoBox['lngC']);
            $a = sin($dlat/2)**2 + cos(deg2rad($geoBox['latC']))*cos(deg2rad((float)$lat))*sin($dlng/2)**2;
            $dist = 3959 * 2 * atan2(sqrt($a), sqrt(1-$a)); // miles
            $p->_sb_distance = $dist;
        } else {
            $p->_sb_distance = PHP_INT_MAX;
        }
    }
    unset($p);

    usort($page_posts, function($a,$b){
        return ($a->_sb_distance <=> $b->_sb_distance);
    });

    // Filter out those beyond radius (if center present)
    $page_posts = array_values(array_filter($page_posts, function($p) use ($geoBox){
        return isset($p->_sb_distance) && $p->_sb_distance <= $geoBox['radius'] + 0.001;
    }));

    // Manual pagination
    $total     = count($page_posts);
    $per_page  = 12;
    $max_pages = max(1, (int)ceil($total / $per_page));
    $paged     = max(1, $paged);
    $offset    = ($paged - 1) * $per_page;
    $page_posts = array_slice($page_posts, $offset, $per_page);
}

/* ---------- UI: Search form ---------- */
$action = esc_url(get_permalink());
?>
<div class="sb-wrap">

  <form method="get" action="<?php echo $action; ?>" class="sb-search-form" style="margin:10px 0 16px 0;">
      <input type="text" name="q" value="<?php echo esc_attr($q); ?>" placeholder="<?php esc_attr_e('Search speakers…', 'speakers-bureau'); ?>" />
      <button type="submit"><?php esc_html_e('Search', 'speakers-bureau'); ?></button>

      <?php if ($geo_enabled): ?>
        <button type="button" id="sb-adv-toggle" aria-expanded="<?php echo $zip !== '' ? 'true':'false'; ?>" style="margin-left:8px;">
            <?php echo $zip !== '' ? esc_html__('Hide location', 'speakers-bureau') : esc_html__('Location…', 'speakers-bureau'); ?>
        </button>
      <?php endif; ?>

      <select name="sort" style="margin-left:8px;">
        <option value="date" <?php selected($sort,'date'); ?>><?php esc_html_e('Newest', 'speakers-bureau'); ?></option>
        <option value="popular" <?php selected($sort,'popular'); ?>><?php esc_html_e('Most Popular', 'speakers-bureau'); ?></option>
        <?php if ($nearest_enabled): ?>
            <option value="nearest" <?php selected($sort,'nearest'); ?>><?php esc_html_e('Nearest', 'speakers-bureau'); ?></option>
        <?php endif; ?>
        <option value="" <?php selected($sort,''); ?>><?php esc_html_e('Default', 'speakers-bureau'); ?></option>
      </select>
  </form>

  <?php if ($geo_enabled): ?>
    <form method="get" action="<?php echo $action; ?>" id="sb-adv" class="sb-advanced" style="display:<?php echo $zip !== '' ? 'block':'none'; ?>; margin:8px 0 18px 0;">
        <input type="hidden" name="q" value="<?php echo esc_attr($q); ?>">
        <div class="sb-adv-row" style="display:flex;gap:10px;align-items:flex-end;flex-wrap:wrap;">
            <div>
                <label><?php esc_html_e('ZIP', 'speakers-bureau'); ?></label><br>
                <input type="text" name="zip" value="<?php echo esc_attr($zip); ?>" style="max-width:120px;">
            </div>
            <div>
                <label><?php esc_html_e('Country', 'speakers-bureau'); ?></label><br>
                <input type="text" name="country" value="<?php echo esc_attr($country ?: $def_country); ?>" style="max-width:90px;">
            </div>
            <div>
                <label><?php esc_html_e('Radius (miles)', 'speakers-bureau'); ?></label><br>
                <input type="number" min="1" step="1" name="radius" value="<?php echo esc_attr($radius ?: $def_radius); ?>" style="max-width:90px;">
            </div>
            <div>
                <label><?php esc_html_e('Sort', 'speakers-bureau'); ?></label><br>
                <select name="sort">
                    <option value="date" <?php selected($sort,'date'); ?>><?php esc_html_e('Newest', 'speakers-bureau'); ?></option>
                    <option value="popular" <?php selected($sort,'popular'); ?>><?php esc_html_e('Most Popular', 'speakers-bureau'); ?></option>
                    <?php if ($nearest_enabled): ?>
                        <option value="nearest" <?php selected($sort,'nearest'); ?>><?php esc_html_e('Nearest', 'speakers-bureau'); ?></option>
                    <?php endif; ?>
                    <option value="" <?php selected($sort,''); ?>><?php esc_html_e('Default', 'speakers-bureau'); ?></option>
                </select>
            </div>
            <div>
                <button type="submit" class="button button-primary" style="margin-top:2px;"><?php esc_html_e('Apply', 'speakers-bureau'); ?></button>
            </div>
        </div>
    </form>
  <?php endif; ?>

  <?php if ($q !== ''): ?>
      <p style="margin-bottom:14px;"><?php
          printf(
              /* translators: %s = search term */
              esc_html__('Showing results for "%s".', 'speakers-bureau'),
              esc_html($q)
          );
      ?></p>
  <?php endif; ?>

  <?php if ($nearest_no_zip): ?>
      <div style="margin-bottom:16px; padding:12px; background:#fff3cd; border:1px solid #ffeaa7; border-radius:4px; color:#856404;">
          <strong><?php esc_html_e('Location Required:', 'speakers-bureau'); ?></strong>
          <?php esc_html_e('To sort by nearest distance, please click "Location…" above and enter your ZIP code.', 'speakers-bureau'); ?>
          <?php esc_html_e('Showing default results instead.', 'speakers-bureau'); ?>
      </div>
  <?php endif; ?>

  <div class="sb-directory-grid sb-cols-<?php echo esc_attr($display_settings['columns']); ?> sb-style-<?php echo esc_attr($display_settings['card_style']); ?> sb-shadow-<?php echo esc_attr($display_settings['card_shadow']); ?>">
    <?php
    $loop_posts = $use_nearest ? $page_posts : $query->posts;

    if (!empty($loop_posts)) :
        foreach ($loop_posts as $post) :
            $pid      = is_object($post) ? $post->ID : (int)$post;
            $title    = get_the_title($pid);
            $permalink= get_permalink($pid);
            $img      = sb_list_image_from_meta_fallback($pid);

            // Build card fields based on display settings
            $card_fields_html = '';
            foreach ($display_settings['card_fields'] as $field_key) {
                $field_value = sb_render_card_field($pid, $field_key, $display_settings['max_words']);
                if ($field_value) {
                    $card_fields_html .= '<div class="sb-card-field sb-field-' . esc_attr($field_key) . '">' . esc_html($field_value) . '</div>';
                }
            }

            // Bio excerpt if enabled AND bio is in card_fields
            $excerpt_html = '';
            if ($display_settings['show_excerpt'] && in_array('bio', $display_settings['card_fields'])) {
                $bio = get_post_meta($pid, 'bio', true);
                if (!$bio) $bio = get_post_field('post_excerpt', $pid);
                if (!$bio) $bio = get_post_field('post_content', $pid);
                if ($bio) {
                    $excerpt_html = '<div class="sb-card-excerpt">' . esc_html(sb_extract_words($bio, $display_settings['excerpt_length'])) . '</div>';
                }
            }
            ?>
            <article class="sb-card sb-img-<?php echo esc_attr($display_settings['image_style']); ?>">
                <a href="<?php echo esc_url($permalink); ?>" aria-label="<?php echo esc_attr($title); ?>">
                    <?php if ($img): ?>
                        <img src="<?php echo esc_url($img); ?>" alt="" class="sb-card-image">
                    <?php endif; ?>
                </a>
                <div class="sb-card-content">
                    <h3 class="sb-card-title"><a href="<?php echo esc_url($permalink); ?>"><?php echo esc_html($title); ?></a></h3>
                    <?php echo $card_fields_html; ?>
                    <?php echo $excerpt_html; ?>
                    <?php if ($use_nearest && isset($post->_sb_distance) && $post->_sb_distance !== PHP_INT_MAX): ?>
                        <div class="sb-card-distance"><?php printf(esc_html__('~%.1f miles', 'speakers-bureau'), $post->_sb_distance); ?></div>
                    <?php endif; ?>
                    <?php
                    // Show date if enabled in display settings
                    if (!empty($display_settings['show_date_created']) || !empty($display_settings['show_date_updated'])) {
                        $show_updated = !empty($display_settings['show_date_updated']);
                        $show_created = !empty($display_settings['show_date_created']);
                        $date_format = $display_settings['date_format'] ?? 'm/d/Y g:ia';

                        $date_to_show = '';
                        $date_label = '';

                        if ($show_updated) {
                            $modified_date = get_the_modified_date('U', $pid);
                            $created_date = get_the_date('U', $pid);
                            // Only show "updated" if it's actually different from created date
                            if ($modified_date && $modified_date != $created_date) {
                                $date_to_show = date($date_format, $modified_date);
                                $date_label = __('updated', 'speakers-bureau');
                            } elseif ($show_created && $created_date) {
                                $date_to_show = date($date_format, $created_date);
                                $date_label = __('created', 'speakers-bureau');
                            }
                        } elseif ($show_created) {
                            $created_date = get_the_date('U', $pid);
                            if ($created_date) {
                                $date_to_show = date($date_format, $created_date);
                                $date_label = __('created', 'speakers-bureau');
                            }
                        }

                        if ($date_to_show && $date_label) {
                            echo '<div class="sb-card-date">' . esc_html($date_label . ' ' . $date_to_show) . '</div>';
                        }
                    }
                    ?>
                </div>
            </article>
        <?php
        endforeach;
    else:
        echo '<p>' . esc_html__('No speakers found.', 'speakers-bureau') . '</p>';
    endif;
    ?>
  </div>

  <div class="sb-pagination">
    <?php
    if ($use_nearest) {
        // Manual pagination for nearest sort
        echo paginate_links([
            'total'   => max(1, $max_pages),
            'current' => $paged,
            'prev_text' => '«',
            'next_text' => '»',
            'type' => 'plain',
        ]);
    } else {
        // WordPress pagination with proper page detection
        echo paginate_links([
            'total'   => max(1, $query->max_num_pages),
            'current' => max(1, $paged),
            'prev_text' => '«',
            'next_text' => '»',
            'type' => 'plain',
        ]);
    }
    ?>
  </div>
</div>

<style>
:root {
    --sb-primary-color: <?php echo esc_html($display_settings['primary_color']); ?>;
    --sb-border-radius: <?php echo esc_html($display_settings['card_border_radius']); ?>px;
}

.sb-directory-grid {
    display: grid;
    gap: 20px;
    margin: 20px 0;
}

/* Grid columns */
.sb-directory-grid.sb-cols-1 { grid-template-columns: 1fr; }
.sb-directory-grid.sb-cols-2 { grid-template-columns: repeat(2, 1fr); }
.sb-directory-grid.sb-cols-3 { grid-template-columns: repeat(3, 1fr); }
.sb-directory-grid.sb-cols-4 { grid-template-columns: repeat(4, 1fr); }
.sb-directory-grid.sb-cols-5 { grid-template-columns: repeat(5, 1fr); }
.sb-directory-grid.sb-cols-6 { grid-template-columns: repeat(6, 1fr); }

/* Responsive adjustments */
@media (max-width: 768px) {
    .sb-directory-grid.sb-cols-3,
    .sb-directory-grid.sb-cols-4,
    .sb-directory-grid.sb-cols-5,
    .sb-directory-grid.sb-cols-6 {
        grid-template-columns: repeat(2, 1fr);
    }
}

@media (max-width: 480px) {
    .sb-directory-grid {
        grid-template-columns: 1fr !important;
    }
}

/* Card styles */
.sb-card {
    background: #fff;
    border: 1px solid #e0e0e0;
    border-radius: var(--sb-border-radius);
    overflow: hidden;
    transition: transform 0.2s ease, box-shadow 0.2s ease;
    display: flex;
    flex-direction: column;
    position: relative;
}

.sb-card:hover {
    transform: translateY(-2px);
}

/* Card shadows */
.sb-shadow-none .sb-card { box-shadow: none; }
.sb-shadow-light .sb-card { box-shadow: 0 2px 4px rgba(0,0,0,0.1); }
.sb-shadow-medium .sb-card { box-shadow: 0 4px 8px rgba(0,0,0,0.15); }
.sb-shadow-strong .sb-card { box-shadow: 0 8px 16px rgba(0,0,0,0.2); }

.sb-shadow-light .sb-card:hover { box-shadow: 0 4px 8px rgba(0,0,0,0.15); }
.sb-shadow-medium .sb-card:hover { box-shadow: 0 8px 16px rgba(0,0,0,0.2); }
.sb-shadow-strong .sb-card:hover { box-shadow: 0 12px 24px rgba(0,0,0,0.25); }

/* Image styles */
.sb-card-image {
    width: 100%;
    height: 200px;
    object-fit: cover;
    display: block;
}

.sb-img-circle .sb-card-image {
    border-radius: 50%;
    width: 150px;
    height: 150px;
    margin: 15px auto 0;
}

.sb-img-rounded .sb-card-image {
    border-radius: 8px;
    margin: 10px;
    width: calc(100% - 20px);
    height: 180px;
}

/* Card content */
.sb-card-content {
    padding: 15px;
    flex-grow: 1;
    display: flex;
    flex-direction: column;
}

.sb-card-title {
    font-size: 1.2em;
    font-weight: 600;
    margin: 0 0 10px 0;
    line-height: 1.3;
}

.sb-card-title a {
    color: var(--sb-primary-color);
    text-decoration: none;
}

.sb-card-title a:hover {
    text-decoration: underline;
}

.sb-card-field,
.sb-card-excerpt {
    margin: 5px 0;
    font-size: 0.9em;
    color: #000;
    line-height: 1.4;
}

.sb-card-distance {
    margin-top: auto;
    padding-top: 10px;
    font-size: 0.85em;
    color: var(--sb-primary-color);
    font-weight: 500;
}

.sb-card-date {
    position: absolute;
    bottom: 8px;
    right: 8px;
    font-size: 0.75em;
    color: #999;
    background: rgba(255, 255, 255, 0.9);
    padding: 2px 6px;
    border-radius: 3px;
    line-height: 1.2;
    font-weight: 400;
}

/* Card style variations */
.sb-style-compact .sb-card {
    padding: 10px;
}

.sb-style-compact .sb-card-content {
    padding: 10px;
}

.sb-style-compact .sb-card-image {
    height: 150px;
}

.sb-style-detailed .sb-card-content {
    padding: 20px;
}

.sb-style-detailed .sb-card-image {
    height: 250px;
}

.sb-style-detailed .sb-card-title {
    font-size: 1.3em;
    margin-bottom: 15px;
}

/* Pagination styles */
.sb-pagination {
    margin: 30px 0;
    text-align: center;
}

.sb-pagination .page-numbers {
    display: inline-block;
    padding: 8px 12px;
    margin: 0 4px;
    background: #f7f7f7;
    color: #333;
    text-decoration: none;
    border: 1px solid #ddd;
    border-radius: 4px;
    transition: all 0.2s ease;
}

.sb-pagination .page-numbers:hover,
.sb-pagination .page-numbers.current {
    background: var(--sb-primary-color);
    color: #fff;
    border-color: var(--sb-primary-color);
}

.sb-pagination .page-numbers.dots {
    background: none;
    border: none;
    color: #999;
}

.sb-pagination .page-numbers.prev,
.sb-pagination .page-numbers.next {
    font-weight: bold;
}
</style>

<script>
document.addEventListener('DOMContentLoaded', function() {
    // Helper function to get cookie value
    function getCookie(name) {
        const value = "; " + document.cookie;
        const parts = value.split("; " + name + "=");
        if (parts.length === 2) return parts.pop().split(";").shift();
        return null;
    }

    // Auto-populate ZIP field from cookie when "nearest" is selected
    function handleNearestZip() {
        const sortSelects = document.querySelectorAll('select[name="sort"]');
        const zipInputs = document.querySelectorAll('input[name="zip"]');

        sortSelects.forEach(function(select) {
            if (select.value === 'nearest') {
                const savedZip = getCookie('sb_nearest_zip');
                if (savedZip && zipInputs.length > 0) {
                    zipInputs.forEach(function(input) {
                        if (!input.value) { // Only populate if empty
                            input.value = savedZip;
                        }
                    });
                    // Auto-expand location form if zip was populated from cookie
                    const toggle = document.getElementById('sb-adv-toggle');
                    const advForm = document.getElementById('sb-adv');
                    if (toggle && advForm && savedZip) {
                        advForm.style.display = 'block';
                        toggle.setAttribute('aria-expanded', 'true');
                        toggle.textContent = '<?php echo esc_js(__('Hide location', 'speakers-bureau')); ?>';
                    }
                }
            }
        });
    }

    // Auto-submit form when sort dropdown changes
    const sortSelects = document.querySelectorAll('select[name="sort"]');
    sortSelects.forEach(function(select) {
        select.addEventListener('change', function() {
            // Handle nearest zip population before form submission
            if (this.value === 'nearest') {
                handleNearestZip();
                // Small delay to allow ZIP population before submit
                setTimeout(() => {
                    this.closest('form').submit();
                }, 100);
            } else {
                // Submit immediately for other sort options
                this.closest('form').submit();
            }
        });
    });

    // Initial check on page load
    handleNearestZip();

    // Handle advanced search toggle
    const toggle = document.getElementById('sb-adv-toggle');
    const advForm = document.getElementById('sb-adv');
    if (toggle && advForm) {
        toggle.addEventListener('click', function() {
            const isVisible = advForm.style.display === 'block';
            advForm.style.display = isVisible ? 'none' : 'block';
            toggle.setAttribute('aria-expanded', !isVisible);
            toggle.textContent = isVisible
                ? '<?php echo esc_js(__('Location…', 'speakers-bureau')); ?>'
                : '<?php echo esc_js(__('Hide location', 'speakers-bureau')); ?>';
        });
    }
});
</script>

<?php
/** END OF FILE: templates/speaker-list.php */
